/***
 * Specs Engine v6: Spectacles Saga Game Engine
  *           Copyright (c) 2017 Power-Command
***/
RequireScript('battleEngine/battle.js');
defScenelet('adjustBGM', {
    start: function (scene, volume, duration) {
        if (duration === void 0) { duration = 0.0; }
        music.adjust(volume, duration);
    },
    update: function (scene) {
        return music.adjusting;
    }
});
// .battle() command
// Starts a battle.
// Arguments:
//     battleID: The ID of the battle definition to use to initialize the fight.
defScenelet('battle', {
    start: function (scene, battleID, session) {
        this.mode = 'battle';
        this.battle = new Battle(session, battleID);
        this.battleThread = this.battle.go();
    },
    update: function (scene) {
        switch (this.mode) {
            case 'battle':
                if (!threads.isRunning(this.battleThread)) {
                    if (this.battle.result == BattleResult.Lose) {
                        term.print("player lost battle, showing Game Over screen");
                        this.mode = 'gameOver';
                        this.gameOver = new GameOverScreen();
                        this.gameOverThread = this.gameOver.show();
                    }
                    else {
                        return false;
                    }
                }
                break;
            case 'gameOver':
                if (!threads.isRunning(this.gameOverThread)) {
                    if (this.gameOver.action === GameOverAction.Retry) {
                        term.print("player asked to retry last battle");
                        this.mode = 'battle';
                        this.battleThread = this.battle.go();
                    }
                    else {
                        return false;
                    }
                }
                break;
        }
        return true;
    }
});
defScenelet('changeBGM', {
    start: function (scene, trackName, fadeTime) {
        music.play("music/" + trackName + ".ogg", fadeTime);
    }
});
defScenelet('marquee', {
    start: function (scene, text, backgroundColor, color) {
        backgroundColor = backgroundColor || Color.Black;
        color = color || Color.White;
        this.text = text;
        this.color = color;
        this.background = backgroundColor;
        this.font = Font.Default;
        this.windowSize = screen.width + this.font.getTextSize(this.text).width;
        this.height = this.font.height + 10;
        this.textHeight = this.font.height;
        this.fadeness = 0.0;
        this.scroll = 0.0;
        this.animation = new Scene()
            .tween(this, 15, 'linear', { fadeness: 1.0 })
            .tween(this, 60, 'easeOutExpo', { scroll: 0.5 })
            .tween(this, 60, 'easeInExpo', { scroll: 1.0 })
            .tween(this, 15, 'linear', { fadeness: 0.0 })
            .run();
    },
    render: function (scene) {
        var boxHeight = this.height * this.fadeness;
        var boxY = screen.height / 2 - boxHeight / 2;
        var textX = screen.width - this.scroll * this.windowSize;
        var textY = boxY + boxHeight / 2 - this.textHeight / 2;
        prim.rect(screen, 0, boxY, screen.width, boxHeight, this.background);
        this.font.drawText(screen, textX + 1, textY + 1, this.text, Color.Black.fade(this.color.a));
        this.font.drawText(screen, textX, textY, this.text, this.color);
    },
    update: function (scene) {
        return this.animation.isRunning();
    }
});
defScenelet('popBGM', {
    start: function (scene) {
        music.pop();
    }
});
defScenelet('pushBGM', {
    start: function (scene, trackName) {
        music.push("music/" + trackName + ".ogg");
    }
});
defScenelet('talk', {
    start: function (scene, speaker, showSpeaker, textSpeed, timeout /*...pages*/) {
        this.speakerName = speaker;
        this.speakerText = this.speakerName != null ? this.speakerName + ":" : null;
        this.showSpeaker = showSpeaker;
        this.textSpeed = textSpeed;
        this.timeout = timeout;
        this.timeoutLeft = this.timeout;
        this.font = GetSystemFont();
        this.text = [];
        var speakerTextWidth = this.font.getStringWidth(this.speakerText);
        var textAreaWidth = GetScreenWidth() - 16;
        for (i = 5; i < arguments.length; ++i) {
            var lineWidth = this.speakerName != null ? textAreaWidth - (speakerTextWidth + 5) : textAreaWidth;
            var wrappedText = this.font.wordWrapString(arguments[i], lineWidth);
            var page = this.text.push([]) - 1;
            for (var iLine = 0; iLine < wrappedText.length; ++iLine) {
                this.text[page].push(wrappedText[iLine]);
            }
        }
        this.boxVisibility = 0.0;
        this.textVisibility = 1.0;
        this.nameVisibility = 0.0;
        this.lineVisibility = 0.0;
        this.scrollOffset = 0.0;
        this.currentPage = 0;
        this.numLinesToDraw = 0;
        this.topLine = 0;
        this.lineToReveal = 0;
        this.textSurface = CreateSurface(textAreaWidth, this.font.getHeight() * 3 + 1, CreateColor(0, 0, 0, 0));
        this.transition = new Scene()
            .tween(this, 20, 'easeOutBack', { boxVisibility: 1.0 })
            .run();
        this.mode = "fadein";
        while (AreKeysLeft()) {
            GetKey();
        }
        if (Sphere.Game.disableTalking) {
            this.mode = "finish";
        }
        return true;
    },
    render: function (scene) {
        var lineHeight = this.font.getHeight();
        var boxHeight = lineHeight * 3 + 11;
        var finalBoxY = GetScreenHeight() * 0.85 - boxHeight / 2;
        var boxY = finalBoxY + (GetScreenHeight() - finalBoxY) * (1.0 - this.boxVisibility);
        OutlinedRectangle(-1, boxY - 1, GetScreenWidth() + 2, boxHeight + 2, CreateColor(0, 0, 0, 144 * this.boxVisibility));
        Rectangle(0, boxY, GetScreenWidth(), boxHeight, CreateColor(0, 0, 0, 128 * this.boxVisibility));
        this.textSurface.setBlendMode(REPLACE);
        this.textSurface.rectangle(0, 0, this.textSurface.width, this.textSurface.height, CreateColor(0, 0, 0, 0));
        this.textSurface.setBlendMode(BLEND);
        var lineCount = this.text[this.currentPage].length;
        var textAreaWidth = this.textSurface.width;
        var textX = 0;
        if (this.speakerName != null) {
            var speakerTextWidth = this.font.getStringWidth(this.speakerText);
            textX = speakerTextWidth + 5;
        }
        textAreaWidth -= textX;
        for (var iLine = Math.min(this.lineToReveal - this.topLine + 1, lineCount - this.topLine, 3) - 1; iLine >= 0; --iLine) {
            var trueLine = this.topLine + iLine;
            var textY = iLine * lineHeight - this.scrollOffset * lineHeight;
            var lineVisibility = iLine == 0 ? 1.0 - this.scrollOffset : 1.0;
            if (this.lineVisibility > 0.0 || this.lineToReveal != trueLine) {
                var lineText = this.text[this.currentPage][trueLine];
                this.font.setColorMask(CreateColor(0, 0, 0, 255 * this.textVisibility * lineVisibility));
                this.textSurface.drawText(this.font, textX + 1, textY + 1, lineText);
                this.font.setColorMask(CreateColor(255, 255, 255, 255 * this.textVisibility * lineVisibility));
                this.textSurface.drawText(this.font, textX, textY, lineText);
                if (this.lineToReveal == trueLine) {
                    var shownArea = textAreaWidth * this.lineVisibility;
                    this.textSurface.setBlendMode(SUBTRACT);
                    this.textSurface.gradientRectangle((textX - lineHeight * 2) + shownArea, textY, lineHeight * 2, lineHeight + 1, CreateColor(0, 0, 0, 0), CreateColor(0, 0, 0, 255), CreateColor(0, 0, 0, 255 * this.boxVisibility), CreateColor(0, 0, 0, 0));
                    this.textSurface.setBlendMode(REPLACE);
                    this.textSurface.rectangle(textX + shownArea, textY, textAreaWidth - shownArea, lineHeight + 1, CreateColor(0, 0, 0, 0));
                    this.textSurface.setBlendMode(BLEND);
                }
            }
            if (this.showSpeaker && this.speakerName != null && trueLine == 0) {
                this.font.setColorMask(CreateColor(0, 0, 0, this.textVisibility * this.nameVisibility * 255));
                this.textSurface.drawText(this.font, 1, textY + 1, this.speakerText);
                this.font.setColorMask(CreateColor(255, 192, 0, this.textVisibility * this.nameVisibility * 255));
                this.textSurface.drawText(this.font, 0, textY, this.speakerText);
            }
        }
        this.textSurface.blit(GetScreenWidth() / 2 - this.textSurface.width / 2, boxY + 5);
    },
    update: function (scene) {
        if (Sphere.Game.disableTalking) {
            this.mode = "finish";
        }
        switch (this.mode) {
            case "idle":
                if (this.timeout !== Infinity) {
                    if (this.topLine + 3 >= this.text[this.currentPage].length) {
                        this.timeoutLeft -= 1.0 / screen.frameRate;
                        if (this.timeoutLeft <= 0.0) {
                            if (this.currentPage < this.text.length - 1) {
                                this.mode = "page";
                            }
                            else {
                                this.mode = "hidetext";
                            }
                            this.timeoutLeft = this.timeout;
                        }
                    }
                    else {
                        this.mode = "nextline";
                        this.numLinesToDraw = 0;
                        this.lineVisibility = 0.0;
                    }
                }
                break;
            case "fadein":
                if (!this.transition.isRunning()) {
                    this.mode = "write";
                }
                break;
            case "write":
                this.nameVisibility = Math.min(this.nameVisibility + 4.0 / screen.frameRate, 1.0);
                if (this.nameVisibility >= 1.0) {
                    this.lineVisibility = Math.min(this.lineVisibility + this.textSpeed / screen.frameRate, 1.0);
                    var lineCount = Math.min(3, this.text[this.currentPage].length - this.topLine);
                    var currentLineText = this.text[this.currentPage][this.lineToReveal];
                    var currentLineWidth = this.font.getStringWidth(currentLineText);
                    var textAreaWidth = this.textSurface.width;
                    if (this.speakerName != null) {
                        var speakerTextWidth = this.font.getStringWidth(this.speakerText);
                        textAreaWidth -= speakerTextWidth + 5;
                    }
                    if (this.lineVisibility >= 1.0 || textAreaWidth * this.lineVisibility >= currentLineWidth + 20) {
                        if (this.lineToReveal - this.topLine == lineCount - 1)
                            this.mode = "idle";
                        ++this.lineToReveal;
                        ++this.numLinesToDraw;
                        if (this.numLinesToDraw < 3 && this.lineToReveal < this.text[this.currentPage].length) {
                            this.mode = "nextline";
                        }
                        else {
                            this.mode = "idle";
                        }
                        this.lineVisibility = 0.0;
                    }
                }
                break;
            case "nextline":
                if (this.lineToReveal < 3) {
                    this.lineVisibility = 0.0;
                    this.mode = "write";
                    break;
                }
                this.scrollOffset = Math.min(this.scrollOffset + 8.0 * this.textSpeed / screen.frameRate, 1.0);
                if (this.scrollOffset >= 1.0) {
                    this.topLine += 1;
                    this.scrollOffset = 0.0;
                    this.lineVisibility = 0.0;
                    this.textVisibility = 1.0;
                    this.mode = "write";
                }
                break;
            case "page":
                this.textVisibility = Math.max(this.textVisibility - (2.0 * this.textSpeed) / screen.frameRate, 0.0);
                if (this.textVisibility <= 0.0) {
                    this.mode = "write";
                    ++this.currentPage;
                    this.lineToReveal = 0;
                    this.numLinesToDraw = 0;
                    this.textVisibility = 1.0;
                    this.topLine = 0;
                    this.lineVisibility = 0.0;
                }
                break;
            case "hidetext":
                this.textVisibility = Math.max(this.textVisibility - (4.0 * this.textSpeed) / screen.frameRate, 0.0);
                if (this.textVisibility <= 0.0) {
                    this.transition = new Scene()
                        .tween(this, 20, 'easeInBack', { boxVisibility: 0.0 })
                        .run();
                    this.mode = "fadeout";
                }
                break;
            case "fadeout":
                if (!this.transition.isRunning()) {
                    this.mode = "finish";
                }
                break;
            case "finish":
                return false;
        }
        return true;
    },
    getInput: function (scene) {
        if (this.mode != "idle")
            return;
        if ((Keyboard.Default.isPressed(Key.Z) || joy.P1.isPressed(0))
            && this.timeout == Infinity) {
            if (this.topLine + 3 >= this.text[this.currentPage].length) {
                if (this.currentPage < this.text.length - 1) {
                    this.mode = "page";
                }
                else {
                    this.mode = "hidetext";
                }
            }
            else {
                this.mode = "nextline";
                this.numLinesToDraw = 0;
                this.lineVisibility = 0.0;
            }
        }
    }
});
//# sourceMappingURL=data:application/json;base64,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