/***
 * Specs Engine v6: Spectacles Saga Game Engine
  *           Copyright (c) 2017 Power-Command
***/
var MenuStrip = (function () {
    function MenuStrip(title, isCancelable, items) {
        if (title === void 0) { title = ""; }
        if (isCancelable === void 0) { isCancelable = true; }
        if (items === void 0) { items = null; }
        var _this = this;
        this.carouselSurface = null;
        this.font = GetSystemFont();
        this.isCancelable = isCancelable;
        this.menuItems = [];
        this.menuThread = null;
        this.selectedItem = 0;
        this.title = title;
        if (items !== null) {
            from(items)
                .each(function (v) { return _this.addItem(v); });
        }
    }
    MenuStrip.prototype.update = function () {
        switch (this.mode) {
            case 'open':
                if (!this.animation.isRunning()) {
                    this.mode = "idle";
                }
                break;
            case 'changeItem':
                if (!this.animation.isRunning()) {
                    var newSelection = this.selectedItem + this.scrollDirection;
                    if (newSelection < 0) {
                        newSelection = this.menuItems.length - 1;
                    }
                    else if (newSelection >= this.menuItems.length) {
                        newSelection = 0;
                    }
                    this.selectedItem = newSelection;
                    this.scrollDirection = 0;
                    this.scrollProgress = 0.0;
                    this.mode = "idle";
                }
                break;
            case 'close':
                return this.animation.isRunning();
        }
        return true;
    };
    MenuStrip.prototype.render = function () {
        var height = this.font.getHeight() + 10;
        var menuY = GetScreenHeight() - height * this.openness;
        var normalStripColor = CreateColor(0, 0, 0, this.openness * 192);
        var litStripColor = CreateColor(255, 255, 255, this.openness * 192);
        var stripColor = BlendColorsWeighted(litStripColor, normalStripColor, this.brightness, 1.0 - this.brightness);
        Rectangle(0, menuY, GetScreenWidth(), height, stripColor);
        var normalTitleColor = CreateColor(64, 64, 64, this.openness * 255);
        var litTitleColor = CreateColor(0, 0, 0, this.openness * 255);
        var titleColor = BlendColorsWeighted(litTitleColor, normalTitleColor, this.brightness, 1.0 - this.brightness);
        this.font.setColorMask(CreateColor(0, 0, 0, this.openness * 255));
        this.font.drawText(6, menuY + 6, this.title);
        this.font.setColorMask(titleColor);
        this.font.drawText(5, menuY + 5, this.title);
        this.carouselSurface.setBlendMode(REPLACE);
        this.carouselSurface.rectangle(0, 0, this.carouselSurface.width, this.carouselSurface.height, CreateColor(0, 0, 0, 0));
        this.carouselSurface.setBlendMode(BLEND);
        var xOffset = (this.selectedItem + this.scrollProgress * this.scrollDirection) * this.carouselSurface.width;
        var normalItemColor = CreateColor(255, 192, 0, this.openness * 255);
        var litItemColor = CreateColor(128, 128, 64, this.openness * 255);
        var itemColor = BlendColorsWeighted(litItemColor, normalItemColor, this.brightness, 1.0 - this.brightness);
        for (var i = -1; i <= this.menuItems.length; ++i) {
            var itemIndex = i;
            if (i >= this.menuItems.length) {
                itemIndex = i % this.menuItems.length;
            }
            else if (i < 0) {
                itemIndex = this.menuItems.length - 1 - Math.abs(i + 1) % this.menuItems.length;
            }
            var itemText = this.menuItems[itemIndex].text;
            var textX = i * this.carouselSurface.width + (this.carouselSurface.width / 2 - this.font.getStringWidth(itemText) / 2);
            this.font.setColorMask(CreateColor(0, 0, 0, this.openness * 255));
            this.carouselSurface.drawText(this.font, textX - xOffset + 1, 6, itemText);
            this.font.setColorMask(itemColor);
            this.carouselSurface.drawText(this.font, textX - xOffset, 5, itemText);
        }
        carouselX = GetScreenWidth() - 5 - this.carouselSurface.width - this.font.getStringWidth(">") - 5;
        this.carouselSurface.blit(carouselX, menuY);
        this.font.setColorMask(CreateColor(128, 128, 128, this.openness * 255));
        this.font.drawText(carouselX - this.font.getStringWidth("<") - 5, menuY + 5, "<");
        if (this.scrollDirection == -1) {
            this.font.setColorMask(CreateColor(255, 192, 0, this.openness * (1.0 - this.scrollProgress) * 255));
            this.font.drawText(carouselX - this.font.getStringWidth("<") - 5, menuY + 5, "<");
        }
        this.font.setColorMask(CreateColor(128, 128, 128, this.openness * 255));
        this.font.drawText(carouselX + this.carouselSurface.width + 5, menuY + 5, ">");
        if (this.scrollDirection == 1) {
            this.font.setColorMask(CreateColor(255, 192, 0, this.openness * (1.0 - this.scrollProgress) * 255));
            this.font.drawText(carouselX + this.carouselSurface.width + 5, menuY + 5, ">");
        }
    };
    MenuStrip.prototype.getInput = function () {
        if (this.mode != 'idle') {
            return;
        }
        var key = AreKeysLeft() ? GetKey() : null;
        if (key == GetPlayerKey(PLAYER_1, PLAYER_KEY_A)) {
            this.chosenItem = this.selectedItem;
            this.animation = new Scene()
                .fork()
                .tween(this, 7, 'easeInOutSine', { brightness: 1.0 })
                .tween(this, 7, 'easeInOutSine', { brightness: 0.0 })
                .end()
                .tween(this, 15, 'easeInQuad', { openness: 0.0 })
                .run();
            this.mode = 'close';
        }
        else if (key == GetPlayerKey(PLAYER_1, PLAYER_KEY_B) && this.isCancelable) {
            this.chosenItem = null;
            this.animation = new Scene()
                .tween(this, 15, 'easeInQuad', { openness: 0.0 })
                .run();
            this.mode = 'close';
        }
        else if (key == GetPlayerKey(PLAYER_1, PLAYER_KEY_LEFT)) {
            this.scrollDirection = -1;
            this.animation = new Scene()
                .tween(this, 15, 'linear', { scrollProgress: 1.0 })
                .run();
            this.mode = 'changeItem';
        }
        else if (key == GetPlayerKey(PLAYER_1, PLAYER_KEY_RIGHT)) {
            this.scrollDirection = 1;
            this.animation = new Scene()
                .tween(this, 15, 'linear', { scrollProgress: 1.0 })
                .run();
            this.mode = 'changeItem';
        }
    };
    MenuStrip.prototype.addItem = function (text, tag) {
        if (tag === void 0) { tag = text; }
        this.menuItems.push({
            text: text,
            tag: tag
        });
        return this;
    };
    MenuStrip.prototype.isOpen = function () {
        return this.menuThread !== null;
    };
    MenuStrip.prototype.open = function () {
        this.openness = 0.0;
        this.scrollDirection = 0;
        this.scrollProgress = 0.0;
        this.brightness = 0.0;
        this.mode = "open";
        var carouselWidth = 0;
        for (i = 0; i < this.menuItems.length; ++i) {
            var itemText = this.menuItems[i].text;
            carouselWidth = Math.max(this.font.getStringWidth(itemText) + 10, carouselWidth);
        }
        this.carouselSurface = CreateSurface(carouselWidth, this.font.getHeight() + 10, CreateColor(0, 0, 0, 0));
        while (AreKeysLeft()) {
            GetKey();
        }
        var menuThread = threads.create(this, 100);
        this.animation = new Scene()
            .tween(this, 15, 'easeOutQuad', { openness: 1.0 })
            .run();
        threads.join(menuThread);
        this.menuThread = null;
        return this.chosenItem === null ? null : this.menuItems[this.chosenItem].tag;
    };
    return MenuStrip;
}());
//# sourceMappingURL=data:application/json;base64,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