/***
 * Specs Engine v6: Spectacles Saga Game Engine
  *           Copyright (c) 2017 Power-Command
***/
Game.statuses =
    {
        // Crackdown status
        // Progressively lowers attack power when the same type of attack is used in succession.
        crackdown: {
            name: "Crackdown",
            tags: ['debuff'],
            initialize: function (unit) {
                this.lastSkillType = null;
                this.multiplier = 1.0;
            },
            acting: function (unit, eventData) {
                var _this = this;
                from(eventData.action.effects)
                    .where(function (it) { return it.type == 'damage'; })
                    .each(function (effect) {
                    var oldPower = effect.power;
                    effect.power = Math.max(Math.round(effect.power * _this.multiplier), 1);
                    if (effect.power != oldPower) {
                        term.print("outgoing POW modified by Crackdown to " + effect.power, "was: " + oldPower);
                    }
                });
            },
            useSkill: function (unit, eventData) {
                var oldMultiplier = this.multiplier;
                this.multiplier = eventData.skill.category == this.lastSkillType
                    ? this.multiplier / Game.bonusMultiplier
                    : 1.0;
                this.lastSkillType = eventData.skill.category;
                if (this.multiplier != oldMultiplier) {
                    if (this.multiplier < 1.0) {
                        term.print("Crackdown POW modifier dropped to ~" + Math.round(this.multiplier * 100) + "%");
                    }
                    else {
                        term.print("Crackdown POW modifier reset to 100%");
                    }
                }
            }
        },
        // Curse status
        // Prevents the affected unit from taking on beneficial statuses such as ReGen and Protect.
        // Wears off after 5 turns.
        curse: {
            name: "Curse",
            tags: ['debuff'],
            initialize: function (unit) {
                unit.liftStatusTags(['buff']);
                this.turnsLeft = 5;
            },
            afflicted: function (unit, eventData) {
                var statusDef = Game.statuses[eventData.statusID];
                if (from(statusDef.tags).anyIs('buff')) {
                    term.print("Status " + statusDef.name + " was blocked by Curse");
                    eventData.cancel = true;
                }
            },
            beginTurn: function (unit, eventData) {
                if (this.turnsLeft <= 0) {
                    term.print(unit.name + "'s Curse has expired");
                    unit.liftStatus('curse');
                }
                else {
                    term.print(unit.name + "'s Curse will expire in " + this.turnsLeft + " more turns");
                }
                --this.turnsLeft;
            }
        },
        // Delusion status
        // ...
        delusion: {
            name: "Delusion",
            tags: ['ailment', 'acute'],
        },
        // Disarray status
        // Randomizes the rank of any action, excluding stance changes, taken by the affected unit.
        // Expires after 3 actions.
        disarray: {
            name: "Disarray",
            tags: ['ailment', 'acute'],
            initialize: function (unit) {
                this.actionsTaken = 0;
            },
            acting: function (unit, eventData) {
                var oldRank = eventData.action.rank;
                eventData.action.rank = random.discrete(1, 5);
                if (eventData.action.rank != oldRank) {
                    term.print("Rank of action changed by Disarray to " + eventData.action.rank, "was: " + oldRank);
                }
                ++this.actionsTaken;
                term.print(this.actionsTaken < 3
                    ? unit.name + "'s Disarray will expire in " + (3 - this.actionsTaken) + " more action(s)"
                    : unit.name + "'s Disarray has expired");
                if (this.actionsTaken >= 3) {
                    unit.liftStatus('disarray');
                }
            }
        },
        // Drunk status
        // Increases attack power, but reduces the affected unit's speed and accuracy and
        // creates a weakness to Earth damage. Also ups the success rate of eaty moves
        // (e.g. Munch).
        drunk: {
            name: "Drunk",
            tags: ['acute'],
            overrules: ['immune'],
            statModifiers: {
                agi: 1 / Game.bonusMultiplier
            },
            ignoreEvents: [
                'itemUsed',
                'skillUsed',
                'unitDamaged',
                'unitHealed',
                'unitTargeted'
            ],
            initialize: function (unit) {
                this.turnsLeft = 7;
            },
            acting: function (unit, eventData) {
                from(eventData.action.effects)
                    .where(function (it) { return it.targetHint == 'selected'; })
                    .where(function (it) { return it.type == 'damage'; })
                    .each(function (effect) {
                    var oldPower = effect.power;
                    effect.power *= Game.bonusMultiplier;
                    if (effect.power != oldPower) {
                        term.print("Outgoing POW modified by Drunk to " + effect.power, "was: " + oldPower);
                    }
                });
            },
            aiming: function (unit, eventData) {
                if (eventData.action.accuracyType === 'devour')
                    eventData.aimRate *= Game.bonusMultiplier;
                else
                    eventData.aimRate /= Game.bonusMultiplier;
            },
            beginTurn: function (unit, eventData) {
                --this.turnsLeft;
                if (this.turnsLeft <= 0)
                    unit.liftStatus('drunk');
            },
            damaged: function (unit, eventData) {
                if (from(eventData.tags).anyIs('earth'))
                    eventData.amount *= Game.bonusMultiplier;
            },
        },
        // Fear status
        // Causes the affected unit to automatically Guard or use healing items instead
        // of attacking.
        fear: {
            name: "Fear",
            tags: ['ailment'],
        },
        // Final Stand status
        // Progressively weakens and causes knockback delay to the affected unit when an
        // attack is countered.
        finalStand: {
            name: "Final Stand",
            tags: ['special'],
            overrules: ['crackdown', 'disarray'],
            initialize: function (unit) {
                this.fatigue = 1.0;
                this.knockback = 5;
            },
            acting: function (unit, eventData) {
                var _this = this;
                from(eventData.action.effects)
                    .where(function (it) { return it.targetHint == 'selected'; })
                    .where(function (it) { return it.type == 'damage'; })
                    .each(function (effect) {
                    var oldPower = effect.power;
                    effect.power = Math.round(effect.power / _this.fatigue);
                    if (effect.power != oldPower) {
                        term.print("Outgoing POW modified by Final Stand to " + effect.power, "was: " + oldPower);
                    }
                });
            },
            attacked: function (unit, eventData) {
                if (eventData.stance == Stance.Counter) {
                    this.fatigue *= Game.bonusMultiplier;
                    unit.resetCounter(this.knockback);
                    ++this.knockback;
                }
            },
            damaged: function (unit, eventData) {
                if (!from(eventData.tags).anyIs('zombie')) {
                    eventData.amount *= this.fatigue;
                }
            }
        },
        // Frostbite status
        // Inflicts a small amount of Ice damage at the end of the affected unit's turn.
        // The effect progressively worsens, up to double its original severity.
        frostbite: {
            name: "Frostbite",
            tags: ['ailment', 'damage'],
            overrules: ['ignite'],
            initialize: function (unit) {
                this.multiplier = 1.0;
            },
            attacked: function (unit, eventData) {
                from(eventData.action.effects)
                    .where(function (it) { return it.type === 'damage'; })
                    .each(function (effect) {
                    if ('addStatus' in effect && effect.addStatus == 'ignite') {
                        delete effect.addStatus;
                    }
                });
                from(eventData.action.effects)
                    .where(function (it) { return it.type == 'addStatus' && it.status == 'ignite'; })
                    .each(function (effect) {
                    effect.type = null;
                });
            },
            damaged: function (unit, eventData) {
                if (from(eventData.tags).anyIs('fire') && unit.stance != Stance.Guard) {
                    eventData.amount *= Game.bonusMultiplier;
                    term.print("Frostbite neutralized by fire, damage increased");
                    unit.liftStatus('frostbite');
                }
            },
            endTurn: function (unit, eventData) {
                var vit = Game.math.statValue(unit.battlerInfo.baseStats.vit, unit.battlerInfo.level);
                unit.takeDamage(0.5 * vit * this.multiplier, ['ice', 'special']);
                this.multiplier = Math.min(this.multiplier + 0.1, 2.0);
            }
        },
        // Ghost status
        // Prevents the affected unit from being hit with physical or projectile attacks
        // from a non-Ghost and vice versa.
        ghost: {
            name: "Ghost",
            tags: ['ailment', 'undead'],
            overrules: ['zombie'],
            aiming: function (unit, eventData) {
                for (var i = 0; i < eventData.action.effects.length; ++i) {
                    var effect = eventData.action.effects[i];
                    if (effect.type != 'damage' || effect.damageType == 'magic') {
                        continue;
                    }
                    if (!from(eventData.targetInfo.statuses).anyIs('ghost')) {
                        eventData.aimRate = 0.0;
                    }
                }
            },
            attacked: function (unit, eventData) {
                for (var i = 0; i < eventData.action.effects.length; ++i) {
                    var effect = eventData.action.effects[i];
                    if (effect.type != 'damage' || effect.damageType == 'magic') {
                        continue;
                    }
                    if (!from(eventData.actingUnitInfo.statuses).anyIs('ghost')) {
                        eventData.action.accuracyRate = 0.0;
                    }
                }
            }
        },
        // Ignite status
        // Inflicts a small amount of Fire damage on the affected unit once per cycle. The
        // effect progressively diminishes, ultimately settling at half of its initial severity.
        ignite: {
            name: "Ignite",
            tags: ['ailment', 'damage'],
            overrules: ['frostbite'],
            initialize: function (unit) {
                this.multiplier = 1.0;
            },
            beginCycle: function (unit, eventData) {
                var vit = Game.math.statValue(unit.battlerInfo.baseStats.vit, unit.battlerInfo.level);
                unit.takeDamage(0.5 * vit * this.multiplier, ['fire', 'special']);
                this.multiplier = Math.max(this.multiplier - 0.05, 0.5);
            },
            attacked: function (unit, eventData) {
                from(eventData.action.effects)
                    .where(function (it) { return it.type === 'damage'; })
                    .each(function (effect) {
                    if ('addStatus' in effect && effect.addStatus == 'frostbite') {
                        delete effect.addStatus;
                    }
                });
                from(eventData.action.effects)
                    .where(function (it) { return it.type == 'addStatus' && it.status == 'frostbite'; })
                    .each(function (effect) {
                    effect.type = null;
                });
            },
            damaged: function (unit, eventData) {
                if (from(eventData.tags).anyIs('ice') && unit.stance != Stance.Guard) {
                    eventData.amount *= Game.bonusMultiplier;
                    term.print("Ignite neutralized by ice, damage increased");
                    unit.liftStatus('ignite');
                }
            }
        },
        // Immune status
        // Grants the affected unit full immunity to most negative status afflictions.
        // Wears off after 5 turns.
        immune: {
            name: "Immune",
            tags: ['buff'],
            initialize: function (unit) {
                this.turnsLeft = 5;
            },
            afflicted: function (unit, eventData) {
                var statusDef = Game.statuses[eventData.statusID];
                if (from(statusDef.tags).anyIs('ailment')) {
                    term.print("Status " + statusDef.name + " was blocked by Immune");
                    eventData.cancel = true;
                }
            },
            beginTurn: function (unit, eventData) {
                if (this.turnsLeft <= 0) {
                    term.print(unit.name + "'s Immune has expired");
                    unit.liftStatus('immune');
                }
                else {
                    term.print(unit.name + "'s Immune will expire in " + this.turnsLeft + " more turns");
                }
                --this.turnsLeft;
            }
        },
        // Off Guard status
        // Imbued as part of several two-turn attacks such as Charge Slash. If the unit
        // is attacked while Off Guard, the damage from the attack will be increased.
        offGuard: {
            name: "Off Guard",
            tags: ['special'],
            beginTurn: function (unit, eventData) {
                unit.liftStatus('offGuard');
            },
            damaged: function (unit, eventData) {
                if (eventData.actingUnit !== null)
                    eventData.amount *= Game.bonusMultiplier;
            }
        },
        // Protect status
        // Reduces damage from attacks. Each time the Protected unit is damaged by an attack,
        // the effectiveness of Protect is reduced.
        protect: {
            name: "Protect",
            tags: ['buff'],
            initialize: function (unit) {
                this.multiplier = 1 / Game.bonusMultiplier;
                this.lossPerHit = (1.0 - this.multiplier) / 10;
            },
            damaged: function (unit, eventData) {
                var isProtected = !from(eventData.tags).anyIn(['special', 'zombie']);
                if (isProtected) {
                    eventData.amount *= this.multiplier;
                    this.multiplier += this.lossPerHit;
                    if (this.multiplier >= 1.0) {
                        unit.liftStatus('protect');
                    }
                }
            }
        },
        // ReGen status
        // Restores a small amount of HP to the affected unit at the beginning of each
        // cycle. Wears off after 10 cycles.
        reGen: {
            name: "ReGen",
            tags: ['buff'],
            initialize: function (unit) {
                this.turnsLeft = 10;
            },
            beginCycle: function (unit, eventData) {
                var unitInfo = unit.battlerInfo;
                var cap = Game.math.hp(unitInfo, unitInfo.level, 1);
                unit.heal(cap / 10, ['cure']);
                --this.turnsLeft;
                if (this.turnsLeft <= 0) {
                    term.print(unit.name + "'s ReGen has expired");
                    unit.liftStatus('reGen');
                }
                else {
                    term.print(unit.name + "'s ReGen will expire in " + this.turnsLeft + " more cycle(s)");
                }
            }
        },
        // Skeleton status
        // The affected unit is still able to battle at 0 HP, but with reduced STR and MAG stats.
        // Taking physical or slash damage, or being hit with an HP restorative, while in this state will
        // result in death.
        skeleton: {
            name: "Skeleton",
            tags: ['undead'],
            overrules: ['ghost', 'zombie'],
            statModifiers: {
                str: 1 / Game.bonusMultiplier,
                mag: 1 / Game.bonusMultiplier
            },
            initialize: function (unit) {
                this.allowDeath = false;
            },
            cured: function (unit, eventData) {
                if (eventData.statusID == 'skeleton') {
                    unit.heal(1, [], true);
                }
            },
            damaged: function (unit, eventData) {
                this.allowDeath = from(eventData.tags)
                    .anyIn(['zombie', 'physical', 'sword', 'earth', 'omni']);
                if (!this.allowDeath) {
                    eventData.cancel = true;
                }
            },
            dying: function (unit, eventData) {
                eventData.cancel = !this.allowDeath;
            },
            healed: function (unit, eventData) {
                if (from(eventData.tags).anyIs('cure')) {
                    unit.takeDamage(eventData.amount, ['zombie']);
                }
                eventData.cancel = true;
            }
        },
        sniper: {
            name: "Sniper",
            tags: ['special'],
            beginTurn: function (unit, eventData) {
                unit.liftStatus('sniper');
            },
            damaged: function (unit, eventData) {
                if (!from(eventData.tags).anyIn(['special', 'zombie'])) {
                    eventData.amount *= Game.bonusMultiplier;
                    unit.clearQueue();
                    unit.liftStatus('sniper');
                    unit.resetCounter(1);
                }
            }
        },
        sleep: {
            name: "Sleep",
            tags: ['acute'],
            overrules: ['drunk', 'offGuard'],
            initialize: function (unit) {
                unit.actor.animate('sleep');
                this.wakeChance = 0.0;
            },
            beginCycle: function (unit, eventData) {
                if (random.chance(this.wakeChance)) {
                    unit.liftStatus('sleep');
                }
                this.wakeChance += 0.01;
            },
            beginTurn: function (unit, eventData) {
                eventData.skipTurn = true;
                unit.actor.animate('snore');
            },
            damaged: function (unit, eventData) {
                var healthLost = 100 * eventData.amount / unit.maxHP;
                if (random.chance(healthLost * 5 * this.wakeChance)
                    && eventData.tags.indexOf('magic') === -1
                    && eventData.tags.indexOf('special') === -1) {
                    unit.liftStatus('sleep');
                }
            }
        },
        // Specs Aura status
        // Restores a tiny amount of HP to the affected unit at the beginning of each
        // cycle. Similar to ReGen, except the effect is perpetual and less HP is recovered per
        // cycle.
        specsAura: {
            name: "Specs Aura",
            tags: ['special'],
            beginCycle: function (unit, eventData) {
                var vit = Game.math.statValue(unit.battlerInfo.baseStats.vit, unit.battlerInfo.level);
                unit.heal(0.25 * vit, ['specs']);
            }
        },
        // Zombie status
        // Causes magic and items which restore HP to inflict an equal amount of damage instead.
        // If the affected unit reaches 0 HP, this status will progress to Skeleton and
        // the unit will be allowed to continue battling. Converted HP restoratives will
        // kill outright, however.
        zombie: {
            name: "Zombie",
            tags: ['ailment', 'undead'],
            initialize: function (unit) {
                this.allowDeath = false;
            },
            damaged: function (unit, eventData) {
                // don't allow Specs Aura to kill the afflicted.
                this.allowDeath = from(eventData.tags).anyIs('zombie')
                    && !from(eventData.tags).anyIs('specs');
            },
            dying: function (unit, eventData) {
                if (!this.allowDeath) {
                    unit.addStatus('skeleton');
                    eventData.cancel = true;
                }
            },
            healed: function (unit, eventData) {
                if (from(eventData.tags).anyIn(['cure', 'specs'])) {
                    var damageTags = from(eventData.tags).anyIs('specs')
                        ? ['zombie', 'specs']
                        : ['zombie'];
                    unit.takeDamage(eventData.amount, damageTags);
                    eventData.cancel = true;
                }
            }
        },
    };
//# sourceMappingURL=data:application/json;base64,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