/***
 * Specs Engine v6: Spectacles Saga Game Engine
  *           Copyright (c) 2017 Power-Command
***/
Game.conditions =
    {
        // Blackout field condition
        // Lowers accuracy and sometimes retargets attacks. Wears off after 10 actions.
        blackout: {
            name: "Blackout",
            initialize: function (battle) {
                this.actionsLeft = 10;
            },
            actionTaken: function (battle, eventData) {
                if (eventData.targets.length == 1 && random.chance(0.5)) {
                    var target = eventData.targets[0];
                    var newTargets = random.chance(0.5) ? battle.alliesOf(target) : battle.enemiesOf(target);
                    eventData.targets = [random.sample(newTargets)];
                }
                --this.actionsLeft;
                if (this.actionsLeft <= 0) {
                    term.print("Blackout has expired");
                    battle.liftCondition('blackout');
                }
                else {
                    term.print("Blackout will expire in " + this.actionsLeft + " more action(s)");
                }
            }
        },
        // General Disarray field condition
        // Randomizes the move rank of any skill or item used. Wears off after
        // 15 actions have been taken.
        generalDisarray: {
            name: "G. Disarray",
            initialize: function (battle) {
                this.actionsLeft = 15;
            },
            actionTaken: function (battle, eventData) {
                var oldRank = eventData.action.rank;
                eventData.action.rank = random.discrete(1, 5);
                if (eventData.action.rank != oldRank) {
                    term.print("Rank of action changed by G. Disarray to " + eventData.action.rank, "was: " + oldRank);
                }
                --this.actionsLeft;
                if (this.actionsLeft > 0) {
                    term.print("G. Disarray will expire in " + this.actionsLeft + " more action(s)");
                }
                else {
                    term.print("G. Disarray has expired");
                    battle.liftCondition('generalDisarray');
                }
            }
        },
        // Healing Aura field condition
        // Restores a small amount of health to a random battler at the beginning of
        // each cycle. Wears off after 25 healings.
        healingAura: {
            name: "Healing Aura",
            initialize: function (battle) {
                this.cyclesLeft = 25;
            },
            beginCycle: function (battle, eventData) {
                var units = from(battle.battleUnits)
                    .where(function (it) { return it.isAlive(); })
                    .toArray();
                var unit = random.sample(units);
                var vit = Game.math.statValue(unit.battlerInfo.baseStats.vit, unit.battlerInfo.level);
                unit.heal(vit, ['cure']);
                --this.cyclesLeft;
                if (this.cyclesLeft <= 0) {
                    term.print("Healing Aura has expired");
                    battle.liftCondition('healingAura');
                }
                else {
                    term.print("Healing Aura will expire in " + this.cyclesLeft + " more cycle(s)");
                }
            }
        },
        // Inferno field condition
        // Inflicts a small amount of Fire damage on all battlers at the beginning of a
        // cycle and boosts any Fire attacks performed. Residual damage from Inferno diminishes
        // over time, eventually settling at half the original output.
        inferno: {
            name: "Inferno",
            initialize: function (battle) {
                from(battle.battleUnits)
                    .where(function (it) { return it.isAlive(); })
                    .each(function (unit) {
                    if (unit.hasStatus('frostbite')) {
                        term.print(unit.name + "'s Frostbite nullified by Inferno installation");
                        unit.liftStatus('frostbite');
                    }
                });
            },
            actionTaken: function (battle, eventData) {
                from(eventData.action.effects)
                    .where(function (it) { return it.type === 'damage'; })
                    .each(function (effect) {
                    if (effect.element == 'fire') {
                        var oldPower = effect.power;
                        effect.power = Math.round(effect.power * Game.bonusMultiplier);
                        term.print("Fire attack strengthened by Inferno to " + effect.power + " POW", "was: " + oldPower);
                    }
                    else if (effect.element == 'ice') {
                        var oldPower = effect.power;
                        effect.power = Math.round(effect.power / Game.bonusMultiplier);
                        term.print("Ice attack weakened by Inferno to " + effect.power + " POW", "was: " + oldPower);
                    }
                });
            },
            beginCycle: function (battle, eventData) {
                var units = from(battle.battleUnits)
                    .where(function (it) { return it.isAlive(); })
                    .toArray();
                var unit = random.sample(units);
                var vit = Game.math.statValue(unit.battlerInfo.baseStats.vit, unit.battlerInfo.level);
                unit.takeDamage(vit, ['special', 'fire']);
            },
            conditionInstalled: function (battle, eventData) {
                if (eventData.conditionID == 'subzero') {
                    term.print("Inferno canceled by Subzero installation, both suppressed");
                    eventData.cancel = true;
                    battle.liftCondition('inferno');
                    from(battle.battleUnits)
                        .where(function (it) { return it.isAlive(); })
                        .each(function (unit) {
                        unit.addStatus('zombie', true);
                    });
                }
            },
            unitAfflicted: function (battle, eventData) {
                if (eventData.statusID == 'frostbite') {
                    eventData.cancel = true;
                    term.print("Frostbite is incompatible with Inferno");
                }
            }
        },
        // Subzero field condition
        // Inflicts a small amount of Ice damage on a battler at the end of his turn.
        // The effect intensifies over time per battler, maxing out at double its original
        // output.
        subzero: {
            name: "Subzero",
            initialize: function (battle) {
                this.multiplier = 1.0;
                this.rank = 0;
                from(battle.battleUnits)
                    .where(function (it) { return it.isAlive(); })
                    .each(function (unit) {
                    if (unit.hasStatus('frostbite')) {
                        term.print(unit.name + "'s Frostbite overruled by Subzero installation");
                        unit.liftStatus('frostbite');
                    }
                    if (unit.hasStatus('ignite')) {
                        term.print(unit.name + "'s Ignite nullified by Subzero installation");
                        unit.liftStatus('ignite');
                    }
                });
            },
            actionTaken: function (battle, eventData) {
                this.rank = eventData.action.rank;
                from(eventData.action.effects)
                    .where(function (it) { return it.type === 'damage'; })
                    .where(function (it) { return it.element === 'ice'; })
                    .each(function (effect) {
                    if (effect.element == 'ice') {
                        var oldPower = effect.power;
                        effect.power = Math.round(effect.power * Game.bonusMultiplier);
                        term.print("Ice attack strengthened by Subzero to " + effect.power + " POW", "was: " + oldPower);
                    }
                    else if (effect.element == 'fire') {
                        var oldPower = effect.power;
                        effect.power = Math.round(effect.power / Game.bonusMultiplier);
                        term.print("Fire attack weakened by Subzero to " + effect.power + " POW", "was: " + oldPower);
                    }
                });
            },
            conditionInstalled: function (battle, eventData) {
                if (eventData.conditionID == 'inferno') {
                    term.print("Subzero canceled by Inferno installation, both suppressed");
                    eventData.cancel = true;
                    battle.liftCondition('subzero');
                    from(battle.battleUnits)
                        .where(function (it) { return it.isAlive(); })
                        .each(function (unit) {
                        unit.addStatus('zombie', true);
                    });
                }
            },
            endTurn: function (battle, eventData) {
                var unit = eventData.actingUnit;
                if (unit.isAlive() && this.rank != 0) {
                    var vit = Game.math.statValue(unit.battlerInfo.baseStats.vit, unit.battlerInfo.level);
                    unit.takeDamage(this.rank * vit * this.multiplier / 5, ['special', 'ice']);
                    var increment = 0.1 * this.rank / 5;
                    this.multiplier = Math.min(this.multiplier + increment, 2.0);
                }
                this.rank = 0;
            },
            unitAfflicted: function (battle, eventData) {
                if (eventData.statusID == 'frostbite') {
                    eventData.cancel = true;
                    term.print("Frostbite infliction overruled by Subzero");
                }
                else if (eventData.statusID == 'ignite') {
                    eventData.cancel = true;
                    term.print("Ignite is incompatible with Subzero");
                }
            }
        },
        // Thunderstorm field condition
        // Sometimes drops a lightning bolt on a unit at the end of their turn, dealing a small amount
        // of lightning damage and inflicting Zombie status. Wears off after 10 strikes.
        thunderstorm: {
            name: "Thunderstorm",
            initialize: function (battle) {
                this.strikesLeft = 10;
            },
            endTurn: function (battle, eventData) {
                if (random.chance(0.5)) {
                    var unit = eventData.actingUnit;
                    term.print(unit.name + " struck by lightning from Thunderstorm");
                    var level = battle.getLevel();
                    var attack = Game.math.statValue(100, level);
                    var defense = Game.math.statValue(0, level);
                    var damage = Game.math.damage.calculate(5, battle.getLevel(), unit.tier, attack, defense);
                    unit.takeDamage(damage, ['special', 'lightning']);
                    unit.liftStatusTags('buff');
                    --this.strikesLeft;
                    if (this.strikesLeft <= 0) {
                        term.print("Thunderstorm has expired");
                        battle.liftCondition('thunderstorm');
                    }
                    else {
                        term.print("Thunderstorm will expire in " + this.strikesLeft + " more strike(s)");
                    }
                }
            }
        }
    };
//# sourceMappingURL=data:application/json;base64,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