"use strict";
/***
 * Specs Engine v6: Spectacles Saga Game Engine
  *           Copyright (c) 2017 Power-Command
***/
Object.defineProperty(exports, "__esModule", { value: true });
var prim = require("prim");
var scenes_1 = require("scenes");
var HPGauge = (function () {
    function HPGauge(capacity, sectorSize, color, maxSectors) {
        if (sectorSize === void 0) { sectorSize = 100; }
        if (color === void 0) { color = Color.White; }
        if (maxSectors === void 0) { maxSectors = 'auto'; }
        this.borderColor = Color.Black.fade(color.a);
        this.capacity = capacity;
        this.colorFadeDuration = 0;
        this.colorFadeTimer = 0;
        this.damage = 0;
        this.damageColor = Color.DarkRed.fade(color.a);
        this.damageFadeness = 1.0;
        this.drainSpeed = 1.0;
        this.emptyColor = Color.of('#202020').fade(color.a);
        this.fadeSpeed = 0.0;
        this.fadeness = 1.0;
        this.hpColor = color.clone();
        this.isVisible = true;
        this.maxSectors = maxSectors;
        this.newColor = color.clone();
        this.newReading = this.capacity;
        this.numCombosRunning = 0;
        this.oldColor = color.clone();
        this.oldReading = this.capacity;
        this.reading = this.capacity;
        this.sectorSize = sectorSize;
    }
    HPGauge.prototype.beginCombo = function () {
        ++this.numCombosRunning;
    };
    HPGauge.prototype.changeColor = function (color, numFrames) {
        if (numFrames === void 0) { numFrames = 0; }
        this.oldColor = this.hpColor.clone();
        this.newColor = color.clone();
        if (numFrames != 0) {
            this.colorFadeDuration = numFrames;
            this.colorFadeTimer = 0;
        }
        else {
            this.hpColor = this.newColor;
        }
    };
    HPGauge.prototype.draw = function (x, y, width, height) {
        if (this.fadeness >= 1.0)
            return; // invisible, skip rendering
        var damageShown = Math.max(this.damage - (this.reading - this.newReading), 0) * (1.0 - this.damageFadeness);
        var numReserves = Math.ceil(this.capacity / this.sectorSize - 1);
        var numReservesFilled = Math.max(Math.ceil(this.reading / this.sectorSize - 1), 0);
        var numReservesDamaged = Math.ceil((damageShown + this.reading) / this.sectorSize - 1);
        var barInUse;
        if (numReservesFilled == numReserves) {
            barInUse = this.capacity % this.sectorSize;
            if (barInUse == 0) {
                barInUse = this.sectorSize;
            }
        }
        else {
            barInUse = this.sectorSize;
        }
        var barFilled = this.reading % this.sectorSize;
        if (barFilled == 0 && this.reading > 0)
            barFilled = barInUse;
        var barDamaged = Math.min(damageShown, this.sectorSize - barFilled);
        var barHeight = Math.ceil(height * 0.5 + 0.5);
        var widthInUse = Math.round((width - 2) * barInUse / this.sectorSize);
        var fillWidth = Math.ceil(widthInUse * barFilled / barInUse);
        var damageWidth = Math.ceil(widthInUse * (barFilled + barDamaged) / barInUse) - fillWidth;
        var emptyWidth = widthInUse - (fillWidth + damageWidth);
        var borderColor = fadeColor(this.borderColor, this.fadeness);
        var fillColor = fadeColor(this.hpColor, this.fadeness);
        var emptyColor = fadeColor(this.emptyColor, this.fadeness);
        var usageColor = Color.mix(emptyColor, fadeColor(this.damageColor, this.fadeness), this.damageFadeness, 1.0 - this.damageFadeness);
        if (barInUse < this.sectorSize && numReservesFilled > 0) {
            prim.lineRect(screen, x, y, width, barHeight, 1, Color.mix(borderColor, Color.Transparent, 25, 75));
            drawSegment(x + 1, y + 1, width - 2, barHeight - 2, Color.mix(fillColor, Color.Transparent, 25, 75));
        }
        var barEdgeX = x + width - 1;
        prim.lineRect(screen, barEdgeX - widthInUse - 1, y, widthInUse + 2, barHeight, 1, borderColor);
        drawSegment(barEdgeX - fillWidth, y + 1, fillWidth, barHeight - 2, fillColor);
        prim.rect(screen, barEdgeX - fillWidth - damageWidth, y + 1, damageWidth, barHeight - 2, usageColor);
        drawSegment(barEdgeX - fillWidth - damageWidth - emptyWidth, y + 1, emptyWidth, barHeight - 2, emptyColor);
        var slotYSize = height - barHeight + 1;
        var slotXSize = this.maxSectors === 'auto'
            ? Math.round(slotYSize * 1)
            : Math.ceil(width / (this.maxSectors - 1));
        var slotX;
        var slotY = y + height - slotYSize;
        for (var i = 0; i < numReserves; ++i) {
            var color;
            if (i < numReservesFilled) {
                color = fillColor;
            }
            else if (i < numReservesDamaged) {
                color = usageColor;
            }
            else {
                color = emptyColor;
            }
            slotX = x + (width - slotXSize) - i * (slotXSize - 1);
            prim.lineRect(screen, slotX, slotY, slotXSize, slotYSize, 1, borderColor);
            drawSegment(slotX + 1, slotY + 1, slotXSize - 2, slotYSize - 2, color);
        }
    };
    HPGauge.prototype.endCombo = function () {
        --this.numCombosRunning;
        if (this.numCombosRunning < 0) {
            this.numCombosRunning = 0;
        }
    };
    HPGauge.prototype.hide = function (duration) {
        if (duration === void 0) { duration = 0.0; }
        if (duration > 0.0) {
            this.fadeSpeed = 1.0 / duration * (1.0 - this.fadeness);
        }
        else {
            this.fadeSpeed = 0.0;
            this.fadeness = 1.0;
        }
    };
    HPGauge.prototype.set = function (value) {
        value = Math.min(Math.max(Math.round(value), 0), this.capacity);
        if (value != this.reading) {
            if (this.numCombosRunning > 0)
                this.damage += this.reading - value;
            else
                this.damage = this.reading - value;
            this.damageFadeness = 0.0;
            this.oldReading = this.reading;
            this.newReading = value;
            this.drainTimer = 0.0;
        }
    };
    HPGauge.prototype.show = function (duration) {
        if (duration === void 0) { duration = 0.0; }
        if (duration > 0.0) {
            this.fadeSpeed = 1.0 / duration * (0.0 - this.fadeness);
        }
        else {
            this.fadeSpeed = 0.0;
            this.fadeness = 0.0;
        }
    };
    HPGauge.prototype.update = function () {
        ++this.colorFadeTimer;
        if (this.colorFadeDuration != 0 && this.colorFadeTimer < this.colorFadeDuration) {
            this.hpColor.r = tween(this.oldColor.r, this.colorFadeTimer, this.colorFadeDuration, this.newColor.r);
            this.hpColor.g = tween(this.oldColor.g, this.colorFadeTimer, this.colorFadeDuration, this.newColor.g);
            this.hpColor.b = tween(this.oldColor.b, this.colorFadeTimer, this.colorFadeDuration, this.newColor.b);
            this.hpColor.a = tween(this.oldColor.a, this.colorFadeTimer, this.colorFadeDuration, this.newColor.a);
        }
        else {
            this.hpColor = this.newColor;
            this.colorFadeDuration = 0;
        }
        this.fadeness = Math.min(Math.max(this.fadeness + this.fadeSpeed / screen.frameRate, 0.0), 1.0);
        this.drainTimer += this.drainSpeed / screen.frameRate;
        if (this.newReading != this.reading && this.drainTimer < 0.25) {
            this.reading = Math.round(tween(this.oldReading, this.drainTimer, 0.25, this.newReading));
        }
        else {
            this.reading = this.newReading;
        }
        if (this.numCombosRunning <= 0 && this.reading == this.newReading) {
            this.damageFadeness += this.drainSpeed / screen.frameRate;
            if (this.damageFadeness >= 1.0) {
                this.damage = 0;
                this.damageFadeness = 1.0;
            }
        }
    };
    return HPGauge;
}());
exports.HPGauge = HPGauge;
var MPGauge = (function () {
    function MPGauge(capacity, color, font) {
        if (color === void 0) { color = Color.DodgerBlue; }
        if (font === void 0) { font = Font.Default; }
        this.color = color;
        this.textFont = font;
        this.animation = null;
        this.capacity = capacity;
        this.reading = capacity;
        this.usage = 0;
        this.usageColor = Color.Transparent;
        this.value = capacity;
    }
    MPGauge.prototype.draw = function (x, y, size) {
        screen.clipTo(x, y, size, size);
        if (this.capacity > 0) {
            var innerFillColor = this.color;
            var outerFillColor = Color.mix(this.color, Color.Black.fade(this.color.a));
            var outerUsageColor = this.usageColor;
            var innerUsageColor = Color.mix(this.usageColor, Color.Black.fade(this.usageColor.a));
            var maxRadius = Math.ceil(size * Math.sqrt(2) / 2);
            prim.circle(screen, x + size / 2, y + size / 2, maxRadius * Math.sqrt((this.reading + this.usage) / this.capacity), innerUsageColor, outerUsageColor);
            prim.circle(screen, x + size / 2, y + size / 2, maxRadius * Math.sqrt(this.reading / this.capacity), innerFillColor, outerFillColor);
            drawText(this.textFont, x + size - 21, y + size / 2 - 8, 1, Color.White, Math.round(this.reading), 'right');
            drawText(this.textFont, x + size - 20, y + size / 2 - 4, 1, new Color(1, 0.75, 0), "MP");
        }
        screen.clipTo(0, 0, screen.width, screen.height);
    };
    MPGauge.prototype.set = function (value) {
        value = Math.min(Math.max(value, 0), this.capacity);
        if (value != this.value) {
            if (this.animation != null) {
                this.animation.stop();
            }
            this.animation = new scenes_1.Scene()
                .fork()
                .tween(this, 15, 'easeInOutSine', { usage: this.reading - value })
                .end()
                .fork()
                .tween(this, 15, 'easeInOutSine', { reading: value })
                .end()
                .tween(this.usageColor, 6, 'easeInOutSine', this.color)
                .tween(this.usageColor, 30, 'easeInOutSine', Color.Transparent)
                .run();
        }
        this.value = value;
    };
    MPGauge.prototype.update = function () {
        if (this.animation != null && !this.animation.isRunning()) {
            this.usage = 0;
        }
    };
    return MPGauge;
}());
exports.MPGauge = MPGauge;
var TurnPreview = (function () {
    function TurnPreview() {
        this.entries = {};
        this.fadeness = 1.0;
        this.font = GetSystemFont();
        this.lastPrediction = null;
        this.thread = null;
    }
    TurnPreview.prototype.dispose = function () {
        threads.kill(this.thread);
    };
    TurnPreview.prototype.ensureEntries = function (unit) {
        if (!(unit.tag in this.entries)) {
            var entry = {
                color: unit.isPartyMember() ? CreateColor(72, 61, 139, 255) : CreateColor(128, 0, 0, 255),
                name: unit.name,
                turnBoxes: []
            };
            for (var i = 0; i < 8; ++i) {
                entry.turnBoxes[i] = { x: 160, tween: null };
            }
            this.entries[unit.tag] = entry;
        }
    };
    TurnPreview.prototype.render = function () {
        var alpha = 255 * (1.0 - this.fadeness);
        var y = -16 * this.fadeness;
        SetClippingRectangle(0, y, 160, 16);
        Rectangle(0, y, 48, 16, CreateColor(0, 0, 0, alpha * 0.75));
        OutlinedRectangle(0, y, 48, 16, CreateColor(0, 0, 0, alpha * 0.125));
        drawTextEx(this.font, 24, y + 2, "next:", CreateColor(128, 128, 128, alpha), 1, 'center');
        Rectangle(48, y, 112, 16, CreateColor(0, 0, 0, alpha * 0.75));
        OutlinedRectangle(48, y, 112, 16, CreateColor(0, 0, 0, alpha * 0.125));
        for (var id in this.entries) {
            var entry = this.entries[id];
            for (var i = 0; i < entry.turnBoxes.length; ++i) {
                var turnBox = entry.turnBoxes[i];
                Rectangle(turnBox.x, y, 16, 16, entry.color);
                OutlinedRectangle(turnBox.x, y, 16, 16, CreateColor(0, 0, 0, alpha * 0.25));
                drawTextEx(this.font, turnBox.x + 4, y + 2, entry.name[0], BlendColors(entry.color, CreateColor(255, 255, 255, 255)), 1);
            }
        }
        SetClippingRectangle(0, 0, GetScreenWidth(), GetScreenHeight());
    };
    TurnPreview.prototype.set = function (prediction) {
        var moveEasing = 'easeInOutExpo';
        var moveTime = 15;
        if (this.lastPrediction !== null) {
            for (var i = 0; i < Math.min(this.lastPrediction.length, 7); ++i) {
                var unit = this.lastPrediction[i].unit;
                var turnIndex = this.lastPrediction[i].turnIndex;
                var turnBox = this.entries[unit.tag].turnBoxes[turnIndex];
                if (turnBox.tween !== null) {
                    turnBox.tween.stop();
                }
                turnBox.tween = new scenes_1.Scene()
                    .tween(turnBox, moveTime, moveEasing, { x: 160 });
                turnBox.tween.run();
            }
        }
        this.lastPrediction = prediction;
        for (var i = 0; i < Math.min(prediction.length, 7); ++i) {
            var unit = prediction[i].unit;
            var turnIndex = prediction[i].turnIndex;
            this.ensureEntries(unit);
            var turnBox = this.entries[unit.tag].turnBoxes[turnIndex];
            if (turnBox.tween !== null) {
                turnBox.tween.stop();
            }
            turnBox.tween = new scenes_1.Scene()
                .tween(turnBox, moveTime, moveEasing, { x: 48 + i * 16 });
            turnBox.tween.run();
        }
    };
    TurnPreview.prototype.show = function () {
        if (this.thread === null) {
            term.print("activate battle screen turn preview");
            this.thread = threads.create(this, 20);
        }
        new scenes_1.Scene()
            .tween(this, 30, 'easeOutExpo', { fadeness: 0.0 })
            .run();
    };
    TurnPreview.prototype.update = function () {
        return true;
    };
    return TurnPreview;
}());
exports.TurnPreview = TurnPreview;
function drawSegment(x, y, width, height, color) {
    var topHeight = Math.ceil(height / 8);
    var bottomHeight = height - topHeight;
    var yBottom = y + topHeight;
    var dimColor = Color.mix(color, Color.Black.fade(color.a), 66, 33);
    prim.rect(screen, x, y, width, topHeight, dimColor, dimColor, color, color);
    prim.rect(screen, x, yBottom, width, bottomHeight, color, color, dimColor, dimColor);
}
;
function drawText(font, x, y, shadowDistance, color, text, alignment) {
    if (alignment === void 0) { alignment = 'left'; }
    var Align = {
        left: function (font, x, text) { return x; },
        center: function (font, x, text) { return x - font.getTextSize(text).width / 2; },
        right: function (font, x, text) { return x - font.getTextSize(text).width; }
    };
    if (!(alignment in Align))
        throw new Error("invalid text alignment '" + alignment + "'.");
    x = Align[alignment](font, x, text);
    font.drawText(screen, x + shadowDistance, y + shadowDistance, text, Color.Black.fade(color.a));
    font.drawText(screen, x, y, text, color);
}
function fadeColor(color, fadeness) {
    return color.fade(1.0 - fadeness);
}
function tween(start, time, duration, end) {
    return -(end - start) / 2 * (Math.cos(Math.PI * time / duration) - 1) + start;
}
//# sourceMappingURL=data:application/json;base64,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