/***
 * Specs Engine v6: Spectacles Saga Game Engine
  *           Copyright (c) 2017 Power-Command
***/
// TargetMenu() constructor
// Creates an object representing a move targeting menu.
// Arguments:
//     unit:      The battler whose move's target will be selected.
//     battle:    The battle session during which the menu will be shown.
//     usable:    Optional. If specified and not null, the Usable move whose target is being determined.
//     moveName:  Optional. The move name displayed while selecting a target. If not specified or null,
//                the move name will be taken from the Usable.
function TargetMenu(unit, battle, usable, moveName) {
    if (usable === void 0) { usable = null; }
    if (moveName === void 0) { moveName = null; }
    this.battle = battle;
    this.doChangeInfo = null;
    this.isChoiceMade = false;
    this.infoBoxFadeness = 1.0;
    this.infoFadeness = 1.0;
    this.isTargetScanOn = from(battle.alliesOf(unit))
        .where(function (unit) { return unit.isAlive(); })
        .any(function (unit) { return unit.allowTargetScan; });
    this.isTargetLocked = false;
    this.isGroupCast = false;
    this.name = moveName !== null ? moveName
        : usable !== null ? usable.name
            : unit.name;
    this.statusNames = null;
    this.cursorFont = GetSystemFont();
    this.infoFont = GetSystemFont();
    this.targets = [];
    this.unit = unit;
    this.unitToShowInfo = null;
    this.usable = usable;
    this.allowDeadUnits = usable !== null ? usable.allowDeadTarget : false;
    this.drawCursor = function (unit) {
        var width = this.cursorFont.getStringWidth(this.name) + 10;
        var x = unit.actor.x < GetScreenWidth() / 2 ? unit.actor.x + 37 : unit.actor.x - 5 - width;
        var y = unit.actor.y + 6;
        Rectangle(x, y, width, 20, CreateColor(0, 0, 0, 128));
        OutlinedRectangle(x, y, width, 20, CreateColor(0, 0, 0, 64));
        drawTextEx(this.cursorFont, x + width / 2, y + 4, this.name, CreateColor(255, 255, 255, 255), 1, 'center');
    };
    this.drawInfoBox = function (x, y, width, height, alpha) {
        Rectangle(x, y, width, height, CreateColor(0, 0, 0, alpha * (1.0 - this.infoBoxFadeness)));
        OutlinedRectangle(x, y, width, height, CreateColor(0, 0, 0, 32 * (1.0 - this.infoBoxFadeness)));
    };
    this.moveCursor = function (direction) {
        if (this.isGroupCast || this.targets == null)
            return;
        var position = this.targets[0].actor.position;
        var candidates = this.battle.alliesOf(this.targets[0]);
        var unitToSelect = null;
        while (unitToSelect === null) {
            position += direction;
            position = position > 2 ? 0 :
                position < 0 ? 2 :
                    position;
            for (var i = 0; i < candidates.length; ++i) {
                if (position == candidates[i].actor.position
                    && (candidates[i].isAlive() || this.allowDeadUnits)) {
                    unitToSelect = candidates[i];
                    break;
                }
            }
        }
        if (unitToSelect !== this.targets[0]) {
            this.targets = [unitToSelect];
            this.updateInfo();
        }
    };
    this.updateInfo = function () {
        var unit = this.targets.length == 1 ? this.targets[0] : null;
        if (this.doChangeInfo != null) {
            this.doChangeInfo.stop();
        }
        this.doChangeInfo = new Scene()
            .fork()
            .tween(this, 15, 'easeInBack', { infoBoxFadeness: 1.0 })
            .end()
            .tween(this, 15, 'easeInOutSine', { infoFadeness: 1.0 })
            .resync()
            .call(function () {
            this.unitToShowInfo = unit;
            if (this.unitToShowInfo !== null) {
                this.statusNames = !this.unitToShowInfo.isAlive() ? ["Knocked Out"] : [];
                for (var i = 0; i < this.unitToShowInfo.statuses.length; ++i) {
                    this.statusNames.push(this.unitToShowInfo.statuses[i].name);
                }
            }
        }.bind(this))
            .fork()
            .tween(this, 15, 'easeOutBack', { infoBoxFadeness: 0.0 })
            .end()
            .tween(this, 15, 'easeInOutSine', { infoFadeness: 0.0 })
            .run();
    };
}
// .getInput() method
// Checks for player input and updates the state of the menu accordingly.
TargetMenu.prototype.getInput = function () {
    switch (AreKeysLeft() ? GetKey() : null) {
        case GetPlayerKey(PLAYER_1, PLAYER_KEY_A):
            new Scene()
                .fork()
                .tween(this, 15, 'easeInBack', { infoBoxFadeness: 1.0 })
                .end()
                .tween(this, 15, 'easeInOutSine', { infoFadeness: 1.0 })
                .resync()
                .call(function () { this.isChoiceMade = true; }.bind(this))
                .run();
            break;
        case GetPlayerKey(PLAYER_1, PLAYER_KEY_B):
            this.targets = null;
            new Scene()
                .fork()
                .tween(this, 15, 'easeInBack', { infoBoxFadeness: 1.0 })
                .end()
                .tween(this, 15, 'easeInOutSine', { infoFadeness: 1.0 })
                .resync()
                .call(function () { this.isChoiceMade = true; }.bind(this))
                .run();
            break;
        case GetPlayerKey(PLAYER_1, PLAYER_KEY_UP):
            if (!this.isTargetLocked) {
                this.moveCursor(-1);
            }
            break;
        case GetPlayerKey(PLAYER_1, PLAYER_KEY_DOWN):
            if (!this.isTargetLocked) {
                this.moveCursor(1);
            }
            break;
        case GetPlayerKey(PLAYER_1, PLAYER_KEY_LEFT):
            if (!this.isTargetLocked && this.targets != null) {
                if (!this.isGroupCast) {
                    this.targets = [this.battle.enemiesOf(this.unit)[0]];
                }
                else {
                    this.targets = this.battle.enemiesOf(this.unit);
                }
                this.updateInfo();
            }
            break;
        case GetPlayerKey(PLAYER_1, PLAYER_KEY_RIGHT):
            if (!this.isTargetLocked && this.targets != null) {
                if (!this.isGroupCast) {
                    this.targets = [this.unit];
                }
                else {
                    this.targets = this.battle.alliesOf(this.unit);
                }
                this.updateInfo();
            }
            break;
    }
};
TargetMenu.prototype.lockTargets = function (targetUnits) {
    this.targets = targetUnits;
    this.isTargetLocked = true;
};
// .open() method
// Opens the targeting menu and waits for the player to select a target.
// Returns:
//     A list of all units chosen by the player.
TargetMenu.prototype.open = function () {
    this.isChoiceMade = false;
    if (!this.isTargetLocked) {
        this.targets = this.usable !== null
            ? this.usable.defaultTargets(this.unit)
            : [this.battle.enemiesOf(this.unit)[0]];
    }
    this.isGroupCast = this.usable !== null ? this.usable.isGroupCast : false;
    this.updateInfo();
    while (AreKeysLeft()) {
        GetKey();
    }
    threads.join(threads.create(this, 10));
    return this.targets;
};
// .render() method
// Renders the menu in its current state.
TargetMenu.prototype.render = function () {
    if (this.targets !== null) {
        for (var i = 0; i < this.targets.length; ++i) {
            this.drawCursor(this.targets[i]);
        }
    }
    if (this.unitToShowInfo != null) {
        SetClippingRectangle(0, 16, 160, GetScreenHeight() - 16);
        var textAlpha = 255 * (1.0 - this.infoBoxFadeness) * (1.0 - this.infoFadeness);
        if (this.isTargetScanOn || this.unitToShowInfo.isPartyMember()) {
            var nameBoxHeight = 20 + 12 * this.statusNames.length;
            var y = 16 - (nameBoxHeight + 20) * this.infoBoxFadeness;
            Rectangle(0, 16, 160, y - 16, CreateColor(0, 0, 0, 128 * (1.0 - this.infoBoxFadeness)));
            this.drawInfoBox(0, y, 160, nameBoxHeight, 160);
            drawTextEx(this.infoFont, 80, y + 4, this.unitToShowInfo.fullName, CreateColor(192, 192, 192, textAlpha), 1, 'center');
            var statusColor = this.statusNames.length == 0 ?
                CreateColor(96, 192, 96, textAlpha) :
                CreateColor(192, 192, 96, textAlpha);
            for (var i = 0; i < this.statusNames.length; ++i) {
                drawTextEx(this.infoFont, 80, y + 16 + 12 * i, this.statusNames[i], CreateColor(192, 192, 96, textAlpha), 1, 'center');
            }
            this.drawInfoBox(0, y + nameBoxHeight, 80, 20, 128);
            drawTextEx(this.infoFont, 40, y + nameBoxHeight + 4, "HP: " + this.unitToShowInfo.hp, CreateColor(192, 192, 144, textAlpha), 1, 'center');
            this.drawInfoBox(80, y + nameBoxHeight, 80, 20, 128);
            drawTextEx(this.infoFont, 120, y + nameBoxHeight + 4, "MP: " + this.unitToShowInfo.mpPool.availableMP, CreateColor(192, 192, 144, textAlpha), 1, 'center');
        }
        else {
            var y = 16 - 20 * this.infoBoxFadeness;
            Rectangle(0, 16, 160, y - 16, CreateColor(0, 0, 0, 128 * (1.0 - this.infoBoxFadeness)));
            this.drawInfoBox(0, y, 160, 20, 160);
            drawTextEx(this.infoFont, 80, y + 4, this.unitToShowInfo.fullName, CreateColor(192, 192, 192, textAlpha), 1, 'center');
        }
        SetClippingRectangle(0, 0, GetScreenWidth(), GetScreenHeight());
    }
};
// .update() method
// Updates the menu for the next frame.
TargetMenu.prototype.update = function () {
    return !this.isChoiceMade;
};
//# sourceMappingURL=data:application/json;base64,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