/***
 * Specs Engine v6: Spectacles Saga Game Engine
  *           Copyright (c) 2017 Power-Command
***/
RequireScript("battleEngine/stat.js");
var SkillUsable = (function () {
    function SkillUsable(skillID, level) {
        if (level === void 0) { level = 1; }
        if (!(skillID in Game.skills))
            throw new ReferenceError("no skill definition for '" + skillID + "'");
        this.levelUpTable = [];
        for (var i = 1; i <= 100; ++i) {
            var xpNeeded = Math.ceil(i > 1 ? Math.pow(i, 3) : 0);
            this.levelUpTable[i] = xpNeeded;
        }
        this.skillInfo = Game.skills[skillID];
        this.experience = this.levelUpTable[level];
        this.givesExperience = true;
        this.isGroupCast = from(['allEnemies', 'allAllies'])
            .anyIs(this.skillInfo.targetType);
        this.name = this.skillInfo.name;
        this.skillID = skillID;
        this.useAiming = true;
        this.allowDeadTarget = 'allowDeadTarget' in this.skillInfo
            ? this.skillInfo.allowDeadTarget
            : false;
    }
    SkillUsable.prototype.defaultTargets = function (user) {
        var _this = this;
        switch (this.skillInfo.targetType) {
            case 'single':
                var enemies = user.battle.enemiesOf(user);
                var target = from(enemies)
                    .where(function (v) { return v.isAlive(); })
                    .sample(1).first();
                if (this.allowDeadTarget && from(enemies).any(function (v) { return !v.isAlive(); })) {
                    target = from(enemies)
                        .where(function (v) { return !v.isAlive(); })
                        .sample(1).first();
                }
                return [target];
            case 'ally':
                var allies = user.battle.alliesOf(user);
                var target = user;
                if (this.allowDeadTarget && from(allies).any(function (v) { return !v.isAlive(); })) {
                    target = from(allies)
                        .where(function (v) { return !v.isAlive(); })
                        .sample(1).first();
                }
                return [target];
            case 'allEnemies':
                return from(user.battle.enemiesOf(user))
                    .where(function (v) { return v.isAlive() || _this.allowDeadUnits; })
                    .toArray();
            case 'allAllies':
                return from(user.battle.alliesOf(user))
                    .where(function (v) { return v.isAlive() || _this.allowDeadUnits; })
                    .toArray();
            default:
                return user;
        }
    };
    SkillUsable.prototype.getLevel = function () {
        for (var level = 100; level >= 2; --level) {
            if (this.experience >= this.levelUpTable[level])
                return level;
        }
        return 1;
    };
    SkillUsable.prototype.getRank = function () {
        return Game.math.skillRank(this.skillInfo);
    };
    SkillUsable.prototype.grow = function (amount) {
        amount = Math.max(Math.round(amount), 0);
        this.experience = Math.min(this.experience + amount, this.levelUpTable[100]);
        term.print("skill " + this.name + " gained " + amount + " EXP", "lv: " + this.getLevel());
    };
    SkillUsable.prototype.isUsable = function (user, stance) {
        if (stance === void 0) { stance = Stance.Attack; }
        var userWeaponType = user.weapon != null ? user.weapon.type : null;
        var skillWeaponType = this.skillInfo.weaponType;
        if (skillWeaponType != null && userWeaponType != skillWeaponType)
            return false;
        var canCharge = ('chargeable' in this.skillInfo ? this.skillInfo.chargeable : true)
            && this.skillInfo.actions.length == 1;
        var isValidCounter = ('allowAsCounter' in this.skillInfo ? this.skillInfo.allowAsCounter : true)
            && this.skillInfo.targetType == 'single' && this.skillInfo.actions.length == 1;
        return this.mpCost(user) <= user.mpPool.availableMP
            && (stance != Stance.Charge || canCharge)
            && (stance != Stance.Counter || isValidCounter)
            && stance != Stance.Guard;
    };
    SkillUsable.prototype.mpCost = function (user) {
        return Math.min(Math.max(Math.ceil(Game.math.mp.usage(this.skillInfo, this.getLevel(), user.battlerInfo)), 0), 999);
    };
    SkillUsable.prototype.peekActions = function () {
        return this.skillInfo.actions;
    };
    SkillUsable.prototype.use = function (unit, targets) {
        if (!this.isUsable(unit, unit.stance))
            throw new Error(unit.name + " tried to use unusable skill " + this.name);
        term.print(unit.name + " is using " + this.name, "targ: " + (targets.length > 1 ? "[multi]" : targets[0].name));
        if (unit.weapon != null && this.skillInfo.weaponType != null)
            term.print("weapon is " + unit.weapon.name, "lv: " + unit.weapon.level);
        unit.mpPool.use(this.mpCost(unit));
        var growthRate = 'growthRate' in this.skillInfo ? this.skillInfo.growthRate : 1.0;
        var targetInfos = [];
        for (var i = 0; i < targets.length; ++i)
            targetInfos.push(targets[i].battlerInfo);
        var experience = Game.math.experience.skill(this.skillInfo, unit.battlerInfo, targetInfos);
        this.grow(experience);
        var eventData = { skill: clone(this.skillInfo) };
        unit.raiseEvent('useSkill', eventData);
        unit.battle.notifyAIs('skillUsed', unit.id, this.skillID, unit.stance, from(targets).select(function (v) { return v.id; }).toArray());
        return eventData.skill.actions;
    };
    return SkillUsable;
}());
//# sourceMappingURL=data:application/json;base64,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