/***
 * Specs Engine v6: Spectacles Saga Game Engine
  *           Copyright (c) 2017 Power-Command
***/
RequireScript('battleEngine/targetMenu.js');
// MoveMenu() constructor
// Creates an object representing a move-choosing menu.
// Arguments:
//     unit:   The BattleUnit the menu belongs to.
//     battle: The battle session during which the menu will be shown.
//     stance: The stance this menu will be used for.
function MoveMenu(unit, battle, stance) {
    this.lockedCursorColor = CreateColor(0, 36, 72, 255);
    this.moveRankColor = CreateColor(255, 255, 255, 255);
    this.normalCursorColor = CreateColor(0, 72, 144, 255);
    this.textColor = CreateColor(255, 255, 255, 255);
    this.usageTextColor = CreateColor(255, 192, 0, 255);
    this.battle = battle;
    this.drawers = null;
    this.expansion = 0.0;
    this.fadeness = 0.0;
    this.font = GetSystemFont();
    this.isExpanded = false;
    this.menuStance = stance;
    this.menuThread = null;
    this.moveCursor = 0;
    this.moveCursorColor = CreateColor(0, 0, 0, 0);
    this.moveMenu = null;
    this.selection = null;
    this.stance = null;
    this.topCursor = 0;
    this.topCursorColor = CreateColor(0, 0, 0, 0);
    this.unit = unit;
    var drawerTable = {};
    from(this.unit.skills).each(function (skill) {
        var category = skill.skillInfo.category;
        if (!(category in drawerTable)) {
            drawerTable[category] = {
                name: Game.skillCategories[category],
                contents: [],
                cursor: 0
            };
        }
        drawerTable[category].contents.push(skill);
    });
    this.drawers = [];
    for (var category in drawerTable) {
        this.drawers.push(drawerTable[category]);
    }
    if (stance == Stance.Attack) {
        this.drawers = this.drawers.concat([
            { name: "Item", contents: this.unit.items, cursor: 0 }
        ]);
    }
    this.chooseMove = new Scene()
        .fork()
        .tween(this.moveCursorColor, 7, 'easeInOutSine', this.lockedCursorColor)
        .end()
        .fork()
        .tween(this, 15, 'easeInBack', { expansion: 0.0 })
        .end()
        .tween(this, 15, 'easeInBack', { fadeness: 0.0 });
    this.hideMoveList = new Scene()
        .fork()
        .tween(this.moveCursorColor, 15, 'linear', CreateColor(0, 0, 0, 0))
        .end()
        .fork()
        .tween(this.topCursorColor, 15, 'easeInOutSine', this.normalCursorColor)
        .end()
        .tween(this, 15, 'easeInBack', { expansion: 0.0 });
    this.showMenu = new Scene()
        .fork()
        .tween(this.topCursorColor, 15, 'easeOutQuad', CreateColor(192, 192, 192, 255))
        .tween(this.topCursorColor, 15, 'easeOutQuad', this.normalCursorColor)
        .end()
        .tween(this, 30, 'easeOutBounce', { fadeness: 1.0 });
    this.showMoveList = new Scene()
        .fork()
        .tween(this.topCursorColor, 15, 'easeInOutSine', this.lockedCursorColor)
        .end()
        .fork()
        .tween(this.moveCursorColor, 15, 'linear', this.normalCursorColor)
        .end()
        .tween(this, 15, 'easeOutExpo', { expansion: 1.0 });
    this.drawCursor = function (x, y, width, height, cursorColor, isLockedIn, isEnabled) {
        if (isEnabled === void 0) { isEnabled = true; }
        var color;
        var color2;
        color = isEnabled ? cursorColor : CreateColor(96, 96, 96, cursorColor.alpha);
        color2 = BlendColors(color, CreateColor(0, 0, 0, color.alpha));
        if (isLockedIn) {
            var mainColor = color;
            color = color2;
            color2 = mainColor;
        }
        var halfHeight = Math.round(height / 2);
        GradientRectangle(x, y, width, halfHeight, color2, color2, color, color);
        GradientRectangle(x, y + halfHeight, width, height - halfHeight, color, color, color2, color2);
        OutlinedRectangle(x, y, width, height, CreateColor(0, 0, 0, cursorColor.alpha / 2));
    };
    this.drawItemBox = function (x, y, width, height, alpha, isSelected, isLockedIn, cursorColor, isEnabled) {
        if (isEnabled === void 0) { isEnabled = true; }
        Rectangle(x, y, width, height, CreateColor(0, 0, 0, alpha));
        OutlinedRectangle(x, y, width, height, CreateColor(0, 0, 0, 24));
        if (isSelected) {
            this.drawCursor(x, y, width, height, cursorColor, isLockedIn, isEnabled);
        }
    };
    this.drawMoveItem = function (x, y, item, isSelected, isLockedIn) {
        var alpha = 255 * this.fadeness * this.expansion;
        var isEnabled = item.isEnabled;
        var textColor = isSelected ? this.textColor : CreateColor(128, 128, 128, alpha);
        var usageTextColor = isSelected ? this.usageTextColor : BlendColors(this.usageTextColor, CreateColor(0, 0, 0, this.usageTextColor.alpha));
        textColor = isEnabled ? textColor : CreateColor(0, 0, 0, 32 * alpha / 255);
        usageTextColor = isEnabled ? usageTextColor : CreateColor(0, 0, 0, 32 * alpha / 255);
        this.drawItemBox(x, y, 160, 18, alpha * 128 / 255, isSelected, isLockedIn, this.moveCursorColor, isEnabled);
        var rankBoxColor = isEnabled ? BlendColors(item.idColor, CreateColor(0, 0, 0, item.idColor.alpha))
            : BlendColorsWeighted(item.idColor, CreateColor(0, 0, 0, item.idColor.alpha), 25, 75);
        var rankColor = isEnabled ? item.idColor : BlendColorsWeighted(item.idColor, CreateColor(0, 0, 0, item.idColor.alpha), 33, 66);
        Rectangle(x + 5, y + 2, 14, 14, rankBoxColor);
        OutlinedRectangle(x + 5, y + 2, 14, 14, CreateColor(0, 0, 0, rankBoxColor.alpha / 2));
        drawTextEx(this.font, x + 12, y + 3, isFinite(item.rank) ? item.rank : "?", rankColor, 1, 'center');
        drawTextEx(this.font, x + 24, y + 3, item.name, textColor, 1 * isEnabled);
        if (item.mpCost > 0) {
            this.drawText(this.font, x + 141, y + 1, isEnabled, textColor, item.mpCost, 'right');
            this.drawText(this.font, x + 142, y + 5, isEnabled, usageTextColor, "MP");
        }
        else if (item.usable instanceof ItemUsable) {
            this.drawText(this.font, x + 148, y + 3, isEnabled, textColor, item.usable.usesLeft, 'right');
            this.drawText(this.font, x + 149, y + 3, isEnabled, usageTextColor, "x");
        }
    };
    this.drawText = function (font, x, y, shadowDistance, color, text, alignment) {
        if (alignment === void 0) { alignment = 'left'; }
        var Align = {
            left: function (font, x, text) { return x; },
            center: function (font, x, text) { return x - font.getStringWidth(text) / 2; },
            right: function (font, x, text) { return x - font.getStringWidth(text); },
        };
        if (!(alignment in Align))
            throw new Error("invalid text alignment '" + alignment + "'.");
        x = Align[alignment](font, x, text);
        font.setColorMask(CreateColor(0, 0, 0, color.alpha));
        font.drawText(x + shadowDistance, y + shadowDistance, text);
        font.setColorMask(color);
        font.drawText(x, y, text);
    };
    this.drawTopItem = function (x, y, width, item, isSelected) {
        var isEnabled = item.contents.length > 0;
        this.drawItemBox(x, y, width, 18, 144 * this.fadeness, isSelected, this.isExpanded, this.topCursorColor, isEnabled);
        var textColor = isSelected ? CreateColor(255, 255, 255, 255 * this.fadeness) : CreateColor(128, 128, 128, 255 * this.fadeness);
        textColor = isEnabled ? textColor : CreateColor(0, 0, 0, 32 * this.fadeness);
        this.drawText(this.font, x + width / 2, y + 3, isEnabled, textColor, item.name.substr(0, 3), 'center');
    };
    this.updateTurnPreview = function () {
        var nextMoveOrRank;
        if (this.stance != Stance.Guard) {
            if (this.isExpanded) {
                nextMoveOrRank = this.moveMenu[this.moveCursor].usable;
            }
            else {
                var drawer = this.drawers[this.topCursor];
                nextMoveOrRank = drawer.contents.length > 0 ? drawer.contents[drawer.cursor] : Game.defaultItemRank;
            }
        }
        else {
            nextMoveOrRank = Game.stanceChangeRank;
        }
        var nextActions = isNaN(nextMoveOrRank) ? nextMoveOrRank.peekActions() : [nextMoveOrRank];
        if (this.stance == Stance.Charge)
            nextActions = [1].concat(nextActions);
        var prediction = this.battle.predictTurns(this.unit, nextActions);
        this.battle.ui.hud.turnPreview.set(prediction);
    };
}
// .getInput() method
// Checks for player input and updates the state of the menu accordingly.
MoveMenu.prototype.getInput = function () {
    var key = AreKeysLeft() ? GetKey() : null;
    /*if (this.showMenu.isRunning()) {
        return;
    }*/
    if (key == GetPlayerKey(PLAYER_1, PLAYER_KEY_A)) {
        if (!this.isExpanded && this.drawers[this.topCursor].contents.length > 0) {
            var usables = this.drawers[this.topCursor].contents;
            this.moveMenu = [];
            for (var i = 0; i < usables.length; ++i) {
                var menuItem = {
                    name: usables[i].name,
                    idColor: CreateColor(192, 192, 192, 255),
                    isEnabled: usables[i].isUsable(this.unit, this.stance),
                    mpCost: usables[i].mpCost(this.unit),
                    rank: usables[i].getRank(),
                    usable: usables[i]
                };
                var actions = menuItem.usable.peekActions();
                for (var i2 = 0; i2 < actions.length; ++i2) {
                    for (var i3 = 0; i3 < actions[i2].effects.length; ++i3) {
                        if ('element' in actions[i2].effects[i3]) {
                            menuItem.idColor = Game.elements[actions[i2].effects[i3].element].color;
                        }
                    }
                }
                this.moveMenu.push(menuItem);
            }
            this.moveCursor = this.drawers[this.topCursor].cursor;
            this.isExpanded = true;
            this.hideMoveList.stop();
            this.showMoveList.run();
            this.updateTurnPreview();
        }
        else if (this.isExpanded && this.moveMenu[this.moveCursor].isEnabled) {
            this.drawers[this.topCursor].cursor = this.moveCursor;
            this.selection = this.moveMenu[this.moveCursor].usable;
            this.showMoveList.stop();
            this.chooseMove.run();
        }
    }
    else if (key == GetPlayerKey(PLAYER_1, PLAYER_KEY_B) && this.isExpanded) {
        this.drawers[this.topCursor].cursor = this.moveCursor;
        this.isExpanded = false;
        this.showMoveList.stop();
        this.hideMoveList.run();
    }
    else if (key == GetPlayerKey(PLAYER_1, PLAYER_KEY_Y) && this.stance != Stance.Guard) {
        this.stance = this.stance == Stance.Attack ? Stance.Charge
            : Stance.Guard;
        this.updateTurnPreview();
        if (this.stance == Stance.Guard) {
            this.showMoveList.stop();
            this.chooseMove.run();
        }
    }
    else if (!this.isExpanded && key == GetPlayerKey(PLAYER_1, PLAYER_KEY_LEFT)) {
        --this.topCursor;
        if (this.topCursor < 0) {
            this.topCursor = this.drawers.length - 1;
        }
        this.updateTurnPreview();
    }
    else if (!this.isExpanded && key == GetPlayerKey(PLAYER_1, PLAYER_KEY_RIGHT)) {
        ++this.topCursor;
        if (this.topCursor >= this.drawers.length) {
            this.topCursor = 0;
        }
        this.updateTurnPreview();
    }
    else if (this.isExpanded && key == GetPlayerKey(PLAYER_1, PLAYER_KEY_UP)) {
        this.moveCursor = this.moveCursor - 1 < 0 ? this.moveMenu.length - 1 : this.moveCursor - 1;
        this.updateTurnPreview();
    }
    else if (this.isExpanded && key == GetPlayerKey(PLAYER_1, PLAYER_KEY_DOWN)) {
        this.moveCursor = (this.moveCursor + 1) % this.moveMenu.length;
        this.updateTurnPreview();
    }
};
// .open() method
// Opens the menu to allow the player to choose an action.
MoveMenu.prototype.open = function () {
    this.battle.suspend();
    this.battle.ui.hud.highlight(this.unit);
    var chosenTargets = null;
    this.stance = this.lastStance = this.menuStance;
    while (chosenTargets === null) {
        this.expansion = 0.0;
        this.isExpanded = false;
        this.selection = null;
        this.stance = this.lastStance;
        while (AreKeysLeft()) {
            GetKey();
        }
        this.showMenu.run();
        this.updateTurnPreview();
        this.menuThread = threads.create(this, 10);
        threads.join(this.menuThread);
        switch (this.stance) {
            case Stance.Attack:
            case Stance.Charge:
                var name = this.stance == Stance.Charge
                    ? "CS " + this.selection.name
                    : this.selection.name;
                var chosenTargets = new TargetMenu(this.unit, this.battle, this.selection, name).open();
                break;
            case Stance.Counter:
                var targetMenu = new TargetMenu(this.unit, this.battle, null, "GS " + this.selection.name);
                targetMenu.lockTargets([this.unit.counterTarget]);
                var chosenTargets = targetMenu.open();
                break;
            case Stance.Guard:
                var targetMenu = new TargetMenu(this.unit, this.battle, null, "Guard");
                targetMenu.lockTargets([this.unit]);
                var chosenTargets = targetMenu.open();
                break;
        }
    }
    this.battle.ui.hud.highlight(null);
    this.battle.resume();
    return {
        usable: this.selection,
        stance: this.stance,
        targets: chosenTargets
    };
};
// .render() method
// Renders the menu in its current state.
MoveMenu.prototype.render = function () {
    var yOrigin = -54 * (1.0 - this.fadeness) + 16;
    var stanceText = this.stance == Stance.Charge ? "CS"
        : this.stance == Stance.Counter ? "GS"
            : this.stance == Stance.Guard ? "GS"
                : "AS";
    Rectangle(0, yOrigin, 136, 16, CreateColor(0, 0, 0, 160 * this.fadeness));
    OutlinedRectangle(0, yOrigin, 136, 16, CreateColor(0, 0, 0, 24 * this.fadeness));
    Rectangle(136, yOrigin, 24, 16, CreateColor(0, 0, 0, 176 * this.fadeness));
    OutlinedRectangle(136, yOrigin, 24, 16, CreateColor(0, 0, 0, 24 * this.fadeness));
    this.drawText(this.font, 68, yOrigin + 2, 1, CreateColor(160, 160, 160, 255 * this.fadeness), this.unit.fullName, 'center');
    this.drawText(this.font, 148, yOrigin + 2, 1, CreateColor(255, 255, 128, 255 * this.fadeness), stanceText, 'center');
    var itemWidth = 160 / this.drawers.length;
    var litTextColor = CreateColor(255, 255, 255, 255);
    var dimTextColor = CreateColor(192, 192, 192, 255);
    Rectangle(0, 16, 160, yOrigin - 16, CreateColor(0, 0, 0, 192 * this.fadeness));
    for (var i = 0; i < this.drawers.length; ++i) {
        var x = Math.floor(i * itemWidth);
        var width = Math.floor((i + 1) * itemWidth) - x;
        this.drawTopItem(x, yOrigin + 16, width, this.drawers[i], i == this.topCursor);
    }
    var itemY;
    if (this.expansion > 0.0) {
        SetClippingRectangle(0, yOrigin + 34, 160, GetScreenHeight() - (yOrigin + 34));
        var height = this.moveMenu.length * 16;
        var y = yOrigin + 34 - height * (1.0 - this.expansion);
        Rectangle(0, 34, 160, y - 34, CreateColor(0, 0, 0, 128 * this.expansion * this.fadeness));
        itemY = y;
        for (var i = 0; i < this.moveMenu.length; ++i) {
            this.drawMoveItem(0, itemY, this.moveMenu[i], i == this.moveCursor, this.chooseMove.isRunning());
            itemY += 18;
        }
        SetClippingRectangle(0, 0, GetScreenWidth(), GetScreenHeight());
    }
    else {
        itemY = yOrigin + 34;
    }
};
// .update() method
// Updates the entity's state for the next frame.
MoveMenu.prototype.update = function () {
    return (this.stance != Stance.Guard && this.selection === null)
        || this.chooseMove.isRunning();
};
//# sourceMappingURL=data:application/json;base64,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