/***
 * Specs Engine v6: Spectacles Saga Game Engine
  *           Copyright (c) 2017 Power-Command
***/
var __read = (this && this.__read) || function (o, n) {
    var m = typeof Symbol === "function" && o[Symbol.iterator];
    if (!m) return o;
    var i = m.call(o), r, ar = [], e;
    try {
        while ((n === void 0 || n-- > 0) && !(r = i.next()).done) ar.push(r.value);
    }
    catch (error) { e = { error: error }; }
    finally {
        try {
            if (r && !r.done && (m = i["return"])) m.call(i);
        }
        finally { if (e) throw e.error; }
    }
    return ar;
};
var __spread = (this && this.__spread) || function () {
    for (var ar = [], i = 0; i < arguments.length; i++) ar = ar.concat(__read(arguments[i]));
    return ar;
};
var __values = (this && this.__values) || function (o) {
    var m = typeof Symbol === "function" && o[Symbol.iterator], i = 0;
    if (m) return m.call(o);
    return {
        next: function () {
            if (o && i >= o.length) o = void 0;
            return { value: o && o[i++], done: !o };
        }
    };
};
RequireScript('battleEngine/item.js');
RequireScript('battleEngine/moveMenu.js');
RequireScript('battleEngine/mpPool.js');
RequireScript('battleEngine/skill.js');
RequireScript('battleEngine/stat.js');
RequireScript('battleEngine/statusEffect.js');
var Row = {
    Front: -1,
    Middle: 0,
    Rear: 1,
};
var Stance = {
    Attack: 0,
    Guard: 1,
    Counter: 2,
    Charge: 3,
    Hippo: 4,
};
var BattleUnit = (function () {
    function BattleUnit(battle, basis, position, startingRow, mpPool) {
        this.actionQueue = [];
        this.actor = null;
        this.affinities = [];
        this.ai = null;
        this.allowTargetScan = false;
        this.battle = battle;
        this.battlerInfo = {};
        this.counterTarget = null;
        this.cv = 0;
        this.hp = 0;
        this.lastAttacker = null;
        this.lazarusFlag = false;
        this.moveTargets = null;
        this.mpPool = null;
        this.newSkills = [];
        this.newStance = Stance.Attack;
        this.partyMember = null;
        this.row = startingRow;
        this.skills = [];
        this.stance = Stance.Attack;
        this.stats = {};
        this.statuses = [];
        this.tag = random.string();
        this.turnRatio = 1.0;
        this.weapon = null;
        if (basis instanceof PartyMember) {
            this.partyMember = basis;
            this.id = this.partyMember.characterID;
            this.character = Game.characters[this.partyMember.characterID];
            this.baseStats = this.character.baseStats;
            this.tier = 1;
            this.maxHP = Math.round(Math.max(Game.math.hp(this.character, this.partyMember.level, this.tier), 1));
            this.hp = this.maxHP;
            this.name = this.partyMember.name;
            this.fullName = this.partyMember.fullName;
            this.allowTargetScan = this.partyMember.isTargetScanOn;
            this.skills = __spread(this.partyMember.getUsableSkills());
            this.items = __spread(this.partyMember.items);
            for (var statID in this.baseStats)
                this.stats[statID] = this.partyMember.stats[statID];
            this.weapon = Game.weapons[this.partyMember.weaponID];
        }
        else {
            if (!(basis in Game.enemies))
                throw new ReferenceError("enemy template '" + basis + "' doesn't exist!");
            this.enemyInfo = Game.enemies[basis];
            this.baseStats = this.enemyInfo.baseStats;
            this.affinities = 'damageModifiers' in this.enemyInfo ? this.enemyInfo.damageModifiers : [];
            this.id = basis;
            this.name = this.enemyInfo.name;
            this.fullName = 'fullName' in this.enemyInfo ? this.enemyInfo.fullName : this.enemyInfo.name;
            for (var statID in this.baseStats)
                this.stats[statID] = new Stat(this.baseStats[statID], battle.getLevel(), false);
            if ('items' in this.enemyInfo) {
                this.items = from(this.enemyInfo.items)
                    .select(function (v) { return new ItemUsable(v); })
                    .toArray();
            }
            else {
                this.items = [];
            }
            this.tier = 'tier' in this.enemyInfo ? this.enemyInfo.tier : 1;
            this.turnRatio = 'turnRatio' in this.enemyInfo ? this.enemyInfo.turnRatio : 1;
            this.maxHP = Math.round(Math.max(Game.math.hp(this.enemyInfo, battle.getLevel(), this.tier), 1));
            this.hp = this.maxHP;
            this.weapon = Game.weapons[this.enemyInfo.weapon];
            if ('hasLifeBar' in this.enemyInfo && this.enemyInfo.hasLifeBar)
                this.battle.ui.hud.createEnemyHPGauge(this);
            this.ai = Reflect.construct(this.enemyInfo.aiClass, [this, battle]);
            this.battle.registerAI(this.ai);
        }
        this.attackMenu = new MoveMenu(this, battle, Stance.Attack);
        this.counterMenu = new MoveMenu(this, battle, Stance.Counter);
        this.refreshInfo();
        this.mpPool = mpPool !== void null ? mpPool
            : new MPPool(this.id + "MP", Math.round(Math.max(Game.math.mp.capacity(this.battlerInfo), 0)));
        this.actor = battle.ui.createActor(this.name, position, this.row, this.isPartyMember() ? 'party' : 'enemy');
        if (this.isPartyMember())
            this.battle.ui.hud.setPartyMember(position == 2 ? 0 : position == 0 ? 2 : position, this, this.hp, this.maxHP);
        if (!this.isPartyMember())
            this.actor.enter(true);
        this.resetCounter(Game.defaultMoveRank, true);
        this.registerCommands();
        var unitType = this.ai === null ? "player" : "AI";
        term.print("create " + unitType + " unit '" + this.name + "'", "hp: " + this.hp + "/" + this.maxHP, "id: " + this.tag);
    }
    BattleUnit.prototype.dispose = function () {
        if (this.ai !== null)
            this.battle.unregisterAI(this.ai);
        term.undefine(this.id);
    };
    BattleUnit.prototype.addStatus = function (statusID, isGuardable) {
        if (isGuardable === void 0) { isGuardable = false; }
        if (this.isAlive() && !this.hasStatus(statusID)) {
            var statusName = Game.statuses[statusID].name;
            var isOverruled = from(this.statuses)
                .any(function (v) { return v.overrules(statusID); });
            if (!this.isPartyMember() && from(this.enemyInfo.immunities).anyIs(statusID)) {
                if (!isGuardable)
                    this.actor.showHealing("immune", CreateColor(192, 192, 192, 255));
                term.print(this.name + " is immune to " + statusName);
            }
            else if (isOverruled) {
                if (!isGuardable)
                    this.actor.showHealing("ward", CreateColor(192, 192, 192, 255));
                term.print(statusName + " overruled by another of " + this.name + "'s statuses");
            }
            else if (this.stance !== Stance.Guard || !isGuardable) {
                var eventData = { unit: this, statusID: statusID, cancel: false };
                this.battle.raiseEvent('unitAfflicted', eventData);
                if (!eventData.cancel)
                    this.raiseEvent('afflicted', eventData);
                if (!eventData.cancel) {
                    var effect = new StatusEffect(eventData.statusID, this);
                    this.statuses.push(effect);
                    this.battlerInfo.statuses = from(this.statuses)
                        .select(function (v) { return v.statusID; })
                        .toArray();
                    term.print("status " + effect.name + " installed on " + this.name);
                }
                else {
                    if (!isGuardable)
                        this.actor.showHealing("ward", CreateColor(192, 192, 192, 255));
                    term.print("status " + statusName + " for " + this.name + " blocked per status/FC");
                }
            }
            else {
                term.print("status " + statusName + " for " + this.name + " blocked by Guard");
            }
        }
    };
    BattleUnit.prototype.announce = function (text) {
        var bannerColor = this.isPartyMember()
            ? Color.mix(Color.Blue, Color.White, 75, 25).fade(0.75)
            : Color.mix(Color.Red, Color.White, 75, 25).fade(0.75);
        this.battle.ui.announceAction(text, this.isPartyMember() ? 'party' : 'enemy', bannerColor);
    };
    BattleUnit.prototype.beginCycle = function () {
        if (!this.isAlive())
            return;
        this.refreshInfo();
        for (var i = 0; i < this.statuses.length; ++i)
            this.statuses[i].beginCycle();
        var eventData = { battlerInfo: this.battlerInfo };
        this.raiseEvent('beginCycle', eventData);
        var baseStatSum = 0;
        var statSum = 0;
        var numStats = 0;
        for (var statID in this.baseStats) {
            ++numStats;
            this.battlerInfo.stats[statID] = Math.round(this.battlerInfo.stats[statID]);
            statSum += this.battlerInfo.stats[statID];
            this.battlerInfo.baseStats[statID] = Math.round(this.battlerInfo.baseStats[statID]);
            baseStatSum += this.battlerInfo.baseStats[statID];
        }
        this.battlerInfo.statAverage = Math.round(statSum / numStats);
        this.battlerInfo.baseStatAverage = Math.round(baseStatSum / numStats);
        this.mpPool.restore(this.battlerInfo.statAverage / 10);
    };
    BattleUnit.prototype.beginTargeting = function (actingUnit) {
        this.lastAttacker = actingUnit;
    };
    BattleUnit.prototype.clearQueue = function () {
        if (this.actionQueue.length > 0) {
            this.actionQueue = [];
            term.print("clear " + this.name + "'s action queue");
        }
    };
    BattleUnit.prototype.die = function () {
        this.battle.notifyAIs('unitKilled', this.id);
        this.lazarusFlag = false;
        this.hp = 0;
        this.battle.ui.hud.setHP(this, this.hp);
        this.statuses = [];
        this.actor.animate('die');
        term.print("death comes near " + this.fullName);
    };
    BattleUnit.prototype.endCycle = function () {
        if (!this.isAlive())
            return;
        if (this.stance === Stance.Counter) {
            this.cv = 0;
            if (this.ai == null) {
                this.actor.animate('active');
                this.battle.ui.hud.turnPreview.set(this.battle.predictTurns(this));
                term.print("ask player for " + this.name + "'s GS counterattack");
                chosenMove = this.counterMenu.open();
            }
            else {
                chosenMove = this.ai.getNextMove();
                chosenMove.targets = [this.counterTarget];
            }
            this.queueMove(chosenMove);
            this.performAction(this.getNextAction(), chosenMove);
            this.actor.animate('dormant');
            this.newStance = Stance.Attack;
        }
        if (this.newStance !== this.stance) {
            this.stance = this.newStance;
            this.battle.notifyAIs('stanceChanged', this.id, this.stance);
            var stanceName = this.stance === Stance.Guard ? "Guard"
                : this.stance === Stance.Counter ? "Counter"
                    : "Attack";
            term.print(this.name + " now in " + stanceName + " Stance");
        }
    };
    BattleUnit.prototype.endTargeting = function () {
        this.lastAttacker = null;
    };
    BattleUnit.prototype.evade = function (actingUnit, action) {
        this.actor.showHealing("miss", CreateColor(192, 192, 192, 255));
        term.print(this.name + " evaded " + actingUnit.name + "'s attack");
        var isGuardBroken = 'preserveGuard' in action ? !action.preserveGuard : true;
        var isMelee = 'isMelee' in action ? action.isMelee : false;
        if (isMelee && this.stance === Stance.Guard && isGuardBroken) {
            this.stance = Stance.Counter;
            this.counterTarget = actingUnit;
            term.print(this.name + "'s Counter Stance activated");
        }
    };
    BattleUnit.prototype.getHealth = function () {
        return Math.ceil(100 * this.hp / this.maxHP);
    };
    BattleUnit.prototype.getLevel = function () {
        if (this.partyMember != null) {
            return this.partyMember.level;
        }
        else {
            return this.battle.getLevel();
        }
    };
    BattleUnit.prototype.growSkill = function (skillID, experience) {
        if (!this.isPartyMember())
            return;
        var hasSkill = false;
        for (var i = 0; i < this.skills.length; ++i) {
            if (skillID == this.skills[i].skillID) {
                hasSkill = true;
                this.skills[i].grow(experience);
            }
        }
        if (!hasSkill) {
            var skill = this.partyMember.learnSkill(skillID);
            this.skills.push(skill);
            this.newSkills.push(skill);
            term.print(this.name + " learned " + skill.name);
        }
    };
    BattleUnit.prototype.getNextAction = function () {
        if (this.actionQueue.length > 0) {
            term.print(this.actionQueue.length + " outstanding action(s) for " + this.name);
            return this.actionQueue.shift();
        }
        else {
            return null;
        }
    };
    BattleUnit.prototype.hasStatus = function (statusID) {
        return from(this.statuses)
            .any(function (v) { return v.statusID === statusID; });
    };
    BattleUnit.prototype.heal = function (amount, tags, isPriority) {
        if (tags === void 0) { tags = []; }
        if (isPriority === void 0) { isPriority = false; }
        if (!isPriority) {
            var eventData = {
                unit: this,
                amount: Math.round(amount),
                tags: tags,
                cancel: false
            };
            this.battle.raiseEvent('unitHealed', eventData);
            if (!eventData.cancel)
                this.raiseEvent('healed', eventData);
            if (!eventData.cancel)
                amount = Math.round(eventData.amount);
            else
                return;
        }
        if (amount > 0) {
            this.hp = Math.min(this.hp + amount, this.maxHP);
            this.actor.showHealing(amount);
            this.battle.ui.hud.setHP(this, this.hp);
            this.battle.notifyAIs('unitHealed', this, amount, tags);
            term.print("heal " + this.name + " for " + amount + " HP", "now: " + this.hp);
        }
        else if (amount < 0) {
            this.takeDamage(-amount, [], true);
        }
    };
    BattleUnit.prototype.isAlive = function () {
        return this.hp > 0 || this.lazarusFlag;
    };
    BattleUnit.prototype.isPartyMember = function () {
        return this.partyMember != null;
    };
    BattleUnit.prototype.liftStatus = function (statusID) {
        var _this = this;
        var eventData = {
            statusID: statusID,
            cancel: false
        };
        this.raiseEvent('unitCured', eventData);
        if (!eventData.cancel)
            this.raiseEvent('cured', eventData);
        if (!eventData.cancel) {
            from(this.statuses)
                .where(function (i) { return i.statusID === statusID; })
                .besides(function (i) { return term.print("lift status effect " + _this.name + "->" + i.name); })
                .remove();
            this.battlerInfo.statuses = from(this.statuses)
                .select(function (v) { return v.statusID; })
                .toArray();
        }
    };
    BattleUnit.prototype.liftStatusTags = function (tags) {
        var _this = this;
        var activeStatuses = __spread(this.statuses);
        from(activeStatuses)
            .where(function (v) { return from(v.statusDef.tags).anyIn(tags); })
            .each(function (v) { return _this.liftStatus(v.statusID); });
    };
    BattleUnit.prototype.performAction = function (action, move) {
        var targetsInfo = from(move.targets)
            .select(function (v) { return v.battlerInfo; })
            .toArray();
        var eventData = { action: action, targetsInfo: targetsInfo };
        this.raiseEvent('acting', eventData);
        eventData.action.rank = Math.max(Math.round(eventData.action.rank), 0);
        if (this.isAlive()) {
            if (this.stance === Stance.Counter)
                action.accuracyRate = 2.0;
            var unitsHit = this.battle.runAction(action, this, move.targets, move.usable.useAiming);
            if (move.usable.givesExperience && unitsHit.length > 0) {
                var allEnemies = this.battle.enemiesOf(this);
                var experience = {};
                for (var i = 0; i < unitsHit.length; ++i) {
                    if (!unitsHit[i].isAlive() && this.battle.areEnemies(this, unitsHit[i])) {
                        for (var statID in unitsHit[i].baseStats) {
                            if (!(statID in experience)) {
                                experience[statID] = 0;
                            }
                            experience[statID] += Game.math.experience.stat(statID, unitsHit[i].battlerInfo);
                        }
                    }
                }
                for (var statID in experience) {
                    this.stats[statID].grow(experience[statID]);
                    term.print(this.name + " got " + experience[statID] + " EXP for " + Game.statNames[statID], "value: " + this.stats[statID].value);
                }
            }
            this.resetCounter(action.rank);
        }
    };
    BattleUnit.prototype.queueMove = function (move) {
        this.moveUsed = move;
        var alliesInBattle = this.battle.alliesOf(this.moveUsed.targets[0]);
        var alliesAlive = from(alliesInBattle)
            .where(function (v) { return v.isAlive(); })
            .toArray();
        this.moveUsed.targets = this.moveUsed.usable.isGroupCast
            ? this.moveUsed.usable.allowDeadTarget ? alliesInBattle : alliesAlive
            : this.moveUsed.targets;
        if (!this.moveUsed.usable.isGroupCast && !this.moveUsed.targets[0].isAlive()
            && !this.moveUsed.usable.allowDeadTarget) {
            this.moveUsed.targets[0] = random.sample(alliesAlive);
        }
        var nextActions = this.moveUsed.usable.use(this, this.moveUsed.targets);
        if (move.stance === Stance.Counter || move.stance === Stance.Charge) {
            from(nextActions)
                .from(function (action) { return action.effects; })
                .where(function (effect) { return 'power' in effect; })
                .each(function (effect) {
                effect.power *= Game.bonusMultiplier;
                effect.statusChance = 100;
                term.print("stance is Counter/Charge, boost attack", "pow: " + effect.power);
            });
        }
        if (move.stance === Stance.Charge) {
            nextActions.splice(0, 0, {
                announceAs: "Charge",
                rank: 1,
                preserveGuard: true,
                effects: [
                    {
                        targetHint: 'user',
                        type: 'addStatus',
                        status: 'offGuard'
                    }
                ]
            });
        }
        if (nextActions !== null) {
            this.battle.ui.hud.turnPreview.set(this.battle.predictTurns(this, nextActions));
            for (var i = 0; i < nextActions.length; ++i)
                this.actionQueue.push(nextActions[i]);
            if (this.actionQueue.length > 0)
                term.print("queue " + this.actionQueue.length + " action(s) for " + this.moveUsed.usable.name);
        }
        else {
            this.battle.ui.hud.turnPreview.set(this.battle.predictTurns());
        }
    };
    BattleUnit.prototype.raiseEvent = function (eventID, data) {
        // event handlers can change the objects referenced in the data object, for example to
        // change the effects of an action taken by a battler.  if you pass in any objects from
        // the gamedef, they should be cloned first to prevent the event from inadvertantly
        // modifying the original definition.
        if (data === void 0) { data = null; }
        var statuses = __spread(this.statuses);
        from(statuses)
            .each(function (v) { return v.invoke(eventID, data); });
    };
    BattleUnit.prototype.refreshInfo = function () {
        this.battlerInfo.name = this.name;
        this.battlerInfo.affinities = clone(this.affinities);
        this.battlerInfo.health = Math.ceil(100 * this.hp / this.maxHP);
        this.battlerInfo.level = this.getLevel();
        this.battlerInfo.weapon = clone(this.weapon);
        this.battlerInfo.tier = this.tier;
        this.battlerInfo.baseStats = {};
        this.battlerInfo.stats = { maxHP: this.maxHP };
        for (var statID in this.baseStats) {
            this.battlerInfo.baseStats[statID] = this.baseStats[statID];
            this.battlerInfo.stats[statID] = this.stats[statID].value;
        }
        this.battlerInfo.statuses = from(this.statuses)
            .select(function (v) { return v.statusID; })
            .toArray();
        this.battlerInfo.stance = this.stance;
    };
    BattleUnit.prototype.registerCommands = function () {
        term.define(this.id, this, {
            'add': function (statusID) {
                if (statusID in Game.statuses)
                    this.addStatus(statusID);
                else
                    term.print("invalid status ID '" + statusID + "'");
            },
            'lift': function (statusID) {
                if (statusID in Game.statuses)
                    this.liftStatus(statusID);
                else
                    term.print("invalid status ID '" + statusID + "'");
            },
            'damage': function (amount) {
                tags = [].slice.call(arguments, 1);
                amount = Math.max(parseInt(amount), 0);
                this.takeDamage(amount, tags);
            },
            'heal': function (amount) {
                tags = [].slice.call(arguments, 1);
                amount = Math.max(parseInt(amount), 0);
                this.heal(amount, tags);
            },
            'inv': function (instruction) {
                if (arguments.length < 1)
                    return term.print("'" + this.id + " inv': No instruction provided");
                switch (instruction) {
                    case 'add':
                        if (arguments.length < 2)
                            return term.print("'" + this.id + " inv add': Item ID required");
                        var itemID = arguments[1];
                        if (!(itemID in Game.items))
                            return term.print("No such item ID '" + itemID + "'");
                        var defaultUses = 'uses' in Game.items[itemID] ? Game.items[itemID].uses : 1;
                        var itemCount = arguments[2] > 0 ? arguments[2] : defaultUses;
                        var addCount = 0;
                        from(this.items)
                            .where(function (item) { return item.itemID === itemID; })
                            .each(function (item) {
                            item.usesLeft += itemCount;
                            addCount += itemCount;
                        });
                        if (addCount == 0) {
                            var usable = new ItemUsable(itemID);
                            usable.usesLeft = itemCount;
                            this.items.push(usable);
                            addCount = itemCount;
                        }
                        term.print(addCount + "x " + Game.items[itemID].name + " added to " + this.name + "'s inventory");
                        break;
                    case 'munch':
                        new Scene().playSound('Munch.wav').run();
                        this.items.length = 0;
                        term.print("maggie ate " + this.name + "'s entire inventory");
                        break;
                    case 'rm':
                        if (arguments.length < 2)
                            return term.print("'" + this.id + " inv add': Item ID required");
                        var itemID = arguments[1];
                        var itemCount = 0;
                        from(this.items)
                            .where(function (v) { return v.itemID === itemID; })
                            .besides(function (v) { return itemCount += v.usesLeft; })
                            .remove();
                        if (itemCount > 0)
                            term.print(itemCount + "x " + Game.items[itemID].name
                                + " deleted from " + this.name + "'s inventory");
                        else
                            term.print("No " + Game.items[itemID].name + " in " + this.name + "'s inventory");
                        break;
                    default:
                        return term.print("'" + this.id + " inv': Unknown instruction '" + instruction + "'");
                }
            },
            'revive': function () {
                this.resurrect();
            },
            'scan': function (flag) {
                flag = flag.toLowerCase();
                if (flag == 'on')
                    this.allowTargetScan = true;
                if (flag == 'off')
                    this.allowTargetScan = false;
                term.print("Target Scan for " + this.name + " is " + (this.allowTargetScan ? "ON" : "OFF"));
            },
        });
    };
    BattleUnit.prototype.resetCounter = function (rank, isFirstTurn) {
        // note: Rank 0 is treated as a special case; passing 0 or less for rank will always give
        //       the unit its next turn immediately.
        if (isFirstTurn === void 0) { isFirstTurn = false; }
        var divisor = isFirstTurn ? 1.0 : this.turnRatio;
        this.cv = rank > 0
            ? Math.max(Math.round(Game.math.timeUntilNextTurn(this.battlerInfo, rank) / divisor), 1)
            : 1;
        term.print("update " + this.name + "'s CV to " + this.cv, "rank: " + rank);
    };
    BattleUnit.prototype.restoreMP = function (amount) {
        amount = Math.round(amount);
        this.mpPool.restore(amount);
        var color = BlendColorsWeighted(CreateColor(255, 0, 255, 255), CreateColor(255, 255, 255, 255), 33, 66);
        this.actor.showHealing(amount + "MP", color);
    };
    BattleUnit.prototype.resurrect = function (isFullHeal) {
        if (isFullHeal === void 0) { isFullHeal = false; }
        if (!this.isAlive()) {
            this.lazarusFlag = true;
            this.heal(isFullHeal ? this.maxHP : 1);
            this.actor.animate('revive');
            this.resetCounter(Game.reviveRank);
            term.print(this.name + " brought back from the dead");
        }
        else {
            this.actor.showHealing("ward", CreateColor(192, 192, 192, 255));
        }
    };
    BattleUnit.prototype.setGuard = function () {
        term.print(this.name + " will switch to Guard Stance");
        this.announce("Guard");
        this.newStance = Stance.Guard;
        this.resetCounter(Game.stanceChangeRank);
    };
    BattleUnit.prototype.setWeapon = function (weaponID) {
        var weaponDef = Game.weapons[weaponID];
        this.announce("equip " + weaponDef.name);
        this.weapon = weaponDef;
        term.print(this.name + " equipped weapon " + weaponDef.name);
        this.resetCounter(Game.equipWeaponRank);
    };
    BattleUnit.prototype.takeDamage = function (amount, tags, isPriority) {
        if (tags === void 0) { tags = []; }
        if (isPriority === void 0) { isPriority = false; }
        amount = Math.round(amount);
        var multiplier = 1.0;
        for (var i = 0; i < tags.length; ++i) {
            if (tags[i] in this.affinities) {
                multiplier *= this.affinities[tags[i]];
            }
        }
        amount = Math.round(amount * multiplier);
        if (amount > 0 && !isPriority) {
            var eventData = {
                unit: this, amount: amount, tags: tags,
                actingUnit: this.lastAttacker,
                cancel: false
            };
            this.battle.raiseEvent('unitDamaged', eventData);
            if (!eventData.cancel) {
                this.raiseEvent('damaged', eventData);
            }
            if (!eventData.cancel) {
                amount = Math.round(eventData.amount);
            }
            else {
                return;
            }
        }
        if (amount >= 0) {
            if (this.stance != Stance.Attack && this.lastAttacker !== null) {
                amount = Math.round(Game.math.guardStance.damageTaken(amount, tags));
                term.print(this.name + " hit in Guard Stance, reduce damage");
            }
            var oldHPValue = this.hp;
            if (this.stance !== Stance.Hippo || amount < this.hp)
                this.hp = Math.max(this.hp - amount, 0);
            else
                this.hp = 1;
            this.battle.notifyAIs('unitDamaged', this, amount, tags, this.lastAttacker);
            term.print("damage " + this.name + " for " + amount + " HP", "left: " + this.hp);
            if (oldHPValue > 0 || this.lazarusFlag) {
                var elementTags = from(tags).where(function (v) { return v in Game.elements; });
                var damageColor = null;
                try {
                    for (var elementTags_1 = __values(elementTags), elementTags_1_1 = elementTags_1.next(); !elementTags_1_1.done; elementTags_1_1 = elementTags_1.next()) {
                        var tag = elementTags_1_1.value;
                        damageColor = damageColor !== null
                            ? BlendColors(damageColor, Game.elements[tag].color)
                            : Game.elements[tag].color;
                    }
                }
                catch (e_1_1) { e_1 = { error: e_1_1 }; }
                finally {
                    try {
                        if (elementTags_1_1 && !elementTags_1_1.done && (_a = elementTags_1.return)) _a.call(elementTags_1);
                    }
                    finally { if (e_1) throw e_1.error; }
                }
                damageColor = damageColor !== null
                    ? BlendColorsWeighted(damageColor, CreateColor(255, 255, 255, 255), 33, 66)
                    : CreateColor(255, 255, 255, 255);
                this.actor.showDamage(amount, damageColor);
            }
            this.battle.ui.hud.setHP(this, this.hp);
            if (this.hp <= 0 && (oldHPValue > 0 || this.lazarusFlag)) {
                term.print(this.name + " dying due to lack of HP");
                this.lazarusFlag = true;
                var eventData = { unit: this, cancel: false };
                this.battle.raiseEvent('unitDying', eventData);
                if (!eventData.cancel) {
                    this.raiseEvent('dying', eventData);
                }
                this.lazarusFlag = eventData.cancel;
                if (!this.lazarusFlag) {
                    this.die();
                }
                else {
                    term.print("suspend KO for " + this.name + " per status/FC");
                }
            }
        }
        else if (amount < 0) {
            this.heal(Math.abs(amount), tags);
        }
        var e_1, _a;
    };
    BattleUnit.prototype.takeHit = function (actingUnit, action) {
        var eventData = {
            actingUnitInfo: actingUnit.battlerInfo,
            action: action,
            stance: actingUnit.stance
        };
        this.raiseEvent('attacked', eventData);
        var isGuardBroken = 'preserveGuard' in action ? !action.preserveGuard : true;
        var isMelee = 'isMelee' in action ? action.isMelee : false;
        if (this.stance === Stance.Guard && isMelee && isGuardBroken) {
            action.accuracyRate = 0.0; //'accuracyRate' in action ? 0.5 * action.accuracyRate : 0.5;
        }
        if (this.stance === Stance.Guard && isGuardBroken) {
            term.print(this.name + "'s Guard Stance broken", "by: " + actingUnit.name);
            this.newStance = Stance.Attack;
            this.resetCounter(Game.guardBreakRank);
        }
    };
    BattleUnit.prototype.tick = function () {
        if (!this.isAlive())
            return false;
        if (--this.cv == 0) {
            this.battle.suspend();
            if (this.stance === Stance.Guard) {
                this.stance = this.newStance = Stance.Attack;
                this.battle.notifyAIs('stanceChanged', this.id, this.stance);
                term.print(this.name + "'s Guard Stance expired");
            }
            else if (this.stance === Stance.Counter) {
                this.newStance = Stance.Attack;
            }
            term.print(this.name + "'s turn is up");
            this.actor.animate('active');
            this.battle.notifyAIs('unitReady', this.id);
            var eventData = { skipTurn: false };
            this.raiseEvent('beginTurn', eventData);
            if (!this.isAlive()) {
                this.battle.resume();
                return true;
            }
            if (eventData.skipTurn) {
                this.clearQueue();
                term.print("skip " + this.name + "'s turn per status/FC");
                this.resetCounter(Game.defaultMoveRank);
                this.battle.resume();
                return true;
            }
            var action = this.getNextAction();
            if (action == null) {
                var chosenMove = null;
                if (this.ai == null) {
                    this.battle.ui.hud.turnPreview.set(this.battle.predictTurns(this));
                    term.print("ask player for " + this.name + "'s next move");
                    chosenMove = this.attackMenu.open();
                }
                else {
                    chosenMove = this.ai.getNextMove();
                }
                if (chosenMove.stance != Stance.Guard) {
                    this.queueMove(chosenMove);
                    action = this.getNextAction();
                }
                else {
                    this.setGuard();
                }
            }
            if (this.isAlive()) {
                if (action !== null) {
                    this.performAction(action, this.moveUsed);
                }
                this.raiseEvent('endTurn');
            }
            var eventData = { actingUnit: this };
            this.battle.raiseEvent('endTurn', eventData);
            this.actor.animate('dormant');
            term.print("end of " + this.name + "'s turn");
            this.battle.resume();
            return true;
        }
        else {
            return false;
        }
    };
    BattleUnit.prototype.timeUntilNextTurn = function () {
        return this.cv;
    };
    BattleUnit.prototype.timeUntilTurn = function (turnIndex, assumedRank, nextActions) {
        if (assumedRank === void 0) { assumedRank = Game.defaultMoveRank; }
        if (nextActions === void 0) { nextActions = null; }
        if (this.isAlive()) {
            nextActions = nextActions !== null
                ? this.actionQueue.concat(nextActions)
                : this.actionQueue;
            var timeLeft = this.cv;
            for (var i = 1; i <= turnIndex; ++i) {
                var rank = assumedRank;
                if (i <= nextActions.length) {
                    rank = isNaN(nextActions[i - 1]) ? nextActions[i - 1].rank
                        : nextActions[i - 1];
                }
                timeLeft += Math.max(Math.round(Game.math.timeUntilNextTurn(this.battlerInfo, rank) / this.turnRatio), 1);
            }
            return timeLeft;
        }
        else {
            return Infinity;
        }
    };
    BattleUnit.prototype.turnIntoAHippo = function () {
        this.actor.animate('hippo');
        this.stance = Stance.Hippo;
    };
    return BattleUnit;
}());
//# sourceMappingURL=data:application/json;base64,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