/***
 * Specs Engine v6: Spectacles Saga Game Engine
  *           Copyright (c) 2013 Power-Command
***/
var HPGauge = require('@/scripts/battleEngine/ui').HPGauge;
var MPGauge = require('@/scripts/battleEngine/ui').MPGauge;
var TurnPreview = require('@/scripts/battleEngine/ui').TurnPreview;
// BattleHUD() constructor
// Creates an object representing the in-battle heads-up display (HUD).
// Arguments:
//     partyMaxMP: The party's current MP capacity.
function BattleHUD(partyMaxMP) {
    this.enemyHPGaugeColor = Color.White;
    this.partyHPGaugeColor = Color.Lime;
    this.partyHighlightColor = CreateColor(25, 25, 112, 255);
    this.partyMPGaugeColor = Color.DarkOrchid;
    this.fadeness = 0.0;
    this.font = GetSystemFont();
    this.highlightColor = CreateColor(0, 0, 0, 0);
    this.highlightedUnit = null;
    this.hpGaugesInfo = [];
    this.mpGauge = new MPGauge(partyMaxMP, this.partyMPGaugeColor);
    this.partyInfo = [null, null, null];
    this.thread = null;
    this.turnPreview = new TurnPreview();
    this.drawElementBox = function (x, y, width, height) {
        Rectangle(x, y, width, height, CreateColor(0, 0, 0, 192));
        OutlinedRectangle(x, y, width, height, CreateColor(0, 0, 0, 32));
    };
    this.drawHighlight = function (x, y, width, height, color) {
        var outerColor = color;
        var innerColor = BlendColors(outerColor, CreateColor(0, 0, 0, color.alpha));
        var halfHeight = Math.round(height / 2);
        GradientRectangle(x, y, width, halfHeight, outerColor, outerColor, innerColor, innerColor);
        GradientRectangle(x, y + halfHeight, width, height - halfHeight, innerColor, innerColor, outerColor, outerColor);
        OutlinedRectangle(x, y, width, height, CreateColor(0, 0, 0, color.alpha / 2));
    };
    this.drawPartyElement = function (x, y, memberInfo, isHighlighted) {
        this.drawElementBox(x, y, 100, 20, CreateColor(0, 32, 0, 192));
        if (isHighlighted) {
            this.drawHighlight(x, y, 100, 20, this.highlightColor);
        }
        this.drawHighlight(x, y, 100, 20, memberInfo.lightColor);
        var headingColor = isHighlighted ?
            BlendColorsWeighted(CreateColor(255, 192, 0, 255), CreateColor(192, 144, 0, 255), this.highlightColor.alpha, 255 - this.highlightColor.alpha) :
            CreateColor(192, 144, 0, 255);
        var textColor = isHighlighted ?
            BlendColorsWeighted(CreateColor(255, 255, 255, 255), CreateColor(192, 192, 192, 255), this.highlightColor.alpha, 255 - this.highlightColor.alpha) :
            CreateColor(192, 192, 192, 255);
        memberInfo.hpGauge.draw(x + 5, y + 5, 24, 10);
        this.drawText(this.font, x + 34, y + 4, 1, textColor, memberInfo.unit.name);
        //this.drawText(this.font, x + 62, y + 6, 1, headingColor, "HP");
        //this.drawText(this.font, x + 61, y + 2, 1, textColor, Math.round(memberInfo.hp), 'right');
        Rectangle(x + 81, y + 3, 14, 14, CreateColor(64, 96, 128, 255));
        OutlinedRectangle(x + 81, y + 3, 14, 14, CreateColor(0, 0, 0, 255));
    };
    this.drawText = function (font, x, y, shadowDistance, color, text, alignment) {
        if (alignment === void 0) { alignment = 'left'; }
        var Align = {
            left: function (font, x, text) { return x; },
            center: function (font, x, text) { return x - font.getStringWidth(text) / 2; },
            right: function (font, x, text) { return x - font.getStringWidth(text); },
        };
        if (!(alignment in Align))
            throw new Error("invalid text alignment '" + alignment + "'.");
        x = Align[alignment](font, x, text);
        font.setColorMask(CreateColor(0, 0, 0, color.alpha));
        font.drawText(x + shadowDistance, y + shadowDistance, text);
        font.setColorMask(color);
        font.drawText(x, y, text);
    };
}
// .dispose() method
// Frees all outstanding resources held by this object.
BattleHUD.prototype.dispose = function () {
    this.turnPreview.dispose();
    threads.kill(this.thread);
};
// .createEnemyHPGauge() method
// Creates an enemy HP gauge to be displayed on the HUD.
// Arguments:
//     unit:     The battle unit that the gauge belongs to.
BattleHUD.prototype.createEnemyHPGauge = function (unit) {
    var gauge = new HPGauge(unit.maxHP, Game.bossHPPerBar, this.enemyHPGaugeColor, 20);
    this.hpGaugesInfo.push({ owner: unit, gauge: gauge });
    gauge.show(0.0);
    term.print("create HP gauge for unit '" + unit.name + "'", "cap: " + unit.maxHP);
};
// .hide() method
// Hides the HUD.
BattleHUD.prototype.hide = function () {
    new Scene()
        .tween(this, 15, 'easeInExpo', { fadeness: 0.0 })
        .run();
};
// .highlight() method
// Highlights a character on the HUD.
// Arguments:
//     unit: The unit whose entry will be highlighted.
BattleHUD.prototype.highlight = function (unit) {
    if (unit !== null) {
        this.highlightedUnit = unit;
        new Scene()
            .tween(this.highlightColor, 6, 'easeInQuad', BlendColors(this.partyHighlightColor, CreateColor(255, 255, 255, this.partyHighlightColor.alpha)))
            .tween(this.highlightColor, 15, 'easeOutQuad', this.partyHighlightColor)
            .run();
    }
    else {
        new Scene()
            .tween(this.highlightColor, 6, 'easeInQuad', CreateColor(0, 0, 0, 0))
            .run();
    }
};
// .render() method
// Renders the HUD.
BattleHUD.prototype.render = function () {
    var y = -((this.partyInfo.length + this.hpGaugesInfo.length) * 20) * (1.0 - this.fadeness);
    var itemY = y;
    this.drawElementBox(260, itemY, 60, 60);
    this.mpGauge.draw(261, itemY + 1, 58);
    for (var i = 0; i < this.partyInfo.length; ++i) {
        var itemX = 160;
        var itemY = y + i * 20;
        if (this.partyInfo[i] !== null) {
            this.drawPartyElement(itemX, itemY, this.partyInfo[i], this.highlightedUnit == this.partyInfo[i].unit);
        }
        else {
            this.drawElementBox(itemX, itemY, 100, 20);
        }
    }
    for (var i = 0; i < this.hpGaugesInfo.length; ++i) {
        var gaugeInfo = this.hpGaugesInfo[i];
        var itemX = 160;
        var itemY = y + this.partyInfo.length * 20 + i * 20;
        this.drawElementBox(itemX, itemY, 160, 20);
        if (this.highlightedUnit == gaugeInfo.owner) {
            this.drawHighlight(itemX, itemY, 160, 20, this.highlightColor);
        }
        Rectangle(itemX + 141, itemY + 3, 14, 14, CreateColor(128, 32, 32, 255));
        OutlinedRectangle(itemX + 141, itemY + 3, 14, 14, CreateColor(0, 0, 0, 255));
        gaugeInfo.gauge.draw(itemX + 5, itemY + 5, 131, 10);
    }
};
// .setHP() method
// Changes the displayed HP for a character on the HUD.
// Arguments:
//     unit: The battle unit whose HP is being changed.
//     hp:   The number of hit points to change the display to.
BattleHUD.prototype.setHP = function (unit, hp) {
    for (var i = 0; i < this.partyInfo.length; ++i) {
        var characterInfo = this.partyInfo[i];
        if (characterInfo !== null && characterInfo.unit == unit && hp != characterInfo.hp) {
            characterInfo.hpGauge.set(hp);
            var gaugeColor = hp / characterInfo.maxHP <= 0.1 ? Color.Red
                : hp / characterInfo.maxHP <= 0.33 ? Color.Yellow
                    : Color.Lime;
            characterInfo.hpGauge.changeColor(gaugeColor, 0.5);
            var flashColor = hp > characterInfo.hp ? CreateColor(0, 192, 0, 255) : CreateColor(192, 0, 0, 255);
            new Scene()
                .fork()
                .tween(characterInfo.lightColor, 15, 'easeOutQuad', flashColor)
                .tween(characterInfo.lightColor, 15, 'easeOutQuad', CreateColor(0, 0, 0, 0))
                .end()
                .tween(characterInfo, 15, 'easeInOutSine', { hp: hp })
                .run();
        }
    }
    for (var i = 0; i < this.hpGaugesInfo.length; ++i) {
        var gaugeInfo = this.hpGaugesInfo[i];
        if (gaugeInfo.owner == unit) {
            gaugeInfo.gauge.set(hp);
        }
    }
};
// .setPartyMember() method
// Changes the character displayed in one of the party slots.
// Arguments:
//     slot:  The slot index (0-2 inclusive) of the party slot to be changed.
//     unit:  The battle unit being switched in.
//     hp:    The amount of remaining hit points to displayed for the character.
//     maxHP: The character's maximum HP.
BattleHUD.prototype.setPartyMember = function (slot, unit, hp, maxHP) {
    if (slot < 0 || slot >= this.partyInfo.length) {
        Abort("BattleHUD.switchOut(): Invalid party slot index '" + slot + "'!");
    }
    var hpGauge = new HPGauge(maxHP, Game.partyHPPerBar, this.partyHPGaugeColor, 10);
    hpGauge.show();
    this.partyInfo[slot] = {
        unit: unit,
        hp: hp,
        maxHP: maxHP,
        hpGauge: hpGauge,
        lightColor: CreateColor(255, 0, 0, 0)
    };
};
// .show() method
// Shows the HUD.
BattleHUD.prototype.show = function () {
    if (this.thread === null) {
        term.print("activate battle screen HUD");
        this.thread = threads.create(this, 20);
    }
    new Scene()
        .tween(this, 30, 'easeOutExpo', { fadeness: 1.0 })
        .run();
};
// .update() method
// Advances the BattleHUD's internal state by one frame.
BattleHUD.prototype.update = function () {
    for (var i = 0; i < this.partyInfo.length; ++i) {
        if (this.partyInfo[i] !== null) {
            this.partyInfo[i].hpGauge.update();
        }
    }
    for (var i = 0; i < this.hpGaugesInfo.length; ++i) {
        this.hpGaugesInfo[i].gauge.update();
    }
    return true;
};
//# sourceMappingURL=data:application/json;base64,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