/***
 * Specs Engine v6: Spectacles Saga Game Engine
  *           Copyright (c) 2012 Power-Command
***/
RequireScript('battleEngine/spriteImage.js');
var BattleActor = (function () {
    function BattleActor(name, position, row, isEnemy) {
        this.damages = [];
        this.fadeScene = null;
        this.hasEntered = false;
        this.healings = [];
        this.isEnemy = isEnemy;
        this.isVisible = true;
        this.messageFont = GetSystemFont();
        this.name = name;
        this.opacity = 1.0;
        this.position = isEnemy ? position : 2 - position;
        this.row = row;
        this.sprite = new SpriteImage("battlers/" + name + ".rss");
        this.sprite.direction = isEnemy ? 'east' : 'west';
        this.x = isEnemy ? -32 : 320;
        this.y = 168 - position * 32;
    }
    BattleActor.prototype.update = function () {
        this.sprite.update();
        for (var i = 0; i < this.damages.length; ++i) {
            var data = this.damages[i];
            var finalY = 20 - 11 * i;
            if (data.finalY != finalY) {
                data.scene.stop();
                data.finalY = finalY;
                var tweenInfo = {};
                for (var i2 = 0; i2 < data.text.length; ++i2) {
                    var yName = 'y' + i2.toString();
                    tweenInfo[yName] = finalY;
                }
                data.scene = new Scene()
                    .tween(data, 30, 'easeOutBounce', tweenInfo)
                    .pause(15);
                data.scene.run();
            }
            if (!data.scene.isRunning()) {
                this.damages.splice(i, 1);
                --i;
            }
        }
        for (var i = 0; i < this.healings.length; ++i) {
            if (!this.healings[i].scene.isRunning()) {
                this.healings.splice(i, 1);
                --i;
            }
        }
        return true;
    };
    BattleActor.prototype.render = function () {
        if (!this.isVisible && this.damages.length == 0 && this.healings.length == 0)
            return;
        this.sprite.blit(this.x, this.y, this.opacity * 255);
        for (var i = 0; i < this.damages.length; ++i) {
            var text = this.damages[i].text;
            var x = this.x + 16 - this.messageFont.getStringWidth(text) / 2;
            for (var i2 = 0; i2 < text.length; ++i2) {
                var yName = 'y' + i2.toString();
                var y = this.y + this.damages[i][yName];
                var color = this.damages[i].color !== null ? this.damages[i].color
                    : CreateColor(255, 255, 255, 255);
                drawTextEx(this.messageFont, x, y, text[i2], color, 1);
                x += this.messageFont.getStringWidth(text[i2]);
            }
        }
        for (var i = 0; i < this.healings.length; ++i) {
            var y = this.y + this.healings[i].y;
            var color = this.healings[i].color !== null ? this.healings[i].color : CreateColor(64, 255, 128, 255);
            var textColor = BlendColors(color, color);
            textColor.alpha *= this.healings[i].alpha / 255;
            drawTextEx(this.messageFont, this.x + 16, y, this.healings[i].amount, textColor, 1, 'center');
        }
    };
    BattleActor.prototype.animate = function (animationID) {
        // TODO: implement me!
        switch (animationID) {
            case 'die':
                this.sprite.direction = 'north';
                new Scene()
                    .tween(this, 60, 'easeInOutSine', { opacity: 0.1 })
                    .run();
                break;
            case 'hippo':
                this.sprite = new SpriteImage('battlers/maggie_hippo.rss');
                this.sprite.direction = this.isEnemy ? 'east' : 'west';
                break;
            case 'revive':
                new Scene()
                    .tween(this, 60, 'easeInOutSine', { opacity: 1.0 })
                    .call(function () { this.sprite.direction = this.isEnemy ? 'east' : 'west'; }.bind(this))
                    .run();
                break;
            case 'sleep':
                new Scene()
                    .talk("maggie", 2.0, this.name + " fell asleep! Hey, does that mean I get to eat him now?")
                    .run(true);
                break;
        }
    };
    BattleActor.prototype.enter = function (isImmediate) {
        if (isImmediate === void 0) { isImmediate = false; }
        if (this.hasEntered)
            return;
        var newX = this.isEnemy ? 64 - this.row * 32 : 224 + this.row * 32;
        var threadID = null;
        if (!isImmediate) {
            var entrance = new Scene()
                .tween(this, 90, 'linear', { x: newX })
                .run(true);
        }
        else {
            this.x = newX;
        }
        this.sprite.stop();
        this.hasEntered = true;
        return threadID;
    };
    BattleActor.prototype.showDamage = function (amount, color) {
        if (color === void 0) { color = null; }
        var finalY = 20 - 11 * this.damages.length;
        var data = { text: amount.toString(), color: color, finalY: finalY };
        var tweenInfo = {};
        for (var i = 0; i < data.text.length; ++i) {
            var yName = 'y' + i.toString();
            data[yName] = finalY - (20 - i * 5);
            tweenInfo[yName] = finalY;
        }
        data.scene = new Scene()
            .tween(data, 30, 'easeOutBounce', tweenInfo)
            .pause(15);
        data.scene.run();
        this.damages.push(data);
    };
    BattleActor.prototype.showHealing = function (amount, color) {
        if (color === void 0) { color = null; }
        var data = { amount: amount, color: color, y: 20, alpha: 255 };
        data.scene = new Scene()
            .tween(data, 60, 'easeOutExpo', { y: -11 * this.healings.length })
            .tween(data, 30, 'easeInOutSine', { alpha: 0 });
        data.scene.run();
        this.healings.push(data);
    };
    return BattleActor;
}());
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYmF0dGxlQWN0b3IuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJiYXR0bGVBY3Rvci5qcyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQTs7O0lBR0k7QUFFSixhQUFhLENBQUMsNkJBQTZCLENBQUMsQ0FBQztBQUU3QztJQUVDLHFCQUFZLElBQUksRUFBRSxRQUFRLEVBQUUsR0FBRyxFQUFFLE9BQU87UUFFdkMsSUFBSSxDQUFDLE9BQU8sR0FBRyxFQUFFLENBQUM7UUFDbEIsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLENBQUM7UUFDdEIsSUFBSSxDQUFDLFVBQVUsR0FBRyxLQUFLLENBQUM7UUFDeEIsSUFBSSxDQUFDLFFBQVEsR0FBRyxFQUFFLENBQUM7UUFDbkIsSUFBSSxDQUFDLE9BQU8sR0FBRyxPQUFPLENBQUM7UUFDdkIsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLENBQUM7UUFDdEIsSUFBSSxDQUFDLFdBQVcsR0FBRyxhQUFhLEVBQUUsQ0FBQztRQUNuQyxJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksQ0FBQztRQUNqQixJQUFJLENBQUMsT0FBTyxHQUFHLEdBQUcsQ0FBQztRQUNuQixJQUFJLENBQUMsUUFBUSxHQUFHLE9BQU8sR0FBRyxRQUFRLEdBQUcsQ0FBQyxHQUFHLFFBQVEsQ0FBQztRQUNsRCxJQUFJLENBQUMsR0FBRyxHQUFHLEdBQUcsQ0FBQztRQUNmLElBQUksQ0FBQyxNQUFNLEdBQUcsSUFBSSxXQUFXLENBQUMsY0FBWSxJQUFJLFNBQU0sQ0FBQyxDQUFDO1FBQ3RELElBQUksQ0FBQyxNQUFNLENBQUMsU0FBUyxHQUFHLE9BQU8sR0FBRyxNQUFNLEdBQUcsTUFBTSxDQUFDO1FBQ2xELElBQUksQ0FBQyxDQUFDLEdBQUcsT0FBTyxHQUFHLENBQUMsRUFBRSxHQUFHLEdBQUcsQ0FBQztRQUM3QixJQUFJLENBQUMsQ0FBQyxHQUFHLEdBQUcsR0FBRyxRQUFRLEdBQUcsRUFBRSxDQUFDO0lBQzlCLENBQUM7SUFFRCw0QkFBTSxHQUFOO1FBRUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsQ0FBQztRQUNyQixHQUFHLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxFQUFFLEVBQUUsQ0FBQyxFQUFFLENBQUM7WUFDOUMsSUFBSSxJQUFJLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUMzQixJQUFJLE1BQU0sR0FBRyxFQUFFLEdBQUcsRUFBRSxHQUFHLENBQUMsQ0FBQztZQUN6QixFQUFFLENBQUMsQ0FBQyxJQUFJLENBQUMsTUFBTSxJQUFJLE1BQU0sQ0FBQyxDQUFDLENBQUM7Z0JBQzNCLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxFQUFFLENBQUM7Z0JBQ2xCLElBQUksQ0FBQyxNQUFNLEdBQUcsTUFBTSxDQUFDO2dCQUNyQixJQUFJLFNBQVMsR0FBRyxFQUFFLENBQUM7Z0JBQ25CLEdBQUcsQ0FBQyxDQUFDLElBQUksRUFBRSxHQUFHLENBQUMsRUFBRSxFQUFFLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsRUFBRSxFQUFFLEVBQUUsQ0FBQztvQkFDOUMsSUFBSSxLQUFLLEdBQUcsR0FBRyxHQUFHLEVBQUUsQ0FBQyxRQUFRLEVBQUUsQ0FBQztvQkFDaEMsU0FBUyxDQUFDLEtBQUssQ0FBQyxHQUFHLE1BQU0sQ0FBQztnQkFDM0IsQ0FBQztnQkFDRCxJQUFJLENBQUMsS0FBSyxHQUFHLElBQUksS0FBSyxFQUFFO3FCQUN0QixLQUFLLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRSxlQUFlLEVBQUUsU0FBUyxDQUFDO3FCQUMzQyxLQUFLLENBQUMsRUFBRSxDQUFDLENBQUM7Z0JBQ1osSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLEVBQUUsQ0FBQztZQUNsQixDQUFDO1lBQ0QsRUFBRSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsRUFBRSxDQUFDLENBQUMsQ0FBQztnQkFDN0IsSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO2dCQUMxQixFQUFFLENBQUMsQ0FBQztZQUNMLENBQUM7UUFDRixDQUFDO1FBQ0QsR0FBRyxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sRUFBRSxFQUFFLENBQUMsRUFBRSxDQUFDO1lBQy9DLEVBQUUsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsU0FBUyxFQUFFLENBQUMsQ0FBQyxDQUFDO2dCQUN6QyxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7Z0JBQzNCLEVBQUUsQ0FBQyxDQUFDO1lBQ0wsQ0FBQztRQUNGLENBQUM7UUFDRCxNQUFNLENBQUMsSUFBSSxDQUFDO0lBQ2IsQ0FBQztJQUVELDRCQUFNLEdBQU47UUFFQyxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxTQUFTLElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLElBQUksQ0FBQyxJQUFJLElBQUksQ0FBQyxRQUFRLENBQUMsTUFBTSxJQUFJLENBQUMsQ0FBQztZQUM1RSxNQUFNLENBQUM7UUFDUixJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQyxFQUFFLElBQUksQ0FBQyxDQUFDLEVBQUUsSUFBSSxDQUFDLE9BQU8sR0FBRyxHQUFHLENBQUMsQ0FBQztRQUNyRCxHQUFHLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxFQUFFLEVBQUUsQ0FBQyxFQUFFLENBQUM7WUFDOUMsSUFBSSxJQUFJLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUM7WUFDaEMsSUFBSSxDQUFDLEdBQUcsSUFBSSxDQUFDLENBQUMsR0FBRyxFQUFFLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1lBQ2hFLEdBQUcsQ0FBQyxDQUFDLElBQUksRUFBRSxHQUFHLENBQUMsRUFBRSxFQUFFLEdBQUcsSUFBSSxDQUFDLE1BQU0sRUFBRSxFQUFFLEVBQUUsRUFBRSxDQUFDO2dCQUN6QyxJQUFJLEtBQUssR0FBRyxHQUFHLEdBQUcsRUFBRSxDQUFDLFFBQVEsRUFBRSxDQUFDO2dCQUNoQyxJQUFJLENBQUMsR0FBRyxJQUFJLENBQUMsQ0FBQyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUM7Z0JBQ3hDLElBQUksS0FBSyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsS0FBSyxLQUFLLElBQUksR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLEtBQUs7c0JBQy9ELFdBQVcsQ0FBQyxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUcsRUFBRSxHQUFHLENBQUMsQ0FBQztnQkFDbkMsVUFBVSxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQyxFQUFFLENBQUMsRUFBRSxJQUFJLENBQUMsRUFBRSxDQUFDLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQyxDQUFDO2dCQUN2RCxDQUFDLElBQUksSUFBSSxDQUFDLFdBQVcsQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7WUFDaEQsQ0FBQztRQUNGLENBQUM7UUFDRCxHQUFHLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsTUFBTSxFQUFFLEVBQUUsQ0FBQyxFQUFFLENBQUM7WUFDL0MsSUFBSSxDQUFDLEdBQUcsSUFBSSxDQUFDLENBQUMsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUNwQyxJQUFJLEtBQUssR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDLEtBQUssS0FBSyxJQUFJLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLEdBQUcsV0FBVyxDQUFDLEVBQUUsRUFBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUcsQ0FBQyxDQUFDO1lBQ3RHLElBQUksU0FBUyxHQUFHLFdBQVcsQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7WUFDMUMsU0FBUyxDQUFDLEtBQUssSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDLEtBQUssR0FBRyxHQUFHLENBQUM7WUFDaEQsVUFBVSxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsSUFBSSxDQUFDLENBQUMsR0FBRyxFQUFFLEVBQUUsQ0FBQyxFQUFFLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsTUFBTSxFQUFFLFNBQVMsRUFBRSxDQUFDLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDL0YsQ0FBQztJQUNGLENBQUM7SUFFRCw2QkFBTyxHQUFQLFVBQVEsV0FBVztRQUVsQixzQkFBc0I7UUFDdEIsTUFBTSxDQUFDLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQztZQUNyQixLQUFLLEtBQUs7Z0JBQ1QsSUFBSSxDQUFDLE1BQU0sQ0FBQyxTQUFTLEdBQUcsT0FBTyxDQUFDO2dCQUNoQyxJQUFJLEtBQUssRUFBRTtxQkFDVCxLQUFLLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRSxlQUFlLEVBQUUsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUM7cUJBQ2xELEdBQUcsRUFBRSxDQUFDO2dCQUNSLEtBQUssQ0FBQztZQUNQLEtBQUssT0FBTztnQkFDWCxJQUFJLENBQUMsTUFBTSxHQUFHLElBQUksV0FBVyxDQUFDLDJCQUEyQixDQUFDLENBQUM7Z0JBQzNELElBQUksQ0FBQyxNQUFNLENBQUMsU0FBUyxHQUFHLElBQUksQ0FBQyxPQUFPLEdBQUcsTUFBTSxHQUFHLE1BQU0sQ0FBQztnQkFDdkQsS0FBSyxDQUFDO1lBQ1AsS0FBSyxRQUFRO2dCQUNaLElBQUksS0FBSyxFQUFFO3FCQUNULEtBQUssQ0FBQyxJQUFJLEVBQUUsRUFBRSxFQUFFLGVBQWUsRUFBRSxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQztxQkFDbEQsSUFBSSxDQUFDLGNBQWEsSUFBSSxDQUFDLE1BQU0sQ0FBQyxTQUFTLEdBQUcsSUFBSSxDQUFDLE9BQU8sR0FBRyxNQUFNLEdBQUcsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztxQkFDdkYsR0FBRyxFQUFFLENBQUM7Z0JBQ1IsS0FBSyxDQUFDO1lBQ1AsS0FBSyxPQUFPO2dCQUNYLElBQUksS0FBSyxFQUFFO3FCQUNULElBQUksQ0FBQyxRQUFRLEVBQUUsR0FBRyxFQUFFLElBQUksQ0FBQyxJQUFJLEdBQUcseURBQXlELENBQUM7cUJBQzFGLEdBQUcsQ0FBQyxJQUFJLENBQUMsQ0FBQztnQkFDWixLQUFLLENBQUM7UUFDUixDQUFDO0lBQ0YsQ0FBQztJQUVELDJCQUFLLEdBQUwsVUFBTSxXQUFtQjtRQUFuQiw0QkFBQSxFQUFBLG1CQUFtQjtRQUV4QixFQUFFLENBQUMsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDO1lBQ25CLE1BQU0sQ0FBQztRQUNSLElBQUksSUFBSSxHQUFHLElBQUksQ0FBQyxPQUFPLEdBQUcsRUFBRSxHQUFHLElBQUksQ0FBQyxHQUFHLEdBQUcsRUFBRSxHQUFHLEdBQUcsR0FBRyxJQUFJLENBQUMsR0FBRyxHQUFHLEVBQUUsQ0FBQztRQUNuRSxJQUFJLFFBQVEsR0FBRyxJQUFJLENBQUM7UUFDcEIsRUFBRSxDQUFDLENBQUMsQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDO1lBQ2xCLElBQUksUUFBUSxHQUFHLElBQUksS0FBSyxFQUFFO2lCQUN4QixLQUFLLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRSxRQUFRLEVBQUUsRUFBRSxDQUFDLEVBQUUsSUFBSSxFQUFFLENBQUM7aUJBQ3RDLEdBQUcsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNiLENBQUM7UUFBQyxJQUFJLENBQUMsQ0FBQztZQUNQLElBQUksQ0FBQyxDQUFDLEdBQUcsSUFBSSxDQUFDO1FBQ2YsQ0FBQztRQUNELElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDbkIsSUFBSSxDQUFDLFVBQVUsR0FBRyxJQUFJLENBQUM7UUFDdkIsTUFBTSxDQUFDLFFBQVEsQ0FBQztJQUNqQixDQUFDO0lBRUQsZ0NBQVUsR0FBVixVQUFXLE1BQU0sRUFBRSxLQUFZO1FBQVosc0JBQUEsRUFBQSxZQUFZO1FBRTlCLElBQUksTUFBTSxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUM7UUFDM0MsSUFBSSxJQUFJLEdBQUcsRUFBRSxJQUFJLEVBQUUsTUFBTSxDQUFDLFFBQVEsRUFBRSxFQUFFLEtBQUssRUFBRSxLQUFLLEVBQUUsTUFBTSxFQUFFLE1BQU0sRUFBRSxDQUFDO1FBQ3JFLElBQUksU0FBUyxHQUFHLEVBQUUsQ0FBQztRQUNuQixHQUFHLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLEVBQUUsQ0FBQyxFQUFFLENBQUM7WUFDM0MsSUFBSSxLQUFLLEdBQUcsR0FBRyxHQUFHLENBQUMsQ0FBQyxRQUFRLEVBQUUsQ0FBQztZQUMvQixJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsTUFBTSxHQUFHLENBQUMsRUFBRSxHQUFHLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztZQUNwQyxTQUFTLENBQUMsS0FBSyxDQUFDLEdBQUcsTUFBTSxDQUFDO1FBQzNCLENBQUM7UUFDRCxJQUFJLENBQUMsS0FBSyxHQUFHLElBQUksS0FBSyxFQUFFO2FBQ3RCLEtBQUssQ0FBQyxJQUFJLEVBQUUsRUFBRSxFQUFFLGVBQWUsRUFBRSxTQUFTLENBQUM7YUFDM0MsS0FBSyxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBQ1osSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUNqQixJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUN6QixDQUFDO0lBRUQsaUNBQVcsR0FBWCxVQUFZLE1BQU0sRUFBRSxLQUFZO1FBQVosc0JBQUEsRUFBQSxZQUFZO1FBRS9CLElBQUksSUFBSSxHQUFHLEVBQUUsTUFBTSxFQUFFLE1BQU0sRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFFLENBQUMsRUFBRSxFQUFFLEVBQUUsS0FBSyxFQUFFLEdBQUcsRUFBRSxDQUFDO1FBQy9ELElBQUksQ0FBQyxLQUFLLEdBQUcsSUFBSSxLQUFLLEVBQUU7YUFDdEIsS0FBSyxDQUFDLElBQUksRUFBRSxFQUFFLEVBQUUsYUFBYSxFQUFFLEVBQUUsQ0FBQyxFQUFFLENBQUMsRUFBRSxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsTUFBTSxFQUFFLENBQUM7YUFDakUsS0FBSyxDQUFDLElBQUksRUFBRSxFQUFFLEVBQUUsZUFBZSxFQUFFLEVBQUUsS0FBSyxFQUFFLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDakQsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUNqQixJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUMxQixDQUFDO0lBQ0Ysa0JBQUM7QUFBRCxDQUFDLEFBeEpELElBd0pDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqKlxuICogU3BlY3MgRW5naW5lIHY2OiBTcGVjdGFjbGVzIFNhZ2EgR2FtZSBFbmdpbmVcbiAgKiAgICAgICAgICAgQ29weXJpZ2h0IChjKSAyMDEyIFBvd2VyLUNvbW1hbmRcbioqKi9cblxuUmVxdWlyZVNjcmlwdCgnYmF0dGxlRW5naW5lL3Nwcml0ZUltYWdlLmpzJyk7XG5cbmNsYXNzIEJhdHRsZUFjdG9yXG57XG5cdGNvbnN0cnVjdG9yKG5hbWUsIHBvc2l0aW9uLCByb3csIGlzRW5lbXkpXG5cdHtcblx0XHR0aGlzLmRhbWFnZXMgPSBbXTtcblx0XHR0aGlzLmZhZGVTY2VuZSA9IG51bGw7XG5cdFx0dGhpcy5oYXNFbnRlcmVkID0gZmFsc2U7XG5cdFx0dGhpcy5oZWFsaW5ncyA9IFtdO1xuXHRcdHRoaXMuaXNFbmVteSA9IGlzRW5lbXk7XG5cdFx0dGhpcy5pc1Zpc2libGUgPSB0cnVlO1xuXHRcdHRoaXMubWVzc2FnZUZvbnQgPSBHZXRTeXN0ZW1Gb250KCk7XG5cdFx0dGhpcy5uYW1lID0gbmFtZTtcblx0XHR0aGlzLm9wYWNpdHkgPSAxLjA7XG5cdFx0dGhpcy5wb3NpdGlvbiA9IGlzRW5lbXkgPyBwb3NpdGlvbiA6IDIgLSBwb3NpdGlvbjtcblx0XHR0aGlzLnJvdyA9IHJvdztcblx0XHR0aGlzLnNwcml0ZSA9IG5ldyBTcHJpdGVJbWFnZShgYmF0dGxlcnMvJHtuYW1lfS5yc3NgKTtcblx0XHR0aGlzLnNwcml0ZS5kaXJlY3Rpb24gPSBpc0VuZW15ID8gJ2Vhc3QnIDogJ3dlc3QnO1xuXHRcdHRoaXMueCA9IGlzRW5lbXkgPyAtMzIgOiAzMjA7XG5cdFx0dGhpcy55ID0gMTY4IC0gcG9zaXRpb24gKiAzMjtcblx0fVxuXG5cdHVwZGF0ZSgpXG5cdHtcblx0XHR0aGlzLnNwcml0ZS51cGRhdGUoKTtcblx0XHRmb3IgKGxldCBpID0gMDsgaSA8IHRoaXMuZGFtYWdlcy5sZW5ndGg7ICsraSkge1xuXHRcdFx0dmFyIGRhdGEgPSB0aGlzLmRhbWFnZXNbaV07XG5cdFx0XHR2YXIgZmluYWxZID0gMjAgLSAxMSAqIGk7XG5cdFx0XHRpZiAoZGF0YS5maW5hbFkgIT0gZmluYWxZKSB7XG5cdFx0XHRcdGRhdGEuc2NlbmUuc3RvcCgpO1xuXHRcdFx0XHRkYXRhLmZpbmFsWSA9IGZpbmFsWTtcblx0XHRcdFx0dmFyIHR3ZWVuSW5mbyA9IHt9O1xuXHRcdFx0XHRmb3IgKGxldCBpMiA9IDA7IGkyIDwgZGF0YS50ZXh0Lmxlbmd0aDsgKytpMikge1xuXHRcdFx0XHRcdHZhciB5TmFtZSA9ICd5JyArIGkyLnRvU3RyaW5nKCk7XG5cdFx0XHRcdFx0dHdlZW5JbmZvW3lOYW1lXSA9IGZpbmFsWTtcblx0XHRcdFx0fVxuXHRcdFx0XHRkYXRhLnNjZW5lID0gbmV3IFNjZW5lKClcblx0XHRcdFx0XHQudHdlZW4oZGF0YSwgMzAsICdlYXNlT3V0Qm91bmNlJywgdHdlZW5JbmZvKVxuXHRcdFx0XHRcdC5wYXVzZSgxNSk7XG5cdFx0XHRcdGRhdGEuc2NlbmUucnVuKCk7XG5cdFx0XHR9XG5cdFx0XHRpZiAoIWRhdGEuc2NlbmUuaXNSdW5uaW5nKCkpIHtcblx0XHRcdFx0dGhpcy5kYW1hZ2VzLnNwbGljZShpLCAxKTtcblx0XHRcdFx0LS1pO1xuXHRcdFx0fVxuXHRcdH1cblx0XHRmb3IgKGxldCBpID0gMDsgaSA8IHRoaXMuaGVhbGluZ3MubGVuZ3RoOyArK2kpIHtcblx0XHRcdGlmICghdGhpcy5oZWFsaW5nc1tpXS5zY2VuZS5pc1J1bm5pbmcoKSkge1xuXHRcdFx0XHR0aGlzLmhlYWxpbmdzLnNwbGljZShpLCAxKTtcblx0XHRcdFx0LS1pO1xuXHRcdFx0fVxuXHRcdH1cblx0XHRyZXR1cm4gdHJ1ZTtcblx0fVxuXG5cdHJlbmRlcigpXG5cdHtcblx0XHRpZiAoIXRoaXMuaXNWaXNpYmxlICYmIHRoaXMuZGFtYWdlcy5sZW5ndGggPT0gMCAmJiB0aGlzLmhlYWxpbmdzLmxlbmd0aCA9PSAwKVxuXHRcdFx0cmV0dXJuO1xuXHRcdHRoaXMuc3ByaXRlLmJsaXQodGhpcy54LCB0aGlzLnksIHRoaXMub3BhY2l0eSAqIDI1NSk7XG5cdFx0Zm9yIChsZXQgaSA9IDA7IGkgPCB0aGlzLmRhbWFnZXMubGVuZ3RoOyArK2kpIHtcblx0XHRcdHZhciB0ZXh0ID0gdGhpcy5kYW1hZ2VzW2ldLnRleHQ7XG5cdFx0XHR2YXIgeCA9IHRoaXMueCArIDE2IC0gdGhpcy5tZXNzYWdlRm9udC5nZXRTdHJpbmdXaWR0aCh0ZXh0KSAvIDI7XG5cdFx0XHRmb3IgKGxldCBpMiA9IDA7IGkyIDwgdGV4dC5sZW5ndGg7ICsraTIpIHtcblx0XHRcdFx0dmFyIHlOYW1lID0gJ3knICsgaTIudG9TdHJpbmcoKTtcblx0XHRcdFx0dmFyIHkgPSB0aGlzLnkgKyB0aGlzLmRhbWFnZXNbaV1beU5hbWVdO1xuXHRcdFx0XHR2YXIgY29sb3IgPSB0aGlzLmRhbWFnZXNbaV0uY29sb3IgIT09IG51bGwgPyB0aGlzLmRhbWFnZXNbaV0uY29sb3Jcblx0XHRcdFx0XHQ6IENyZWF0ZUNvbG9yKDI1NSwgMjU1LCAyNTUsIDI1NSk7XG5cdFx0XHRcdGRyYXdUZXh0RXgodGhpcy5tZXNzYWdlRm9udCwgeCwgeSwgdGV4dFtpMl0sIGNvbG9yLCAxKTtcblx0XHRcdFx0eCArPSB0aGlzLm1lc3NhZ2VGb250LmdldFN0cmluZ1dpZHRoKHRleHRbaTJdKTtcblx0XHRcdH1cblx0XHR9XG5cdFx0Zm9yIChsZXQgaSA9IDA7IGkgPCB0aGlzLmhlYWxpbmdzLmxlbmd0aDsgKytpKSB7XG5cdFx0XHR2YXIgeSA9IHRoaXMueSArIHRoaXMuaGVhbGluZ3NbaV0ueTtcblx0XHRcdHZhciBjb2xvciA9IHRoaXMuaGVhbGluZ3NbaV0uY29sb3IgIT09IG51bGwgPyB0aGlzLmhlYWxpbmdzW2ldLmNvbG9yIDogQ3JlYXRlQ29sb3IoNjQsIDI1NSwgMTI4LCAyNTUpO1xuXHRcdFx0dmFyIHRleHRDb2xvciA9IEJsZW5kQ29sb3JzKGNvbG9yLCBjb2xvcik7XG5cdFx0XHR0ZXh0Q29sb3IuYWxwaGEgKj0gdGhpcy5oZWFsaW5nc1tpXS5hbHBoYSAvIDI1NTtcblx0XHRcdGRyYXdUZXh0RXgodGhpcy5tZXNzYWdlRm9udCwgdGhpcy54ICsgMTYsIHksIHRoaXMuaGVhbGluZ3NbaV0uYW1vdW50LCB0ZXh0Q29sb3IsIDEsICdjZW50ZXInKTtcblx0XHR9XG5cdH1cblxuXHRhbmltYXRlKGFuaW1hdGlvbklEKVxuXHR7XG5cdFx0Ly8gVE9ETzogaW1wbGVtZW50IG1lIVxuXHRcdHN3aXRjaCAoYW5pbWF0aW9uSUQpIHtcblx0XHRcdGNhc2UgJ2RpZSc6XG5cdFx0XHRcdHRoaXMuc3ByaXRlLmRpcmVjdGlvbiA9ICdub3J0aCc7XG5cdFx0XHRcdG5ldyBTY2VuZSgpXG5cdFx0XHRcdFx0LnR3ZWVuKHRoaXMsIDYwLCAnZWFzZUluT3V0U2luZScsIHsgb3BhY2l0eTogMC4xIH0pXG5cdFx0XHRcdFx0LnJ1bigpO1xuXHRcdFx0XHRicmVhaztcblx0XHRcdGNhc2UgJ2hpcHBvJzpcblx0XHRcdFx0dGhpcy5zcHJpdGUgPSBuZXcgU3ByaXRlSW1hZ2UoJ2JhdHRsZXJzL21hZ2dpZV9oaXBwby5yc3MnKTtcblx0XHRcdFx0dGhpcy5zcHJpdGUuZGlyZWN0aW9uID0gdGhpcy5pc0VuZW15ID8gJ2Vhc3QnIDogJ3dlc3QnO1xuXHRcdFx0XHRicmVhaztcblx0XHRcdGNhc2UgJ3Jldml2ZSc6XG5cdFx0XHRcdG5ldyBTY2VuZSgpXG5cdFx0XHRcdFx0LnR3ZWVuKHRoaXMsIDYwLCAnZWFzZUluT3V0U2luZScsIHsgb3BhY2l0eTogMS4wIH0pXG5cdFx0XHRcdFx0LmNhbGwoZnVuY3Rpb24oKSB7IHRoaXMuc3ByaXRlLmRpcmVjdGlvbiA9IHRoaXMuaXNFbmVteSA/ICdlYXN0JyA6ICd3ZXN0JzsgfS5iaW5kKHRoaXMpKVxuXHRcdFx0XHRcdC5ydW4oKTtcblx0XHRcdFx0YnJlYWs7XG5cdFx0XHRjYXNlICdzbGVlcCc6XG5cdFx0XHRcdG5ldyBTY2VuZSgpXG5cdFx0XHRcdFx0LnRhbGsoXCJtYWdnaWVcIiwgMi4wLCB0aGlzLm5hbWUgKyBcIiBmZWxsIGFzbGVlcCEgSGV5LCBkb2VzIHRoYXQgbWVhbiBJIGdldCB0byBlYXQgaGltIG5vdz9cIilcblx0XHRcdFx0XHQucnVuKHRydWUpO1xuXHRcdFx0XHRicmVhaztcblx0XHR9XG5cdH1cblxuXHRlbnRlcihpc0ltbWVkaWF0ZSA9IGZhbHNlKVxuXHR7XG5cdFx0aWYgKHRoaXMuaGFzRW50ZXJlZClcblx0XHRcdHJldHVybjtcblx0XHR2YXIgbmV3WCA9IHRoaXMuaXNFbmVteSA/IDY0IC0gdGhpcy5yb3cgKiAzMiA6IDIyNCArIHRoaXMucm93ICogMzI7XG5cdFx0dmFyIHRocmVhZElEID0gbnVsbDtcblx0XHRpZiAoIWlzSW1tZWRpYXRlKSB7XG5cdFx0XHR2YXIgZW50cmFuY2UgPSBuZXcgU2NlbmUoKVxuXHRcdFx0XHQudHdlZW4odGhpcywgOTAsICdsaW5lYXInLCB7IHg6IG5ld1ggfSlcblx0XHRcdFx0LnJ1bih0cnVlKTtcblx0XHR9IGVsc2Uge1xuXHRcdFx0dGhpcy54ID0gbmV3WDtcblx0XHR9XG5cdFx0dGhpcy5zcHJpdGUuc3RvcCgpO1xuXHRcdHRoaXMuaGFzRW50ZXJlZCA9IHRydWU7XG5cdFx0cmV0dXJuIHRocmVhZElEO1xuXHR9XG5cblx0c2hvd0RhbWFnZShhbW91bnQsIGNvbG9yID0gbnVsbClcblx0e1xuXHRcdHZhciBmaW5hbFkgPSAyMCAtIDExICogdGhpcy5kYW1hZ2VzLmxlbmd0aDtcblx0XHR2YXIgZGF0YSA9IHsgdGV4dDogYW1vdW50LnRvU3RyaW5nKCksIGNvbG9yOiBjb2xvciwgZmluYWxZOiBmaW5hbFkgfTtcblx0XHR2YXIgdHdlZW5JbmZvID0ge307XG5cdFx0Zm9yIChsZXQgaSA9IDA7IGkgPCBkYXRhLnRleHQubGVuZ3RoOyArK2kpIHtcblx0XHRcdHZhciB5TmFtZSA9ICd5JyArIGkudG9TdHJpbmcoKTtcblx0XHRcdGRhdGFbeU5hbWVdID0gZmluYWxZIC0gKDIwIC0gaSAqIDUpO1xuXHRcdFx0dHdlZW5JbmZvW3lOYW1lXSA9IGZpbmFsWTtcblx0XHR9XG5cdFx0ZGF0YS5zY2VuZSA9IG5ldyBTY2VuZSgpXG5cdFx0XHQudHdlZW4oZGF0YSwgMzAsICdlYXNlT3V0Qm91bmNlJywgdHdlZW5JbmZvKVxuXHRcdFx0LnBhdXNlKDE1KTtcblx0XHRkYXRhLnNjZW5lLnJ1bigpO1xuXHRcdHRoaXMuZGFtYWdlcy5wdXNoKGRhdGEpO1xuXHR9XG5cblx0c2hvd0hlYWxpbmcoYW1vdW50LCBjb2xvciA9IG51bGwpXG5cdHtcblx0XHR2YXIgZGF0YSA9IHsgYW1vdW50OiBhbW91bnQsIGNvbG9yOiBjb2xvciwgeTogMjAsIGFscGhhOiAyNTUgfTtcblx0XHRkYXRhLnNjZW5lID0gbmV3IFNjZW5lKClcblx0XHRcdC50d2VlbihkYXRhLCA2MCwgJ2Vhc2VPdXRFeHBvJywgeyB5OiAtMTEgKiB0aGlzLmhlYWxpbmdzLmxlbmd0aCB9KVxuXHRcdFx0LnR3ZWVuKGRhdGEsIDMwLCAnZWFzZUluT3V0U2luZScsIHsgYWxwaGE6IDAgfSk7XG5cdFx0ZGF0YS5zY2VuZS5ydW4oKTtcblx0XHR0aGlzLmhlYWxpbmdzLnB1c2goZGF0YSk7XG5cdH1cbn1cbiJdfQ==