/***
 * Specs Engine v6: Spectacles Saga Game Engine
  *           Copyright (c) 2017 Power-Command
***/
var __values = (this && this.__values) || function (o) {
    var m = typeof Symbol === "function" && o[Symbol.iterator], i = 0;
    if (m) return m.call(o);
    return {
        next: function () {
            if (o && i >= o.length) o = void 0;
            return { value: o && o[i++], done: !o };
        }
    };
};
RequireScript('battleEngine/item.js');
RequireScript('battleEngine/skill.js');
RequireScript('battleEngine/weapon.js');
var BattleAI = (function () {
    function BattleAI(unit, battle) {
        term.print("initialize AI for " + unit.fullName);
        this.battle = battle;
        this.data = {};
        this.defaultSkillID = null;
        this.moveQueue = [];
        this.currentPhase = 0;
        this.phasePoints = null;
        this.targets = null;
        this.turnsTaken = 0;
        this.unit = unit;
    }
    Object.defineProperty(BattleAI.prototype, "defaultSkill", {
        get: function () {
            return this.defaultSkillID;
        },
        set: function (value) {
            term.print("default skill for " + this.unit.name + " is " + Game.skills[value].name);
            this.defaultSkillID = value;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(BattleAI.prototype, "phase", {
        get: function () {
            return this.currentPhase;
        },
        enumerable: true,
        configurable: true
    });
    BattleAI.prototype.definePhases = function (thresholds, sigma) {
        if (sigma === void 0) { sigma = 0; }
        term.print("set up " + (thresholds.length + 1) + " phases for " + this.unit.name);
        this.phasePoints = from(thresholds)
            .select(function (v) { return Math.round(random.normal(v, sigma)); })
            .toArray();
        var phase = 1;
        try {
            for (var _a = __values(this.phasePoints), _b = _a.next(); !_b.done; _b = _a.next()) {
                var milestone = _b.value;
                term.print("phase " + ++phase + " will start at <= " + milestone + " HP");
            }
        }
        catch (e_1_1) { e_1 = { error: e_1_1 }; }
        finally {
            try {
                if (_b && !_b.done && (_c = _a.return)) _c.call(_a);
            }
            finally { if (e_1) throw e_1.error; }
        }
        this.currentPhase = 0;
        this.lastPhase = 0;
        var e_1, _c;
    };
    BattleAI.prototype.getNextMove = function () {
        var moveToUse = null;
        do {
            if (this.moveQueue.length == 0) {
                term.print("defer to AI for " + this.unit.name + "'s next move");
                var enemyList = this.battle.enemiesOf(this.unit);
                this.enemies = [];
                for (var i = 0; i < enemyList.length; ++i) {
                    var enemy = enemyList[i];
                    this.enemies.push(enemy);
                    this.enemies[enemy.id] = enemy;
                }
                var allyList = this.battle.alliesOf(this.unit);
                this.allies = [];
                for (var i = 0; i < allyList.length; ++i) {
                    var ally = allyList[i];
                    this.allies.push(ally);
                    this.allies[ally.id] = ally;
                }
                this.targets = null;
                this.updatePhase();
                if (this.moveQueue.length == 0)
                    this.strategize();
                if (this.moveQueue.length == 0) {
                    term.print("no moves queued for " + this.unit.name + ", using default");
                    if (this.defaultSkillID !== null) {
                        this.queueSkill(this.defaultSkillID);
                    }
                    else {
                        throw new Error("no moves queued and no default skill");
                    }
                }
            }
            var candidateMove;
            var isUsable;
            do {
                candidateMove = this.moveQueue.shift();
                var isLegal = candidateMove.stance != Stance.Attack || candidateMove.usable.isUsable(this.unit, this.unit.stance);
                isUsable = isLegal && candidateMove.predicate();
                if (!isUsable)
                    term.print("discard " + this.unit.name + "'s " + candidateMove.usable.name + ", not usable");
            } while (!isUsable && this.moveQueue.length > 0);
            if (isUsable)
                moveToUse = candidateMove;
            else if (this.defaultSkillID !== null)
                this.queueSkill(this.defaultSkillID);
        } while (moveToUse === null);
        ++this.turnsTaken;
        return moveToUse;
    };
    BattleAI.prototype.hasMovesQueued = function () {
        return this.moveQueue.length > 0;
    };
    BattleAI.prototype.hasStatus = function (statusID) {
        return this.unit.hasStatus(statusID);
    };
    BattleAI.prototype.isItemQueued = function (itemID) {
        return from(this.moveQueue)
            .where(function (v) { return v.usable instanceof ItemUsable; })
            .any(function (v) { return v.usable.itemID == itemID; });
    };
    BattleAI.prototype.isItemUsable = function (itemID) {
        var _this = this;
        return from(this.unit.items)
            .where(function (v) { return v.itemID === itemID; })
            .any(function (v) { return v.isUsable(_this, _this.unit.stance); });
    };
    BattleAI.prototype.isSkillQueued = function (skillID) {
        return from(this.moveQueue)
            .where(function (v) { return v.usable instanceof SkillUsable; })
            .any(function (v) { return v.usable.skillID == skillID; });
    };
    BattleAI.prototype.isSkillUsable = function (skillID) {
        var skillToUse = new SkillUsable(skillID, 100);
        return skillToUse.isUsable(this.unit, this.unit.stance);
    };
    BattleAI.prototype.itemsLeft = function (itemID) {
        var _this = this;
        var item = from(this.unit.items)
            .where(function (v) { return v.itemID === itemID; })
            .besides(function (v) { return term.print(_this.unit.name + " counting remaining " + v.name, "left: " + v.usesLeft); })
            .first();
        return item.usesLeft;
    };
    BattleAI.prototype.predictItemTurns = function (itemID) {
        if (!(itemID in Game.items))
            throw new ReferenceError("no item definition for '" + itemID + "'");
        var itemRank = 'rank' in Game.items[itemID] ? Game.items[itemID].rank : Game.defaultItemRank;
        var forecast = this.battle.predictTurns(this.unit, [itemRank]);
        term.print(this.unit.name + " considering " + Game.items[itemID].name, "next: " + forecast[0].unit.name);
        return forecast;
    };
    BattleAI.prototype.predictSkillTurns = function (skillID) {
        if (!(skillID in Game.skills))
            throw new ReferenceError("no skill definition for '" + skillID + "'");
        var forecast = this.battle.predictTurns(this.unit, Game.skills[skillID].actions);
        term.print(this.unit.name + " considering " + Game.skills[skillID].name, "next: " + forecast[0].unit.name);
        return forecast;
    };
    BattleAI.prototype.queueGuard = function () {
        this.moveQueue.push({
            usable: null,
            stance: Stance.Guard,
            predicate: function () { return true; },
        });
    };
    BattleAI.prototype.queueItem = function (itemID, unitID) {
        var _this = this;
        if (unitID === void 0) { unitID = null; }
        var itemToUse = from(this.unit.items)
            .where(function (v) { return v.itemID === itemID; })
            .where(function (v) { return v.isUsable(_this.unit, _this.unit.stance); })
            .first();
        if (itemToUse === undefined)
            throw new Error(this.unit.name + " tried to use an item '" + itemID + "' not owned");
        var targets = this.targets !== null ? this.targets
            : unitID !== null ? [this.battle.findUnit(unitID)]
                : itemToUse.defaultTargets(this.unit);
        this.moveQueue.push({
            usable: itemToUse,
            stance: Stance.Attack,
            targets: targets,
            predicate: function () { return true; },
        });
        term.print(this.unit.name + " queued use of item " + itemToUse.name);
    };
    BattleAI.prototype.queueSkill = function (skillID, stance, unitID, predicate) {
        if (stance === void 0) { stance = Stance.Attack; }
        if (unitID === void 0) { unitID = null; }
        if (predicate === void 0) { predicate = function () { return true; }; }
        var skillToUse = new SkillUsable(skillID, 100);
        var targetUnit = unitID !== null ? this.battle.findUnit(unitID) : null;
        var targets = this.targets !== null ? this.targets
            : targetUnit !== null ? [targetUnit]
                : skillToUse.defaultTargets(this.unit);
        this.moveQueue.push({
            usable: skillToUse,
            stance: stance,
            targets: targets,
            predicate: predicate
        });
        term.print(this.unit.name + " queued use of skill " + skillToUse.name);
    };
    BattleAI.prototype.queueWeapon = function (weaponID) {
        var weaponUsable = new WeaponUsable(weaponID);
        this.moveQueue.push({
            usable: weaponUsable,
            stance: Stance.Attack,
            targets: weaponUsable.defaultTargets(this.unit),
            predicate: function () { return true; },
        });
        var weaponDef = Game.weapons[weaponID];
        term.print(this.unit.name + " queued weapon change to " + weaponDef.name);
    };
    BattleAI.prototype.setTarget = function (targetID) {
        var unit = this.battle.findUnit(targetID);
        this.targets = unit !== null ? [unit] : null;
    };
    BattleAI.prototype.strategize = function () {
        if (this.defaultSkillID !== null)
            this.queueSkill(this.defaultSkillID);
        else
            throw new Error("AI has no strategy");
    };
    BattleAI.prototype.updatePhase = function () {
        var _this = this;
        var phaseToEnter = 1;
        if (this.phasePoints !== null) {
            var milestone = from(this.phasePoints)
                .last(function (v) { return v >= _this.unit.hp; });
            phaseToEnter = 2 + this.phasePoints.indexOf(milestone);
        }
        var lastPhase = this.currentPhase;
        this.currentPhase = Math.max(phaseToEnter, this.currentPhase); // ratcheting
        if (this.currentPhase > lastPhase) {
            term.print(this.unit.name + " is entering phase " + this.currentPhase, "prev: " + (lastPhase > 0 ? lastPhase : "none"));
            this.on_phaseChanged(this.currentPhase, lastPhase);
        }
    };
    BattleAI.prototype.on_itemUsed = function (userID, itemID, targetIDs) { };
    BattleAI.prototype.on_skillUsed = function (userID, skillID, targetIDs) { };
    BattleAI.prototype.on_stanceChanged = function (userID, stance) { };
    BattleAI.prototype.on_phaseChanged = function (phase, lastPhase) { };
    BattleAI.prototype.on_unitDamaged = function (unitID, amount, tags, attacker) { };
    BattleAI.prototype.on_unitHealed = function (unitID, amount, tags) { };
    BattleAI.prototype.on_unitKilled = function (unitID) { };
    BattleAI.prototype.on_unitReady = function (unitID) { };
    BattleAI.prototype.on_unitTargeted = function (targetID, action, unitID) { };
    return BattleAI;
}());
//# sourceMappingURL=data:application/json;base64,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