/***
 * Specs Engine v6: Spectacles Saga Game Engine
  *           Copyright (c) 2017 Power-Command
***/
var __read = (this && this.__read) || function (o, n) {
    var m = typeof Symbol === "function" && o[Symbol.iterator];
    if (!m) return o;
    var i = m.call(o), r, ar = [], e;
    try {
        while ((n === void 0 || n-- > 0) && !(r = i.next()).done) ar.push(r.value);
    }
    catch (error) { e = { error: error }; }
    finally {
        try {
            if (r && !r.done && (m = i["return"])) m.call(i);
        }
        finally { if (e) throw e.error; }
    }
    return ar;
};
var __spread = (this && this.__spread) || function () {
    for (var ar = [], i = 0; i < arguments.length; i++) ar = ar.concat(__read(arguments[i]));
    return ar;
};
RequireScript('battleEngine/battleScreen.js');
RequireScript('battleEngine/battleUnit.js');
RequireScript('battleEngine/fieldCondition.js');
RequireScript('battleEngine/mpPool.js');
var BattleResult = {
    Win: 1,
    Flee: 2,
    Lose: 3,
};
var Battle = (function () {
    function Battle(session, battleID) {
        if (!(battleID in Game.battles))
            throw new ReferenceError("no encounter data for '" + battleID + "'");
        term.print("initialize battle context for '" + battleID + "'");
        this.aiList = [];
        this.battleID = battleID;
        this.mode = null;
        this.parameters = Game.battles[battleID];
        this.partyMPPool = null;
        this.session = session;
        this.suspendCount = 0;
        this.timer = 0;
        this.battleLevel = 'battleLevel' in this.parameters ? this.parameters.battleLevel : session.party.level;
    }
    Battle.prototype.update = function () {
        switch (this.mode) {
            case 'setup':
                var heading = ('isFinalBattle' in this.parameters && this.parameters.isFinalBattle)
                    ? "Final Battle: " : "Boss Battle: ";
                this.ui.go('title' in this.parameters ? heading + this.parameters.title : null);
                var walkInThreads_1 = [];
                from(this.enemyUnits, this.playerUnits)
                    .each(function (unit) {
                    var threadID = unit.actor.enter();
                    walkInThreads_1.push(threadID);
                });
                this.ui.hud.turnPreview.show();
                if (!from(this.session.battlesSeen).anyIs(this.battleID)) {
                    this.session.battlesSeen.push(this.battleID);
                    if ('onFirstStart' in this.parameters) {
                        term.print("call onFirstStart() for battle '" + this.battleID + "'");
                        this.parameters.onFirstStart.call(this);
                    }
                }
                if ('onStart' in this.parameters) {
                    term.print("call onStart() for battle '" + this.battleID + "'");
                    this.parameters.onStart.call(this);
                }
                this.ui.showTitle();
                this.mode = 'battle';
                break;
            case 'battle':
                this.tick();
                break;
        }
        if (this.result !== null) {
            term.print("shut down battle engine");
            from(this.battleUnits)
                .each(function (unit) { return unit.dispose(); });
            this.ui.dispose();
            music.pop();
            music.adjust(1.0, 0);
            term.undefine('battle');
            return false;
        }
        else {
            return true;
        }
    };
    Battle.prototype.addCondition = function (conditionID) {
        if (this.hasCondition(conditionID))
            return; // nop if already installed
        var eventData = { conditionID: conditionID, cancel: false };
        this.raiseEvent('conditionInstalled', eventData);
        if (!eventData.cancel) {
            var effect = new FieldCondition(eventData.conditionID, this);
            this.conditions.push(effect);
            term.print("install field condition " + effect.name);
        }
        else {
            term.print("cancel FC '" + conditionID + "' per existing FC");
        }
    };
    Battle.prototype.alliesOf = function (unit) {
        if (unit.isPartyMember())
            return this.playerUnits;
        else
            return this.enemyUnits;
    };
    Battle.prototype.areEnemies = function (unit1, unit2) {
        return from(this.enemiesOf(unit1)).anyIs(unit2);
    };
    Battle.prototype.enemiesOf = function (unit) {
        if (unit.isPartyMember()) {
            return this.enemyUnits;
        }
        else {
            return this.playerUnits;
        }
    };
    Battle.prototype.findUnit = function (unitID) {
        var unit = from(this.enemyUnits, this.playerUnits)
            .first(function (v) { return v.id == unitID; });
        return unit !== undefined ? unit : null;
    };
    Battle.prototype.getLevel = function () {
        return this.battleLevel;
    };
    Battle.prototype.go = function () {
        if (Sphere.Game.disableBattles) {
            term.print("battles disabled, automatic win", "battleID: " + this.battleID);
            this.result = BattleResult.Win;
            return null;
        }
        term.print("");
        term.print("start battle engine", "battleID: " + this.battleID);
        var partyMaxMP = 0;
        for (id in this.session.party.members) {
            var battlerInfo = this.session.party.members[id].getInfo();
            var mpDonated = Math.round(Game.math.mp.capacity(battlerInfo));
            partyMaxMP += mpDonated;
            term.print(Game.characters[battlerInfo.characterID].name + " donated " + mpDonated + " MP to shared pool");
        }
        partyMaxMP = Math.min(Math.max(partyMaxMP, 0), 9999);
        var partyMPPool = new MPPool('partyMP', Math.min(Math.max(partyMaxMP, 0), 9999));
        partyMPPool.gainedMP.add(function (mpPool, availableMP) {
            this.ui.hud.mpGauge.set(availableMP);
        }, this);
        partyMPPool.lostMP.add(function (mpPool, availableMP) {
            this.ui.hud.mpGauge.set(availableMP);
        }, this);
        this.ui = new BattleScreen(partyMaxMP);
        this.battleUnits = [];
        this.playerUnits = [];
        this.enemyUnits = [];
        this.conditions = [];
        for (var i_1 = 0; i_1 < this.parameters.enemies.length; ++i_1) {
            var enemyID = this.parameters.enemies[i_1];
            var unit = new BattleUnit(this, enemyID, i_1 == 0 ? 1 : i_1 == 1 ? 0 : i_1, Row.Middle);
            this.battleUnits.push(unit);
            this.enemyUnits.push(unit);
        }
        var i = 0;
        for (var name in this.session.party.members) {
            var unit = new BattleUnit(this, this.session.party.members[name], i == 0 ? 1 : i == 1 ? 0 : i, Row.Middle, partyMPPool);
            this.battleUnits.push(unit);
            this.playerUnits.push(unit);
            ++i;
        }
        var battleBGMTrack = Game.defaultBattleBGM;
        if ('bgm' in this.parameters) {
            battleBGMTrack = this.parameters.bgm;
        }
        this.ui.hud.turnPreview.set(this.predictTurns());
        music.push(battleBGMTrack !== null
            ? "music/" + battleBGMTrack + ".ogg"
            : null);
        this.result = null;
        this.timer = 0;
        this.mode = 'setup';
        term.define('battle', this, {
            'spawn': this.spawnEnemy
        });
        var battleThread = threads.create(this);
        return battleThread;
    };
    Battle.prototype.hasCondition = function (conditionID) {
        return from(this.conditions)
            .select(function (it) { return it.conditionID; })
            .anyIs(conditionID);
    };
    Battle.prototype.isActive = function () {
        return this.result === null;
    };
    Battle.prototype.liftCondition = function (conditionID) {
        from(this.conditions)
            .where(function (v) { return v.conditionID === conditionID; })
            .besides(function (v) { return term.print("lift field condition " + v.name); })
            .remove();
    };
    Battle.prototype.notifyAIs = function (eventName) {
        var args = [];
        for (var _i = 1; _i < arguments.length; _i++) {
            args[_i - 1] = arguments[_i];
        }
        from(this.aiList)
            .besides(function (v) { return term.print("notify AI battler " + v.unit.name + " '" + eventName + "'"); })
            .each(function (v) { return v["on_" + eventName].apply(v, __spread(args)); });
    };
    Battle.prototype.predictTurns = function (actingUnit, nextActions) {
        if (actingUnit === void 0) { actingUnit = null; }
        if (nextActions === void 0) { nextActions = null; }
        var forecast = [];
        var _loop_1 = function (turnIndex) {
            var bias = 0;
            from(this_1.enemyUnits, this_1.playerUnits)
                .where(function (v) { return v !== actingUnit || turnIndex > 0; })
                .each(function (unit) {
                ++bias;
                var timeUntilUp = unit.timeUntilTurn(turnIndex, Game.defaultMoveRank, actingUnit === unit ? nextActions : null);
                forecast.push({
                    bias: bias,
                    remainingTime: timeUntilUp,
                    turnIndex: turnIndex,
                    unit: unit
                });
            });
        };
        var this_1 = this;
        for (var turnIndex = 0; turnIndex < 8; ++turnIndex) {
            _loop_1(turnIndex);
        }
        forecast.sort(function (a, b) {
            var sortOrder = a.remainingTime - b.remainingTime;
            var biasOrder = a.bias - b.bias;
            return sortOrder !== 0 ? sortOrder : biasOrder;
        });
        forecast = forecast.slice(0, 10);
        return forecast;
    };
    Battle.prototype.raiseEvent = function (eventID, data) {
        if (data === void 0) { data = null; }
        var conditions = __spread(this.conditions);
        from(conditions)
            .each(function (v) { return v.invoke(eventID, data); });
    };
    Battle.prototype.registerAI = function (ai) {
        this.aiList.push(ai);
    };
    Battle.prototype.resume = function () {
        if (--this.suspendCount < 0)
            this.suspendCount = 0;
    };
    Battle.prototype.runAction = function (action, actingUnit, targetUnits, useAiming) {
        if (useAiming === void 0) { useAiming = true; }
        var eventData = { action: action, targets: targetUnits };
        this.raiseEvent('actionTaken', eventData);
        targetUnits = eventData.targets;
        if ('announceAs' in action && action.announceAs != null)
            actingUnit.announce(action.announceAs);
        from(action.effects)
            .where(function (it) { return it.targetHint === 'user'; })
            .each(function (effect) {
            term.print("apply effect '" + effect.type + "'", "retarg: " + effect.targetHint);
            var effectHandler = Game.moveEffects[effect.type];
            effectHandler(actingUnit, [actingUnit], effect);
        });
        from(targetUnits)
            .each(function (v) { return v.takeHit(actingUnit, action); });
        if (action.effects === null)
            return [];
        var targetsHit = [];
        var accuracyRate = 'accuracyRate' in action ? action.accuracyRate : 1.0;
        for (var i = 0; i < targetUnits.length; ++i) {
            var baseOdds = 'accuracyType' in action ? Game.math.accuracy[action.accuracyType](actingUnit.battlerInfo, targetUnits[i].battlerInfo) : 1.0;
            var aimRate = 1.0;
            if (useAiming) {
                var eventData_1 = {
                    action: clone(action),
                    aimRate: 1.0,
                    targetInfo: clone(targetUnits[i].battlerInfo)
                };
                actingUnit.raiseEvent('aiming', eventData_1);
                aimRate = eventData_1.aimRate;
            }
            var odds = Math.min(Math.max(baseOdds * accuracyRate * aimRate, 0.0), 1.0);
            var isHit = random.chance(odds);
            term.print("odds of hitting " + targetUnits[i].name + " at ~" + Math.round(odds * 100) + "%", isHit ? "hit" : "miss");
            if (isHit) {
                this.notifyAIs('unitTargeted', targetUnits[i].id, action, actingUnit.id);
                targetsHit.push(targetUnits[i]);
            }
            else {
                targetUnits[i].evade(actingUnit, action);
            }
        }
        if (targetsHit.length == 0)
            return [];
        // apply move effects to target(s)
        from(targetsHit)
            .each(function (v) { return v.beginTargeting(actingUnit); });
        var animContext = {
            effects: from(action.effects)
                .where(function (it) { return from(['selected', 'random']).anyIs(it.targetHint); })
                .where(function (it) { return it.type != null; })
                .toArray(),
            pc: 0,
            nextEffect: function () {
                if (this.pc < this.effects.length) {
                    var effect = this.effects[this.pc++];
                    var targets = effect.targetHint == 'random'
                        ? [random.sample(targetsHit)]
                        : targetsHit;
                    term.print("apply effect '" + effect.type + "'", "retarg: " + effect.targetHint);
                    Game.moveEffects[effect.type](actingUnit, targets, effect);
                }
                return this.pc < this.effects.length;
            }
        };
        if (action.animation in Game.animations) {
            Game.animations[action.animation]
                .call(animContext, actingUnit, targetsHit, false);
        }
        while (animContext.nextEffect())
            ;
        from(targetsHit)
            .each(function (v) { return v.endTargeting(); });
        return targetsHit;
    };
    Battle.prototype.spawnEnemy = function (enemyClass) {
        term.print("spawn new enemy '" + enemyClass + "'");
        var newUnit = new BattleUnit(this, enemyClass);
        this.battleUnits.push(newUnit);
        this.enemyUnits.push(newUnit);
    };
    Battle.prototype.suspend = function () {
        ++this.suspendCount;
    };
    Battle.prototype.tick = function () {
        if (this.suspendCount > 0 || this.result != null)
            return;
        term.print("");
        term.print("begin CTB turn cycle #" + (this.timer + 1));
        ++this.timer;
        var isUnitDead = function (unit) { return !unit.isAlive(); };
        var unitLists = [this.enemyUnits, this.playerUnits];
        from.apply(void 0, __spread(unitLists)).each(function (unit) { return unit.beginCycle(); });
        from(this.conditions)
            .each(function (condition) { return condition.beginCycle(); });
        this.raiseEvent('beginCycle');
        var actionTaken = false;
        while (!actionTaken) {
            from.apply(void 0, __spread(unitLists)).each(function (unit) {
                actionTaken = unit.tick() || actionTaken;
            });
            if (from(this.playerUnits).all(isUnitDead)) {
                music.adjust(0.0, 120);
                this.ui.fadeOut(2.0);
                this.result = BattleResult.Lose;
                term.print("all player characters have been KO'd");
                return;
            }
            if (from(this.enemyUnits).all(isUnitDead)) {
                music.adjust(0.0, 60);
                this.ui.fadeOut(1.0);
                this.result = BattleResult.Win;
                term.print("all enemies have been KO'd");
                return;
            }
        }
        from.apply(void 0, __spread(unitLists)).each(function (unit) { return unit.endCycle(); });
    };
    Battle.prototype.unregisterAI = function (ai) {
        from(this.aiList)
            .where(function (v) { return v === ai; })
            .remove();
    };
    return Battle;
}());
//# sourceMappingURL=data:application/json;base64,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