/**
 * persist.js - store persistent data with maps and persons.
 *
 * persist.js lets you store variables with persons and maps.  It also
 * keeps them, even when you switch maps.  Data is kept in one place,
 * making it easy to save and load.
 *
 * persist.js also lets you write map scripts inside Sphere's code
 * editor, giving you F7 syntax checking, syntax highlighting, and
 * full editing comfort.
 *
 * Wiki: http://www.spheredev.org/wiki/Persist.js
 */

var persist = (function () {
    var mapEvents = [{fn:'enter',      event:SCRIPT_ON_ENTER_MAP},
                     {fn:'leave',      event:SCRIPT_ON_LEAVE_MAP},
                     {fn:'leaveNorth', event:SCRIPT_ON_LEAVE_MAP_NORTH},
                     {fn:'leaveSouth', event:SCRIPT_ON_LEAVE_MAP_SOUTH},
                     {fn:'leaveEast',  event:SCRIPT_ON_LEAVE_MAP_EAST},
                     {fn:'leaveWest',  event:SCRIPT_ON_LEAVE_MAP_WEST}];
    var personEvents = [{fn:'create',    event:SCRIPT_ON_CREATE},
                        {fn:'destroy',   event:SCRIPT_ON_DESTROY},
                        {fn:'touch',     event:SCRIPT_ON_ACTIVATE_TOUCH},
                        {fn:'talk',      event:SCRIPT_ON_ACTIVATE_TALK},
                        {fn:'generator', event:SCRIPT_COMMAND_GENERATOR}];

    /****************************************
     * File layer: process paths and files. *
     ****************************************/

    /* Read in all the text in a file. */
    function readFile(path)
    {
        var bytes;
        var f = OpenRawFile(path);
        try {
            bytes = f.read(f.getSize());
        } finally {
            f.close();
        }
        return CreateStringFromByteArray(bytes);
    }

    /* Path of the map script directory, relative to 'other/'.
     * Include the slash at the end. */
    var scriptPath = "../scripts/maps/";

    /**
     * Get the path of script files for maps, relative to 'other/'.
     * @returns {string} the script path
     */
    function getScriptPath()
    {
        return scriptPath;
    }

    /**
     * Set the path of script files for maps, relative to 'other/'.
     * @param newPath {string} the new path for map scripts
     */
    function setScriptPath(newPath)
    {
        scriptPath = newPath;
    }

    /* Get the map script file, given the map filename. */
    function scriptFromMap(map)
    {
        // Lose the '.rmp' extension
        var mapExt = '.rmp';
        var baseName = map.substring(0, map.length - mapExt.length);
        return scriptPath + baseName + '.js';
    }

    /* Load the script of the map into a string. */
    function loadMapScriptFile(map) {
        return readFile(scriptFromMap(map));
    }

    /*******************************************************
     * State layer: manage world/map/person state objects. *
     *******************************************************/

    /* Holds all world/map/person state variables. */
    var world = {};

    /**
     * Get the state of the world.
     * @returns {Object} the world state
     */
    function getWorldState()
    {
        return world;
    }

    /**
     * Set the state of the world.
     * @param newWorld {Object} the new world state
     */
    function setWorldState(newWorld)
    {
        world = newWorld;
    }

    /* Check if a map state exists in the world. */
    function mapStateExists(map)
    {
        return map in getWorldState();
    }

    /* Get the state of a map. */
    function getMapState(map)
    {
        var ws = getWorldState();
        if (mapStateExists(map))
            return ws[map];
        return ws[map] = {};
    }

    /* Set the state of a map. */
    function setMapState(map, newState)
    {
        var ws = getWorldState();
        ws[map] = newState;
    }

    /* Check if a person state exists in a map. */
    function personStateExists(map, person)
    {
        return mapStateExists(map) && (person in getMapState(map));
    }

    /* Get the state of a person. */
    function getPersonState(map, person)
    {
        var ms = getMapState(map);
        if (person in ms)
            return ms[person];
        return ms[person] = {};
    }

    /* Set the state of a person. */
    function setPersonState(map, person, newState)
    {
        var ms = getMapState(map);
        ms[person] = newState;
    }

    /* Check that a member of an object is not internal or a prototype member. */
    function isCustomMember(object, member)
    {
        return Object.prototype.hasOwnProperty.call(object, member) && Object.prototype.propertyIsEnumerable.call(object, member);
    }

    /* Check if a member is a map variable. */
    function isMapVariable(member)
    {
        for (var me = 0; me < mapEvents.length; ++me) {
            if (member == mapEvents[me].fn)
                return false;
        }

        for (var p in GetPersonList()) {
            if (member == p)
                return false;
        }

        return true;
    }

    /* Copy map variables from a map script object. */
    function initMapState(map, script)
    {
        var mapState = {};
        for (var element in script) {
            if (isCustomMember(script, element) && isMapVariable(element))
                mapState[element] = script[element];
        }
        setMapState(map, mapState);
    }

    /* Check if a member is a person variable. */
    function isPersonVariable(member)
    {
        for (var pe = 0; pe < personEvents.length; ++pe) {
            if (member == personEvents[pe].fn)
                return false;
        }
        return true;
    }

    /* Copy person variables from a person script object */
    function initPersonState(map, person, script)
    {
        var personState = {};
        for (var element in script) {
            if (isCustomMember(script, element) && isPersonVariable(element))
                personState[element] = script[element];
        }
        setPersonState(map, person, personState);
    }

    /***********************************************************
     * Evaluation layer: eval files and manage loaded scripts. *
     ***********************************************************/

    /* Loaded map script objects. */
    var scriptCache = {};

    /* Evaluate the matching script for a map. */
    function loadMapScript(map)
    {
        if (map in scriptCache)
            return scriptCache[map];
        loadMapScriptFile(map);
        return scriptCache[map] = eval(loadMapScriptFile(map));
    }

    /**
     * Clear the script cache.
     */
    function clearScriptCache()
    {
        scriptCache = {};
    }

    /* Run a map event script. */
    function runMapScript(map, event)
    {
        var mapScript = loadMapScript(map);
        if (event in mapScript)
            mapScript[event](world, world[map]);
    }

    /* Run a person event script. */
    function runPersonScript(map, person, event)
    {
        var mapScript = loadMapScript(map);
        if (person in mapScript && event in mapScript[person])
            mapScript[person][event](world, world[map], world[map][person]);
    }

    /**********************************************
     * Hook layer: define what happens in events. *
     **********************************************/

    /* List the persons to whom events should be attached. */
    function getImportantPersons()
    {
        var persons = [];
        for (var p = 0, plist = GetPersonList(); p < plist.length; ++p) {
            if (plist[p] != "" && (!IsInputAttached() || plist[p] != GetInputPerson()))
                persons.push(plist[p]);
        }
        return persons;
    }

    /* Make sure current map (and person) state variables are set. */
    function ensureState()
    {
        var map = GetCurrentMap();

        // If the map state exists, chances are its person states do too.
        if (mapStateExists(map))
            return;

        var template = loadMapScript(map);
        initMapState(map, template);
        for (var p = 0, persons = getImportantPersons(); p < persons.length; ++p) {
            if (persons[p] in template)
                initPersonState(map, persons[p], template[persons[p]]);
        }
    }

    /* Run a map event for the current map. */
    function triggerMapEvent(which)
    {
        runMapScript(GetCurrentMap(), which);
    }

    /* Run a person event for the current person. */
    function triggerPersonEvent(which)
    {
        runPersonScript(GetCurrentMap(), GetCurrentPerson(), which);
    }

    /* Re-run the create scripts of each person on the current map. */
    function recreatePersons()
    {
        var persons = getImportantPersons();
        for (var p = 0; p < persons.length; ++p)
            CallPersonScript(persons[p], SCRIPT_ON_CREATE);
    }

    /****************************************
     * Binding layer: wire events to hooks. *
     ****************************************/

    /* Connect map events to scripts. */
    function bindMapEvents()
    {
        for (var e = 0; e < mapEvents.length; ++e)
            SetDefaultMapScript(mapEvents[e].event, 'persist.triggerMapEvent("' + mapEvents[e].fn + '")');

        // Overwrite map enter event, it needs special handling.
        SetDefaultMapScript(SCRIPT_ON_ENTER_MAP, 'persist.ensureState(); ' +
                                                 'persist.bindPersonsEvents(); ' +
                                                 'persist.recreatePersons(); ' +
                                                 'persist.triggerMapEvent("enter"); ');
    }

    /* Connect events of all persons (sic) to scripts. */
    function bindPersonsEvents()
    {
        var persons = getImportantPersons();
        for (var p = 0; p < persons.length; ++p) {
            for (var e = 0; e < personEvents.length; ++e) {
                SetPersonScript(persons[p], personEvents[e].event, 'persist.triggerPersonEvent("' + personEvents[e].fn + '")');
            }
        }
    }

    /**
     * Activate the persist.js framework.
     */
    function init()
    {
        bindMapEvents();
    }

    /*************
     * Interface *
     *************/
    return {
        // Use these...
        getScriptPath: getScriptPath,
        setScriptPath: setScriptPath,
        getWorldState: getWorldState,
        setWorldState: setWorldState,
        clearScriptCache: clearScriptCache,
        init: init,

        // ... not these!
        ensureState: ensureState,
        triggerMapEvent: triggerMapEvent,
        triggerPersonEvent: triggerPersonEvent,
        recreatePersons: recreatePersons,
        bindPersonsEvents: bindPersonsEvents,
    };
})();
