// scenario.js
// Scenario class for Sphere

/*  Copyright (C) 2008  Stephen R. Gold

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
    
// A Scenario is a set of Parameters that can be 
//   loaded from or saved to a text file.

RequireScript("ui/menu.js");
RequireScript("ui/scaled_font.js");
RequireScript("ui/text_line.js");
RequireScript("utilities/parameter.js");

// global variables

ActiveScenario = new Scenario();
PlayerPreferences = new Scenario("preferences");
EditScenario = undefined;

FlagValueList = [false, true];
PlayerFlagValue = "player preference (yes/no)";
PlayerFlagValueList = [false, PlayerFlagValue, true];

// constructor

function Scenario(name) {
  if (this instanceof Scenario == false) {
    return new Scenario(name);
  }
  //DebugCall("Scenario", [name]);
  
  this.name = name;
  this.parameters = [];

  // colors for user interface
  this.headTextColor = CreateColor(255, 255, 128);       // bright yellow
  this.headBackgroundColor = CreateColor(0, 0, 64);      // dark blue
  this.backgroundColor = CreateColor(44, 44, 44);        // dark grey
  
  this.font = ScaledFont(Screen.height/40);
  this.fileSuffix = ".txt";
  if (name == undefined) {
    this.directory = "scenarios";
  }
}

Scenario.prototype.addParameter =
function (name, eval_string, default_value, value_list, enabler) {
  //DebugCall("Scenario.addParameter", [name, eval, default_value, value_list, enabler]);

  if (typeof(name) != "string" || name.substr(-4) == "Flag") {
    Abort("Invalid parameter name in Parameter(): " + Quote(name));
  }
  if (this.parameters[name] != undefined) {
    Abort("Duplicated scenario parameter: " + name);
  }
  
  var parm = new Parameter(eval_string, default_value, value_list, enabler);

  this.parameters[name] = parm;
}

// apply preferences
Scenario.prototype.apply =
function (scenario) {
  //DebugCall("Scenario.apply", [scenario]);
  
  var file_name = scenario.name + scenario.fileSuffix;
  if (scenario.directory != undefined) {
    file_name = scenario.directory + "/" + file_name;
  }
  var file = OpenFile(file_name);

  for (var name in scenario.parameters) {
    var old_value = this.parameters[name].getValue();
    if (old_value == "player preference" || old_value == PlayerFlagValue) {
      var default_value = scenario.parameters[name].defaultValue;
      var value = file.read(name, default_value);
      this.parameters[name].setValue(value);
    }
  }
}

Scenario.prototype.edit =
function () {
  //DebugCall("Scenario.edit");
  
  EditScenario = this;
  
  var saved_rects = ActiveWindows;
  ActiveWindows = [];

  var font = this.font;
  var line_height = font.getHeight();
  
  // title
  var body_rect = Screen.clone();
  var title_rect = body_rect.carve("top", 0, Screen.height/20);
  var title_text = "Scenario Editor";
  if (this.directory == undefined) {
    title_text = 'Editing player preferences';
  } else if (this.name != undefined) {
    title_text = 'Editing \"' + this.name + '" scenario';
  }
  this.title = TextLine(title_rect, this.headBackgroundColor, this.headTextColor,
    title_text);
  
  // put menu in the middle of the screen
  var middle_rect = body_rect.overlay(0.04, 0.05, 0.62, 0.9);

  var tab_x = middle_rect.width/2;
  var right_header_rect = middle_rect.carve("top", 0, line_height);
  var left_header_rect = right_header_rect.carve("left", tab_x, 0);

  // TextLines for header
  this.headBox1 = TextLine(left_header_rect, 
    this.headBackgroundColor, this.headTextColor, 
    "Parameter:", "left");
  this.headBox2 = TextLine(right_header_rect, 
    this.headBackgroundColor, this.headTextColor, 
    "Value", "center");

  var menu = new Menu(middle_rect, font);
  
  menu.drawLine = ScenarioEditDrawLine;
  menu.update = ScenarioEditUpdate;  

  menu.bindKey(KEY_MINUS, "ScenarioEditPrev()");
  menu.bindKey(KEY_LEFT, "ScenarioEditPrev()");
  menu.bindKey(KEY_EQUALS, "ScenarioEditNext()");
  menu.bindKey(KEY_RIGHT, "ScenarioEditNext()");

  // save and quit icons
  var rect = body_rect.overlay(0.7, 0.2, 0.26, 0.07);
  this.enter = new Icon(rect);
  if (this.name == "preferences") {
    this.enter.bindAction(MenuEnter, "'Save [Enter]");
  } else {
    this.enter.bindAction(MenuEnter, "'Save and Test");
  }

  rect = body_rect.overlay(0.7, 0.3, 0.26, 0.07);
  this.escape = new Icon(rect);
  this.escape.bindAction(MenuEscape, "'Quit [Esc]");
 
  ScenarioEditExtra(menu);
  menu.execute();

  ActiveWindows = saved_rects;
  var result = menu.lastKey;
  
  EditScenario = undefined;
  
  //DebugLog.write("Scenario.edit() returns " + Quote(result));
  return result;
}

Scenario.prototype.load =
function (scenario_name) {
  //DebugCall("Scenario.load", [scenario_name]);

  if (scenario_name == undefined) {
    scenario_name = this.name;
  } else {
    this.name = scenario_name;
  }
  
  var file_name = scenario_name + this.fileSuffix;
  if (this.directory != undefined) {
    file_name = this.directory + "/" + file_name;
  }
  var file = OpenFile(file_name);
  
  // load all parameters from the file
  for (var name in this.parameters) {
    var default_value = this.parameters[name].defaultValue;
    var value = file.read(name, default_value);
    if (default_value == PlayerFlagValue && value != PlayerFlagValue) {
      value = Number(value);
    }

    this.parameters[name].setValue(value);
    
    //DebugLog.write(" :" + name + "=" + Quote(value));
  }
  file.close(); 
}

// override any parameters that are disabled
Scenario.prototype.overrides =
function () {
  //DebugCall("Scenario.overrides");
  var done_flag = false;
  
  while (!done_flag) {
    for (var name in this.parameters) {
      var p = this.parameters[name];
      if (!p.isEnabled()) {
        done_flag = !p.setToDefault();
      }
    }
  }
}

Scenario.prototype.save =
function (scenario_name) {
  //DebugCall("Scenario.save", [scenario_name]);

  if (scenario_name == undefined) {
    scenario_name = this.name;
  } else {
    this.name = scenario_name;
  }
 
  var file_name = scenario_name + this.fileSuffix;
  if (this.directory != undefined) {
    file_name = this.directory + "/" + file_name;
  }
  var file = OpenFile(file_name);
  
  // save all parameters to the file
  for (var name in this.parameters) {
    var value = this.parameters[name].getValue();
    file.write(name, value);
    //DebugLog.write(" :" + name + "=" + Quote(value));
  }
  file.close();
}

// draw one line of the scenario edit menu
function ScenarioEditDrawLine(line) {
  //DebugCall("ScenarioEditDrawLine", [line]);

  if (!(line instanceof ScreenRect)) {
    Abort("Invalid line");
  }

  var font = ActiveMenu.font;
  var choice = line.choice;

  var bg, fg;
  if (choice == ActiveMenu.choice) {
    bg = ActiveMenu.selectionBackgroundColor;    
    fg = ActiveMenu.selectionTextColor;
  } else {
    bg = ActiveMenu.normalBackgroundColor;    
    fg = ActiveMenu.normalTextColor;
  }

  // put parameter name in lower case with spaces between words  
  var nameText = "";
  for (var i = 0; i < choice.length; i++) {
    var c = choice.charAt(i);
    var lc = c.toLowerCase();
    if (c != lc && i != 0) {
      nameText += " ";
    } else if (c == ".") {
      lc = " ";
    }
    nameText += lc;
  }
  
  var parm = EditScenario.parameters[choice];
  var value = parm.getValue();
  var valueText = String(value);
  if (parm.valueList == PlayerFlagValueList) {
    nameText += "?";
    if (value == PlayerFlagValue) {
      valueText = value;
    } else {
      valueText = value ? "yes" : "no";
    }
  
  } else if (parm.valueList == FlagValueList) {
    nameText += "?";
    valueText = value ? "yes" : "no";
  }
  
  var nameLine = line.nameLine;
  nameLine.setBackgroundColor(bg);
  nameLine.setTextColor(fg);
  nameLine.draw(nameText);  

  var valueLine = line.valueLine.clone();
  if (choice == ActiveMenu.choice) {
    // prev/next icons
    var pixels = font.getHeight();
    var prev_rect = valueLine.carve("left", pixels, 0);
    var next_rect = valueLine.carve("right", pixels, 0);
           
    ActiveMenu.prevIcon.copyRect(prev_rect);
    ActiveMenu.prevIcon.draw();

    ActiveMenu.nextIcon.copyRect(next_rect);
    ActiveMenu.nextIcon.draw();
  }
  valueLine = new TextLine(valueLine, bg, fg, valueText);
  valueLine.draw();
  
  //DebugLog.write(" leaving ScenarioEditDrawLine()");
}

// menu extensions for scenario edit
function ScenarioEditExtra(menu) {
  //DebugCall("ScenarioEditExtra");
  
  // extra windows used in the scenario edit menu
  var font = menu.font;
  var lines = menu.textLines;
  for (var i in lines) {
    var line = lines[i];

    var tab_x = lines[i].width/2;
    var valueArea = line.clone();
    var nameArea = valueArea.carve("left", tab_x, 0);
    line.nameLine = TextLine(nameArea, undefined, undefined, undefined, "left");
    line.valueLine = TextLine(valueArea, undefined, undefined, undefined, "center");
  }

  var pixels = menu.font.getHeight();
  var rect = new ScreenRect(0, 0, pixels, pixels);
  menu.prevIcon = new Icon(rect);
  menu.prevIcon.setStyle(); // plain rectangle
  menu.prevIcon.bindAction(ScenarioEditPrev, "left");
  
  menu.nextIcon = new Icon(rect);
  menu.nextIcon.setStyle(); // plain rectangle
  menu.nextIcon.bindAction(ScenarioEditNext, "right");
}

function ScenarioEditPrev() {
  //DebugCall("ScenarioEditPrev");
  
  var choice = ActiveMenu.choice;
  var parm = EditScenario.parameters[ActiveMenu.choice];
  parm.previousValue();
}

function ScenarioEditNext() {
  //DebugCall("ScenarioEditNext");
  
  var choice = ActiveMenu.choice;
  var parm = EditScenario.parameters[ActiveMenu.choice];
  parm.nextValue();
}

// update function for scenario edit menu
function ScenarioEditUpdate() {
  //DebugCall("ScenarioEditUpdate");
  
  Screen.clip();   
  Screen.fill(EditScenario.backgroundColor);
  
  // draw header and icons
  EditScenario.title.draw();
  EditScenario.headBox1.draw();
  EditScenario.headBox2.draw();
  EditScenario.enter.draw();
  EditScenario.escape.draw();

  // list the names of all enabled parameters
  var name_list = [];
  var parameters = EditScenario.parameters;
  for (var name in parameters) {
    var parm = parameters[name];
    if (parm.isEnabled()) {
      name_list.push(name);
    }
  }
    
  // sort the list
  name_list.sort();
  
  //DebugLog.write(" ScenarioEditUpdate() returns " + Quote(name_list));
  return name_list;
}