// raw_file.js
// byte-level file reads and writes for Sphere

/*  Copyright (C) 2008  Stephen R. Gold

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
    

RequireScript("utilities/debug.js");

DumpFlag = false;

function CopyRawFile(from_path, to_path) {
  //DebugCall("CopyRawFile", [from_path, to_path]);
  
  var in_file = OpenRawFile("../" + from_path, false);
  var out_file = OpenRawFile("../" + to_path, true);
  var buffer_size = 16384;  // bytes
  
  do {
    var buffer = in_file.read(buffer_size);
    out_file.write(buffer);
  } while (buffer.length == buffer_size);
  
  in_file.close();
  out_file.close();
}

// read common data types

function ReadByte(file) {
  return ReadInt(file, 1);
}
function ReadDword(file) {
  return ReadInt(file, 4);
}
function ReadString(file) {
  var num_bytes = ReadWord(file);
  var byte_array = ReadBytes(file, num_bytes);
  return CreateStringFromByteArray(byte_array);
}
function ReadWord(file) {
  return ReadInt(file, 2);
}

// fixed-length littleendian integer 
function ReadInt(file, num_bytes) {
  //DebugCall("ReadInt", [num_bytes]);

  if (num_bytes == undefined) {
    Abort("Undefined num_bytes in ReadInt()");
  }

  var byte_array = ReadBytes(file, num_bytes);
  
  var result = 0;
  var shift = 0;
  for (var i = 0; i < num_bytes; i++) {
    result += (byte_array[i] << shift);
    shift += 8;
  }

  //DebugLog.write("ReadInt(file, " + Quote(num_bytes) + ") returns " + Quote(result));
  return result;
}

// bytearray of known length
function ReadBytes(file, num_bytes) {
  if (num_bytes == undefined) {
    Abort("Undefined num_bytes in ReadBytes()");
  }
  
  var byte_array = file.read(num_bytes);
  if (byte_array.length != num_bytes) {
    Abort("Unexpected end-of-file in ReadBytes()");
  }

  if (DumpFlag) {
    var msg = " ReadBytes(file, " + Quote(num_bytes) + ") = " + ArrayHex(byte_array);
    DebugLog.write(msg);
  }
     
  return byte_array;
}

// write common data types

function WriteByte(file, value) {
  //DebugCall("WriteByte", [file, value]);
  
  WriteInt(file, 1, value);
}
function WriteDword(file, value) {
  return WriteInt(file, 4, value);
}
function WriteString(file, str) {
  //DebugCall("WriteString", [file, str]);
  
  var byte_array = CreateByteArrayFromString(str);
  WriteWord(file, str.length);
  WriteBytes(file, byte_array);
}
function WriteStringNullTerminated(file, str) {
  //DebugCall("WriteStringNullTerminated", [file, str]);
  
  var byte_array = CreateByteArrayFromString(str);
  WriteWord(file, str.length + 1);
  WriteBytes(file, byte_array);
  WriteByte(file, 0);
}
function WriteWord(file, value) {
  return WriteInt(file, 2, value);
}

// fixed-length littleendian integer 
function WriteInt(file, num_bytes, value) {
  //DebugCall("WriteInt", [file, num_bytes, value]);
  
  if (num_bytes == undefined) {
    Abort("Undefined num_bytes in WriteInt()");
  }
  if (value == undefined) {
    Abort("Undefined value in WriteInt()");
  }
  var byte_array = CreateByteArray(num_bytes);
  
  var shift = 0;
  for (var i = 0; i < num_bytes; i++) {
    byte_array[i] = (value >> shift) & 0xFF;
    shift += 8;
  }

  WriteBytes(file, byte_array);
}

// entire bytearray
function WriteBytes(file, byte_array) {

  if (byte_array == undefined) {
    Abort("Undefined byte_array in WriteBytes()");
  }

  var num_bytes = byte_array.length; 
  file.write(byte_array);
  
  if (DumpFlag) {
    var msg = " WriteBytes(file, " + ArrayHex(byte_array) + ")";
    DebugLog.write(msg);
  }
}


// hexadecimal notation

function ArrayHex(byte_array) {
  if (byte_array == undefined) {
    Abort("Undefined byte_array in ArrayHex()");
  }
  var result = "";
  
  if (byte_array != undefined) {
    for (var i = 0; i < byte_array.length; i++) {
      result += " " + Hex(byte_array[i]);
    }
  }
  
  return result;
}

function Hex(byt) {
  if (byt == undefined || byt < 0 || byt > 0xFF) {
    Abort("Undefined byt in Hex(): " + Quote(byt));
  }
  var result = "";
  
  var d2 = byt & 0xF;
  var d1 = (byt & 0xF0) >> 4;
  
  if (d1 < 10) {
    result += d1.toString();
  } else {
    var code = d1 + 55;
    result += CreateStringFromCode(code);
  }

  if (d2 < 10) {
    result += d2.toString();
  } else {
    var code = d2 + 55;
    result += CreateStringFromCode(code);
  }

  return result;
}