// parameter.js
// Parameter class for Sphere

/*  Copyright (C) 2008  Stephen R. Gold

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/

// A Parameter represents a parameter in a Scenario.

// It has three properties:
//   eval_string:   name of the corresponding Sphere variable (must be a global)
//   default_value: default_value
//   value_list:    list of legal values

RequireScript("utilities/debug.js");

// constructor

function Parameter(eval_string, default_value, value_list, enabler) {
  if (this instanceof Parameter == false) {
    return new Parameter(eval_string, default_value, value_list);
  }
  //DebugCall("Parameter", [eval_string, default_value, value_list, enabler]);
  //DebugLog.write(" value_list=" + QuoteList(value_list));

  if (typeof(eval_string) != "string") {
    Abort("Invalid eval in Parameter(): " + Quote(eval_string));
  }
  if (typeof(value_list) != "object") {
    Abort("Invalid value_list in Parameter(): " + Quote(value_list));
  }
  
  this.defaultValue = default_value;
  this.evalString = eval_string;
  this.enabler = enabler;
  this.valueList = value_list;
}

Parameter.prototype.isEnabled =
function () {
  //DebugCall("Parameter.isEnabled", [], "for " + Quote(this.evalString) + " has enabler=" + Quote(this.enabler));
  
  if (this.enabler == undefined) {
    return true;
  }
  var flag = eval(this.enabler);
  
  //DebugLog.write ("  returns " + Quote(flag));
  return flag;
}

Parameter.prototype.nextValue =
function () {
  //DebugCall("Parameter.nextValue");

  var list = this.valueList;

  var value = this.getValue();
  var value_index;
  for (var j in list) {
    if (list[j] == value) {
      value_index = Number(j);
      break;
    }
  }

  var next_index = value_index;  
  if (value_index == undefined) {
    next_index = 0;
  } else if (value_index < list.length-1) {
    next_index = value_index + 1;
  }
  
  var new_value = list[next_index];
  
  this.setValue(new_value);
}

Parameter.prototype.previousValue =
function () {
  //DebugCall("Parameter.previousValue");

  var list = this.valueList;

  var value = this.getValue();
  var value_index;
  for (var j in list) {
    if (list[j] == value) {
      value_index = Number(j);
      break;
    }
  }

  var previous_index = value_index;  
  if (value_index == undefined) {
    previous_index = 0;
  } else if (value_index > 0) {
    previous_index = value_index - 1;
  }

  var new_value = this.valueList[previous_index];

  this.setValue(new_value);
}

Parameter.prototype.quote =
function () {
  var result;
  
  result = "Parameter{eval_string:" + Quote(this.eval_string)
         + ", default_value:" + Quote(this.default_value) 
         + ", value_list:" + QuoteList(this.value_list) + "}";
  
  return result;
}

// returns true if value was changed
Parameter.prototype.setToDefault =
function () {
  //DebugCall("Parameter.setToDefault");
  
  var old_value = this.getValue();
  if (old_value == this.defaultValue) {
    // already at default
    return false;
  }
  
  this.setValue(this.defaultValue);
  return true;
}

// value get/set

Parameter.prototype.getValue =
function () {
  //DebugCall("Parameter.getValue");

  var value = eval(this.evalString);
  
  return value;
}

Parameter.prototype.setValue =
function (new_value) {
  //DebugCall("Parameter.setValue", [new_value], this.evalString);

  var eval_string = this.evalString + "=" + Quote(new_value);
  eval(eval_string);
}