// graphics.js
// graphics primitives for Sphere

/*  Copyright (C) 2008-2009  Stephen R. Gold

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/

RequireScript("utilities/debug.js");

//DefaultFont = GetSystemFont();
DefaultFont = LoadFont("ProFont-24pt.rfn");

ImageCache = [];

function CacheImage(image_name) {
  var image = ImageCache[image_name];
  if (image == undefined) {
    image = LoadImage(image_name);
    ImageCache[image_name] = image;
  }
  return image;
}

// load and scale image to fit in a rectangular area

function LoadScaledImage(image_name, max_width, max_height, directory) {
  //DebugCall("LoadScaledImage", [image_name, max_width, max_height, directory]);
  
  if (image_name == undefined) {
    return undefined;  
  }
  if (max_width < 1) {
    Abort("Invalid max_width in LoadScaledImage()");
  }
  if (max_height < 1) {
    Abort("Invalid max_height in LoadScaledImage()");
  }
  
  var surface, zoom_factor;
  if (image_name.charAt(0) == "'") {
    // image of black text using DefaultFont
    
    var text = image_name.substring(1);

    var font = DefaultFont;
    var black = CreateColor(0, 0, 0);
    font.setColorMask(black);

    var w = font.getStringWidth(text);
    var h = font.getHeight();
    var transparent = CreateColor(255, 255, 255, 0);
    surface = CreateSurface(w, h, transparent);
    surface.drawText(font, 0, 0, text);
    zoom_factor = undefined; // unlimited scaling
    
  } else {
    // load PNG image from file
    var filename = image_name + ".png";
    if (directory != undefined) {
      filename = directory + "/" + filename;
    }
    surface = LoadSurface(filename);
    zoom_factor = 1; // don't scale up PNG images, only scale them down
  }

  // figure out how much to shrink the surface
  //   (keeping its proportions)
  if (max_width != undefined) {
    var f = max_width/surface.width;
    if (zoom_factor == undefined || f < zoom_factor) {
      zoom_factor = f;
    }
  }
  if (max_height != undefined) {
    var f = max_height/surface.height;
    if (zoom_factor == undefined || f < zoom_factor) {
      zoom_factor = f;
    }
  }
  if (zoom_factor == undefined) {
    zoom_factor = 1;
  }

  surface.rescale(surface.width * zoom_factor, surface.height * zoom_factor);
  
  return surface.createImage();
}


// display a silhouette of an image
function Silhouette(x, y, image, color) {
  //DebugCall("Silhouette", [x, y, image, color]);
  
  var sil = image.createSurface();
  var w = sil.width;
  var h = sil.height;
  if (GetVersion() > 1.13) {
    sil.setBlendMode(MULTIPLY);
    var black = CreateColor(0, 0, 0, 255);
    sil.rectangle(0, 0, w, h, black);
    if (color != undefined) {
      sil.setBlendMode(ADD); // broken in Sphere 1.4 -- probably a bug
      color.alpha = 0;
      sil.rectangle(0, 0, w, h, color);
    }
    
  } else {
    // note: this is very slow
    if (color == undefined) {
      color = CreateColor(0, 0, 0);
    }
    for (var yi = 0; yi < h; yi++) {
      for (var xi = 0; xi < w; xi++) {
        var pix = sil.getPixel(xi, yi);
        if (pix.alpha > 0) {
          pix.red = color.red;
          pix.green = color.green;
          pix.blue = color.blue;
          sil.setPixel(xi, yi, pix);
        }
      }
    }
  }
  
  // blit to screen
  sil.blit(x, y);
}

// directional arrow
function DrawVector(x1, y1, x2, y2, color) {
  //DebugCall("DrawVector", [x1, y1, x2, y2]);

  // draw stem
  Line(x1, y1, x2, y2, color);
  
  // find corners of head
  var dx = x2 - x1;
  var dy = y2 - y1;
  var den = Norm(dx, dy)/8;
  dx = dx/den;
  dy = dy/den;
  var x3 = x2 - dx - dy;
  var y3 = y2 + dx - dy;
  var x4 = x2 - dx + dy;
  var y4 = y2 - dx - dy;

  // draw head
  Triangle(x2, y2, x3, y3, x4, y4, color);
}

// outlined circle
if (GetVersion() < 1.4) {
  function OutlinedCircle(xc, yc, radius, color) {
    OutlinedEllipse(xc, yc, radius, radius, color);
  }
}

// outlined ellipse
if (GetVersion() < 1.4) {
  function OutlinedEllipse(xc, yc, rx, ry, color) {
    var x1 = xc;
    var y1 = yc + ry;
    for (var z = 1; z <= 24; z++) {
      var radians = z*Math.PI/12;
      var x2 = xc + rx*Math.sin(radians);
      var y2 = yc + ry*Math.cos(radians);
      Line(x1, y1, x2, y2, color);
      x1 = x2;
      y1 = y2;
    }
  }
}

// outlined rectangle
if (GetVersion() < 1.4) {
  function OutlinedRectangle(x, y, width, height, color) {
    if (width == 0 || height == 0) {
      return;
    }
    var x2 = x + width - 1;
    var y2 = y + height - 1;
    
    Point(x, y, color) // top left corner
    if (width > 1) {
      Line(x+1, y, x2, y, color);     // top edge & TRC 
    }
    if (height > 1) {
      Line(x, y+1, x, y2, color);     // left edge & BLC
    }
    if (width > 2 && height > 1) {
      Line(x+1, y2, x2-1, y2, color); // bottom edge
    }
    if (width > 1 && height > 2) {
      Line(x2, y+1, x2, y2-1, color); // right edge
    }
    if (width > 1 && height > 1) {
      Point(x2, y2, color)      // bottom right corner
    } 
  }
}

function RemoveColor(surface) {
  surface.setBlendMode(REPLACE);
  
  for (var x = 0; x < surface.width; x++) {
    for (var y = 0; y < surface.height; y++) {
      var color = surface.getPixel(x, y);
      var alpha = color.alpha;
      var avg = 255;
      if (alpha > 0) {
        var sum = color.red + color.green + color.blue;
        avg = Math.round(sum/3);
      }
      var grey = CreateColor(avg, avg, avg, alpha);
      surface.setPixel(x, y, grey);
    }
  }
  return surface;
}

TrimAlphaThreshold = 1;

function EmptyColumn(surface, x) {
  for (var y = 0; y < surface.height; y++) {
    var color = surface.getPixel(x, y);
    if (color.alpha > TrimAlphaThreshold) {
      return false;
    }
  }
  
  return true;
}

function EmptyRow(surface, y) {
  for (var x = 0; x < surface.width; x++) {
    var color = surface.getPixel(x, y);
    if (color.alpha > TrimAlphaThreshold) {
      return false;
    }
  }
  
  return true;
}