// direction.js
// direction routines for Sphere

/*  Copyright (C) 2008  Stephen R. Gold

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/

// Sphere convention seems to be that:
//   North = negative y direction = top of screen
//   East = postive x direction = right side of screen
//   South = postive y direction = bottom side of screen
//   West = negative x direction = left side of screen
//
// Note! This makes (x, y) a left-handed coordinate system.

RequireScript("utilities/random.js");

EightDirections = [ "north", "northeast", "east", "southeast",
                    "south", "southwest", "west", "northwest" ];

SixteenDirections = [ "north", "nne", "northeast", "ene", 
                      "east", "ese", "southeast", "sse",
                      "south", "ssw", "southwest", "wsw",
                      "west", "wnw", "northwest", "nnw" ];

// get the composite offsets indicated by the arrow keys
function ArrowKeysEast() {
  var east = 0;
  
  if (IsKeyPressed(KEY_LEFT)) {
    //DebugLog.write(" go west");
    east--;
  }
  if (IsKeyPressed(KEY_RIGHT)) {
    //DebugLog.write(" go east");
    east++;
  }
  
  return east;
}

function ArrowKeysSouth() {
  var south = 0;
  
  if (IsKeyPressed(KEY_UP)) {
    //DebugLog.write(" go north");
    south--;
  }
  if (IsKeyPressed(KEY_DOWN)) {
    //DebugLog.write(" go south");
    south++;
  }

  return south;
}

// return a random named direction
function RandomDirectionN(resolution) {
  //DebugLog.write("RandomDirection(" + Quote(resolution) + ")");
  var direction;

  var n = Math.floor(Uniform(0, resolution));
  switch (n) {
    case 0: direction = "north"; break;
    case 1: direction = "south"; break;
    case 2: direction = "east"; break;
    case 3: direction = "west"; break;
    case 4: direction = "southeast"; break;
    case 5: direction = "southwest"; break;
    case 6: direction = "northeast"; break;
    case 7: direction = "northwest"; break;
    case 8: direction = "nne"; break;
    case 9: direction = "ene"; break;
    case 10: direction = "ese"; break;
    case 11: direction = "sse"; break;
    case 12: direction = "ssw"; break;
    case 13: direction = "wsw"; break;
    case 14: direction = "wnw"; break;
    case 15: direction = "nnw"; break;
    default: Abort("error in RandomDirectionN()");
  }
  
  return direction;
}

function SlideSequence(goal_es) {
  //DebugCall("SlideSequence", [goal_es]);
  var dir_sequence;
  
  var direction = Direction4(goal_es[0], goal_es[1]); // principal direction
  if (direction == "north") {
    if (goal_es[0] > 0) {
      dir_sequence = ["north", "east", "west", "south"];
    } else {
      dir_sequence = ["north", "west", "east", "south"];
    }
  } else if (direction == "east") {
    if (goal_es[1] > 0) {
      dir_sequence = ["east", "south", "north", "west"];
    } else {
      dir_sequence = ["east", "north", "south", "west"];
    }
  } else if (direction == "west") {
    if (goal_es[1] > 0) {
      dir_sequence = ["west", "south", "north", "east"];
    } else {
      dir_sequence = ["west", "north", "south", "east"];
    }
  } else if (direction == "south") {
    if (goal_es[0] > 0) {
      dir_sequence = ["south", "east", "west", "north"];
    } else {
      dir_sequence = ["south", "west", "east", "north"];
    }
  } else {
    Abort("Unknown direction in SlideSequence(): " + Quote(direction));
  }
  
  return dir_sequence;
}

// convert named direction to move command
function MoveCommand(direction) {
  //DebugLog.write("MoveCommand(" + Quote(direction) + ")");
  var command;

  switch(direction) {
    case "north": command = COMMAND_MOVE_NORTH; break;
    case "east":  command = COMMAND_MOVE_EAST;  break;
    case "south": command = COMMAND_MOVE_SOUTH; break;
    case "west":  command = COMMAND_MOVE_WEST;  break;
    default: Abort("direction=" + Quote(direction) + " in MoveCommand()");
  }
  
  return command;
}

// convert named direction to face command
function FaceCommand(direction) {
  //DebugLog.write("FaceCommand(" + Quote(direction) + ")");
  var command;
  
  switch(direction) {
    case "north":     command = COMMAND_FACE_NORTH;     break;
    case "northeast": command = COMMAND_FACE_NORTHEAST; break;
    case "east":      command = COMMAND_FACE_EAST;      break;
    case "southeast": command = COMMAND_FACE_SOUTHEAST; break;
    case "south":     command = COMMAND_FACE_SOUTH;     break;
    case "southwest": command = COMMAND_FACE_SOUTHWEST; break;
    case "west":      command = COMMAND_FACE_WEST;      break;
    case "northwest": command = COMMAND_FACE_NORTHWEST; break;
    default: Abort("direction=" + Quote(direction) + " in FaceCommand()");
  }
  
  return command;
}

// convert offset to named direction

function DirectionN(d_east, d_south, number_of_directions) {
  //DebugCall("DirectionN", [d_east, d_south, number_of_directions]);

  var direction;
  
  switch(number_of_directions) {
    case 1: direction = "north"; break;   // spritesets with single direction use "north"
    case 4: direction = Direction4(d_east, d_south);   break;
    case 8: direction = Direction8(d_east, d_south);   break;
    case 16: direction = Direction16(d_east, d_south); break;
    default:  Abort("number=" + Quote(number_of_directions) + " in DirectionN()");
  }
  
  return direction;
}

// convert offset to one of sixteen named directions
function Direction16(d_east, d_south) {
  //DebugCall("Direction16", [d_east, d_south]);

  var direction;
  
  if (d_east == 0 && d_south == 0)
    return undefined;
  
  var ad_east = Math.abs(d_east);
  var ad_south = Math.abs(d_south);
  
  if (ad_east > 5*ad_south) {
    if (d_east > 0) {
      direction = "east";
    } else {
      direction = "west";
    }
  } else if (ad_east > 1.5*ad_south) {
    if (d_east > 0) {
      if (d_south > 0) {
        direction = "ese";
      } else {
        direction = "ene";
      }
    } else {
      if (d_south > 0) {
        direction = "wsw";
      } else {
        direction = "wnw";
      }
    }
  } else if (ad_south > 5*ad_east) {
    if (d_south > 0) {
      direction = "south";
    } else {
      direction = "north";
    }
  } else if (ad_south > 1.5*ad_east) {
    if (d_east > 0) {
      if (d_south > 0) {
        direction = "sse";
      } else {
        direction = "nne";
      }
    } else {
      if (d_south > 0) {
        direction = "ssw";
      } else {
        direction = "nnw";
      }
    }
  } else if (d_east > 0) {
    if (d_south > 0) {
      direction = "southeast";
    } else {
      direction = "northeast";
    }
  } else { // east < 0
    if (d_south > 0) {
      direction = "southwest";
    } else {
      direction = "northwest";
    }
  }
  
  //DebugLog.write(" Direction16() returns " + Quote(direction));
  return direction;
}

// convert offset to one of eight named directions
function Direction8(d_east, d_south) {
  //DebugCall("Direction8", [d_east, d_south]);

  var direction;
  
  if (d_east == 0 && d_south == 0)
    return undefined;
    
  var ad_east = Math.abs(d_east);
  var ad_south = Math.abs(d_south);
  
  if (ad_east > 2*ad_south) {
    if (d_east > 0) {
      direction = "east";
    } else {
      direction = "west";
    }
  } else if (ad_south > 2*ad_east) {
    if (d_south > 0) {
      direction = "south";
    } else {
      direction = "north";
    }
  } else if (d_east > 0) {
    if (d_south > 0) {
      direction = "southeast";
    } else {
      direction = "northeast";
    }
  } else { // east < 0
    if (d_south > 0) {
      direction = "southwest";
    } else {
      direction = "northwest";
    }
  }
  
  //DebugLog.write(" Direction8() returns " + Quote(direction));
  return direction;
}

// convert offset to one of four named directions
function Direction4(d_east, d_south) {
  //DebugCall("Direction4", [d_east, d_south]);

  var direction;
  
  if (d_east == 0 && d_south == 0)
    return undefined;
    
  var ad_east = Math.abs(d_east);
  var ad_south = Math.abs(d_south);
  
  if (ad_east > ad_south) {
    if (d_east > 0) {
      direction = "east";
    } else {
      direction = "west";
    }
    
  } else {
    if (d_south > 0) {
      direction = "south";
    } else {
      direction = "north";
    }
  }
  
  return direction;
}

// convert named direction to degrees clockwise from north
function Degrees(direction) {
  //DebugLog.write("Degrees(" + Quote(direction) + ")");
  var degrees;
  
  switch(direction) {
    case "north":     degrees =   0;   break;
    case "nne":       degrees =  22.5; break;
    case "northeast": degrees =  45;   break;
    case "ene":       degrees =  67.5; break;
    case "east":      degrees =  90;   break;
    case "ese":       degrees = 112.5; break;
    case "southeast": degrees = 135;   break;
    case "sse":       degrees = 157.5; break;
    case "south":     degrees = 180;   break;
    case "ssw":       degrees = 202.5; break;
    case "southwest": degrees = 225;   break;
    case "wsw":       degrees = 247.5; break;
    case "west":      degrees = 270;   break;
    case "wnw":       degrees = 292.5; break;
    case "northwest": degrees = 315;   break;
    case "nnw":       degrees = 337.5; break;
    default: Abort("direction=" + Quote(direction) + " in Degrees()");
  }

  return degrees;
}

// convert named direction to easting (not normalized)
function Easting(direction) {
  //DebugLog.write("Easting(" + Quote(direction) + ")");
  var easting;

  switch(direction) {
    case "north":     easting =  0; break;
    case "northeast": easting = +1; break;
    case "east":      easting = +1; break;
    case "southeast": easting = +1; break;
    case "south":     easting =  0; break;
    case "southwest": easting = -1; break;
    case "west":      easting = -1; break;
    case "northwest": easting = -1; break;
    default: Abort("direction=" + Quote(direction) + " in Easting()");
  }

  return easting;
}

// convert named direction to southing (not normalized)
function Southing(direction) {
  //DebugLog.write("Southing(" + Quote(direction) + ")");
  var southing;
  
  switch(direction) {
    case "north":     southing = -1; break;
    case "northeast": southing = -1; break;
    case "east":      southing =  0; break;
    case "southeast": southing = +1; break;
    case "south":     southing = +1; break;
    case "southwest": southing = +1; break;
    case "west":      southing =  0; break;
    case "northwest": southing = -1; break;
    default: Abort("direction=" + Quote(direction) + " in Southing()");
  }

  return southing;
}