// window.js
// Window class for Sphere

/*  Copyright (C) 2008  Stephen R. Gold

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/

// A Window is a ScreenRect with click() and draw() methods, 
//  a background color, and potentially a window style.

// Windows which need to process mouse clicks should be registered
//  at time of creation by adding them to the ActiveWindows list.

RequireScript("ui/screen_rect.js");

// constructor

Window.prototype = new ScreenRect();
Window.prototype.constructor = Window;

function Window(rect, bg_color, style) {
  if (this instanceof Window == false) {
    return new Window(rect, bg_color, style);
  }
  //DebugCall("Window", [rect, bg_color, style]);
  if (rect == undefined) {
    return this;
  }
  
  //initialize properties
  ScreenRect.apply(this, [rect.x, rect.y, rect.width, rect.height]);
  
  this.backgroundColor = bg_color;
  this.click = EatClick;
  this.setStyle(style);
  
  return this;
}

Window.prototype.clone =
function () {
  //DebugCall("Window.clone");
  
  var result = new Window(this, this.backgroundColor, this.style);
  result.click = this.click;
  
  return result;
}

Window.prototype.copyInterior =
function (rect) {
  //DebugCall("Window.copyInterior", [rect]);

  this.interior.copyRect(rect);
  
  this.x = rect.x - this.thickness;
  this.y = rect.y - this.thickness;
  this.width = rect.width + 2*this.thickness;
  this.height = rect.height + 2*this.thickness;
}

Window.prototype.copyRect =
function (rect) {
  //DebugCall("Window.copyRect");

  ScreenRect.prototype.copyRect.apply(this, [rect]);
  this.interior = this.trim(this.thickness);
}

Window.prototype.draw =
function (bg_color) {
  //DebugCall("Window.draw", [bg_color]);

  if (bg_color == undefined) {
    bg_color = this.backgroundColor;
  }
  if (bg_color == undefined) {
    Abort("Missing bg_color in Window.draw()");
  }
  
  var style = this.style;
  if (style == undefined) {
    // plain solid rectangle
    this.fill(bg_color);
    
  } else {
    // window with a stylish border
    style.setColorMask(bg_color);
    style.drawWindow(this.interior.x, this.interior.y, 
       this.interior.width, this.interior.height);
  }
}

Window.prototype.setBackgroundColor =
function (bg_color) {
  if (bg_color == undefined) {
    Abort("Missing bg_color in Window.setBackgroundColor()");
  }
  
  this.backgroundColor = bg_color;
}

Window.prototype.setInteriorWidth =
function (width) {
  //DebugCall("Window.setWidth", [width]);
  
  this.width = width + 2 * this.thickness;
  this.interior.width = width;
}

Window.prototype.setInteriorX =
function (x) {
  //DebugCall("Window.setInteriorX", [x]);
  
  this.x = x - this.thickness;
  this.interior.x = x;
}

Window.prototype.setStyle =
function (style) {
  //DebugCall("Window.setStyle", [style]);
  
  this.style = style;

  var thickness = 0;
  if (style != undefined) {
    if (style.thickness == undefined) {
      Abort("Missing thickness in Window.setStyle()");
    }
    thickness = style.thickness;
  }
  this.thickness = thickness;
  
  this.interior = this.trim(thickness);
}

function EatClick(button, screen_x, screen_y) {
  // do nothing
}


// debugging aids

Window.prototype.quote =
function () {
  var result;
  
  result = "Window{x:" 
         + Quote(this.x) + ", y:" + Quote(this.y) 
         + ", width:" + Quote(this.width) 
         + ", height:" + Quote(this.height) 
         + ", backgroundColor:" + FormatColor(this.backgroundColor) + "}";
  
  return result;
}

function LogActiveWindows() {
  DebugLog.write(" " + Quote(ActiveWindows.length) + " active windows:");
  
  for (var i in ActiveWindows) {
    var window = ActiveWindows[i];
    DebugLog.write(String(i) + " " + Quote(window));
  }
}