// text_line.js
// TextLine class for Sphere

/*  Copyright (C) 2008  Stephen R. Gold

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/

// A TextLine is a Window that displays a single line of text.
// The colors and alignment can be specified.
// If the window has non-zero height or width, then the text is 
// scaled to fit in the window.

RequireScript("ui/scaled_font.js");
RequireScript("ui/window.js");

// constructor

TextLine.prototype = new Window();
TextLine.prototype.constructor = TextLine;

function TextLine(rect, bg_color, text_color, text, h_align, v_align) {
  if (this instanceof TextLine == false) {
    return new TextLine(rect, bg_color, text_color, text, h_align, v_align);
  }
  //DebugCall("TextLine", [rect, bg_color, text_color, text, h_align, v_align]);
  
  if (rect == undefined) {
    rect = new ScreenRect();
    return;
  }
  Window.apply(this, [rect, bg_color]);
  
  if (bg_color == undefined) {  
    bg_color = CreateColor(0, 0, 0); // black
  }
  if (text_color == undefined) {  
    text_color = CreateColor(255, 255, 255); // white
  }
  if (text == undefined) {
    text = "";
  }
  if (h_align == undefined) {
    h_align = "center";
  }
  if (v_align == undefined) {
    v_align = "center";
  }
  
  // initialize properties
  this.horizontalAlign = h_align;
  this.text = text;
  this.textColor = text_color;
  this.verticalAlign = v_align;
}


TextLine.prototype.draw =
function(text, text_color, bg_color) {
  //DebugCall("TextLine.draw", [text, text_color, bg_color], Quote(this));
  
  if (text == undefined) {
    text = this.text;
  }
  if (text_color == undefined) {
    text_color = this.textColor;
  }
  
  if (text_color == undefined) {
    Abort("Missing text_color in TextLine.draw()");
  }

  var font = new ScaledFont(this.interior.height);
  if (this.width > 0) {
    font.fitString(text, this.interior.width);
  }
  
  var text_width = font.getStringWidth(text);
  var text_height = font.getHeight();

  var text_x = this.interior.x;
  if (this.horizontalAlign == "center") {
    text_x += (this.interior.width - text_width)/2;
  } else if (this.horizontalAlign == "right") {
    text_x += (this.interior.width - text_width);
  } else if (this.horizontalAlign != "left") {
    Abort("Unknown horizontalAlign in TextLine.draw(): " + Quote(this.horizontalAlign));
  }

  var text_y = this.interior.y;
  if (this.verticalAlign == "center") {
    text_y += (this.interior.height - text_height)/2;
  } else if (this.verticalAlign == "bottom") {
    text_y += (this.interior.height - text_height);
  } else if (this.verticalAlign != "top") {
    Abort("Unknown verticalAlign in TextLine.draw(): " + Quote(this.verticalAlign));
  }

  var fill_window = this.clone();
  if (this.width == 0) {
    // just fill the width of the text
    fill_window.setInteriorX(text_x);
    fill_window.setInteriorWidth(text_width);
  }

  Window.prototype.draw.apply(fill_window, [bg_color]);
  
  font.setColorMask(text_color);
  font.drawText(text_x, text_y, text);
  
  //DebugLog.write(" leaving TextLine.draw()");
}

TextLine.prototype.quote =
function () {
  var result;
  
  result = "TextLine{x:" + Quote(this.x) + ", y:" + Quote(this.y) 
         + ", width:" + Quote(this.width) 
         + ", height:" + Quote(this.height) 
         + ", text:" + Quote(this.text)
         + ", ha:" + Quote(this.horizontalAlign)
         + ", va:" + Quote(this.verticalAlign)
         + ", bg:" + FormatColor(this.backgroundColor)
         + ", fg:" + FormatColor(this.textColor) + "}";
  
  return result;
}

TextLine.prototype.setText =
function (text) {
  //DebugCall("TextLine.setText", [text]);
  this.text = text;
}

TextLine.prototype.setTextColor =
function (color) {
  //DebugCall("TextLine.setTextColor", [color]);
  this.textColor = color;
}