// text_box.js
// TextBox class for Sphere

/*  Copyright (C) 2008  Stephen R. Gold

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/

// A TextBox is a Window that displays auto-wrapped, left-aligned text.
//
// The colors, (scaled) font, and vertical alignment can be specified.

RequireScript("ui/scaled_font.js");
RequireScript("ui/window.js");

// constructor

TextBox.prototype = new Window();
TextBox.prototype.constructor = TextBox;

function TextBox(window, text_color, text, font, v_align) {
  if (this instanceof TextBox == false) {
    return new TextBox(window, text_color, text, font, v_align);
  }
  //DebugCall("TextBox", [window, text_color, text, font, v_align]);
  
  if (window == undefined) {
    window = new Window();
  }
  if (text_color == undefined) {  
    text_color = CreateColor(255, 255, 255); // white
  }
  if (v_align == undefined) {
    v_align = "center";
  }

  // initialize properties
  Window.apply(this, [window, window.backgroundColor, window.style]);
  this.setFont(font);
  this.setTextColor(text_color);
  this.setText(text);
  this.setOffset(0);
  this.verticalAlign = v_align;
}


TextBox.prototype.draw =
function (text, text_color, bg_color) {
  //DebugCall("TextBox.draw", [text, text_color, bg_color]);
  
  if (text == undefined) {
    text = this.text;
  }
  if (text_color == undefined) {
    text_color = this.textColor;
  }
  
  if (text_color == undefined) {
    Abort("Missing text_color in TextBox.draw()");
  }

  var font = this.font;
  var box_height = this.interior.height;
  var text_height = this.textHeight;
  var offset = 0;
  if (this.verticalAlign == "center") {
    offset = (box_height - text_height)/2;
  } else if (this.verticalAlign == "bottom") {
    offset = box_height - text_height;
  } else if (this.verticalAlign == "top") {
    offset = this.offset;
  } else {
    Abort("Unknown verticalAlign in TextBox.draw(): " + Quote(this.verticalAlign));
  }

  Window.prototype.draw.apply(this, [bg_color]);
  
  font.setColorMask(text_color);
  font.drawTextBox(this.interior.x, this.interior.y, 
    this.interior.width, this.interior.height, offset, text);
  
  //DebugLog.write(" leaving TextBox.draw()");
}

TextBox.prototype.quote =
function () {
  var result;
  
  result = "TextBox{x:" + Quote(this.x) + ", y:" + Quote(this.y) 
         + ", width:" + Quote(this.width) 
         + ", height:" + Quote(this.height) 
         + ", text:" + Quote(this.text)
         + ", va:" + Quote(this.verticalAlign)
         + ", bg:" + FormatColor(this.backgroundColor)
         + ", fg:" + FormatColor(this.textColor) + "}";
  
  return result;
}

TextBox.prototype.moveDown =
function (pixels) {
  //DebugCall("TextBox.moveDown", [pixels]);

  var new_offset = this.offset - pixels;
  this.setOffset(new_offset);
}

TextBox.prototype.moveToBottom =
function () {
  //DebugCall("TextBox.moveToBottom");

  var bottom_offset = this.height - this.textHeight;
  this.setOffset(bottom_offset);
}

TextBox.prototype.moveToTop =
function () {
  //DebugCall("TextBox.moveToTop");
  
  this.setOffset(0);
}

TextBox.prototype.setFont =
function (font) {
  //DebugCall("TextBox.setFont", [font]);

  if (font == undefined) {
    font = DefaultFont;
  }
  this.font = font;
  
  var text = this.text;
  if (text == undefined) {
    text = "";
  }
  
  this.textHeight = font.getStringHeight(text, this.width);
}

TextBox.prototype.setOffset =
function (pixels) {
  //DebugCall("TextBox.setOffset", [pixels]);
  
  var bottomOffset = this.interior.height - this.textHeight;
  if (pixels < bottomOffset) {
    pixels = this.interior.height - this.textHeight;
  }
  if (pixels > 0) {
    pixels = 0;
  }
  this.offset = pixels; 
}

TextBox.prototype.setText =
function (text) {
  //DebugCall("TextBox.setText", [text]);

  if (text == undefined) {
    text = "";
  }

  this.text = text;
  this.textHeight = this.font.getStringHeight(text, this.interior.width);
}

TextBox.prototype.setTextColor =
function (color) {
  //DebugCall("TextBox.setTextColor", [color]);
  
  this.textColor = color;
}

TextBox.prototype.write =
function (text) {
  //DebugCall("TextBox.write", [text]);

  if (typeof(text) != "string") {
    Abort("No text in TextBox.write");
  }
  this.text += text + "\n";
  this.textHeight = this.font.getStringHeight(this.text, this.interior.width);
  this.draw();
  FlipScreen();
}
