// screen_rect.js
// ScreenRect class for Sphere

/*  Copyright (C) 2008  Stephen R. Gold

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/

// A ScreenRect represents a rectangular region of the screen.

// global instance

// a ScreenRect for the entire screen (or game window, if running windowed)
Screen = new ScreenRect(0, 0, GetScreenWidth(), GetScreenHeight());    

RequireScript("utilities/debug.js");

// constructor
function ScreenRect(screen_x, screen_y, width, height) {
  if (this instanceof ScreenRect == false) {
    return new ScreenRect(screen_x, screen_y, width, height);
  }
  //DebugCall("ScreenRect", [screen_x, screen_y, width, height]);
    
  if (screen_x == undefined) {
    // initialize to graphics clipping window
    var clip = GetClippingRectangle();
    this.x = clip.x;
    this.y = clip.y;
    this.width = clip.width;
    this.height = clip.height;

  } else {
    //initialize to specified geometry
    if (screen_y == undefined || width == undefined || height == undefined) {
      Abort("Invalid argment(s) in ScreenRect()");
    }
    this.x = screen_x;
    this.y = screen_y;
    this.width = width;
    this.height = height;
  }
}

// create a new ScreenRect by carving off a chunk from one edge

ScreenRect.prototype.carve =
function(edge, min_width, min_height) {
  //DebugCall("ScreenRect.carve", [edge, min_width, min_height]);

  if (!(this.width >= min_width)) {
    Abort("Unable to carve ScreenRect; this.width=" + Quote(this.width) + ", min_width=" + Quote(min_width));
  }
  if (!(this.height >= min_height)) {
    Abort("Unable to carve ScreenRect; this.height=" + Quote(this.height) + ", min_height=" + Quote(min_height));
  }
  
  var x, y, width, height;
  
  switch (edge) {
    case "left":
      width = min_width;
      height = this.height;
      x = this.x;
      y = this.y;
      this.x += width;
      this.width -= width;
      break;
      
    case "right":
      width = min_width;
      height = this.height;
      x = this.x + this.width - width;
      y = this.y;
      this.width -= width;
      break;
    
    case "top":
      width = this.width;
      height = min_height;
      x = this.x;
      y = this.y;
      this.y += height;
      this.height -= height;
      break;

    case "bottom":
      width = this.width;
      height = min_height;
      x = this.x;
      y = this.y + this.height - height;
      this.height -= height;
      break;
      
    default:
      Abort("Invalid edge in ScreenRect.carve(): " + Quote(edge));
  }

  var new_rect = new ScreenRect(x, y, width, height);

  return new_rect;
}

ScreenRect.prototype.centerBlit =
function (image) {
  //DebugCall("ScreenRect.centerBlit", [image]);
  
  var x = this.x + (this.width - image.width)/2;
  var y = this.y + (this.height - image.height)/2;

  image.blit(x, y);
}


// set graphics clipping window based on the ScreenRect
ScreenRect.prototype.clip = 
function () {
  //DebugCall("ScreenRect.clip");

  SetClippingRectangle(this.x, this.y, this.width, this.height);  
}

// test whether pixel (screen_x, screen_y) is in the ScreenRect
ScreenRect.prototype.contains = 
function (screen_x, screen_y) {
  //DebugCall("ScreenRect.contains", [screen_x, screen_y]);
  
  screen_x -= this.x;
  screen_y -= this.y;
  var result = ((screen_x >= 0) && (screen_x < this.width) 
             && (screen_y >= 0) && (screen_y < this.height));
  
  return result;
}

// create a new ScreenRect by cloning
ScreenRect.prototype.clone =
function () {
  //DebugCall("ScreenRect.clone");

  var new_rect = new ScreenRect(this.x, this.y, this.width, this.height);
    
  return new_rect;
}

// copy another ScreenRect
ScreenRect.prototype.copyRect =
function (rect) {
  //DebugCall("ScreenRect.copyRect");

  this.x = rect.x;
  this.y = rect.y;
  this.width = rect.width;
  this.height = rect.height;
}

// draw a text label below the ScreenRect
ScreenRect.prototype.drawLabel =
function (text) {
  //DebugCall("ScreenRect.drawLabel", [text]);
  if (text == undefined) {
    return;
  }
  
  var font = GetSystemFont();
  var w = font.getStringWidth(text);
  var x = this.x + (this.width - w)/2;
  var y = this.y + this.height;
  font.drawText(x, y, text);
}

// fill the ScreenRect with a color
ScreenRect.prototype.fill =
function (color) {
  //DebugCall("ScreenRect.fill", [color]);
  //DebugLog.write(" this=" + Quote(this));
  
  if (typeof(color) != "object") {
    Abort("Invalid color in ScreenRect.fill(): " + Quote(color));
  }
  
  Rectangle(this.x, this.y, this.width, this.height, color);
}

// fill the ScreenRect with a disc of color
ScreenRect.prototype.fillCircle =
function (color) {
  //DebugCall("ScreenRect.fillCircle", [color]);
  //DebugLog.write(" this=" + Quote(this));
  
  if (typeof(color) != "object") {
    Abort("Invalid color in ScreenRect.fillCircle(): " + Quote(color));
  }
  
  var x = this.x + this.width/2;
  var y = this.y + this.height/2;
  var r = Math.min(this.width, this.height)/2;
  FilledCircle(x, y, r, color);
}

ScreenRect.prototype.grab =
function () {
  //DebugCall("ScreenRect.grab");

  return GrabImage(this.x, this.y, this.width, this.height);
}

// outline the ScreenRect with a color
ScreenRect.prototype.outline =
function (color) {
  //DebugCall("ScreenRect.outline", [color]);
  
  OutlinedRectangle(this.x, this.y, this.width, this.height, color);
}

// create a new overlay ScreenRect

ScreenRect.prototype.overlay =
function (x_frac, y_frac, w_frac, h_frac) {
  //DebugCall("ScreenRect.overlay", [x_frac, y_frac, w_frac, h_frac]);

  var x = Math.round(this.x + x_frac * this.width);
  var y = Math.round(this.y + y_frac * this.height);
  var width = Math.round(w_frac * this.width);
  var height = Math.round(h_frac * this.height);

  // clip
  if (x < this.x) {
    width -= (this.x - x);
    x = this.x;
  }
  if (y < this.y) {
    height -= (this.y - y);
    y = this.y;
  }
  if (x + width > this.x + this.width) {
    width = this.x + this.width - x;
  }
  if (y + height > this.y + this.height) {
    height = this.y + this.height - y;
  }
  
  var new_rect = new ScreenRect(x, y, width, height);  
  return new_rect;
}

ScreenRect.prototype.quote =
function () {
  var result;
  
  result = "ScreenRect{x:" + Quote(this.x) 
         + ", y:" + Quote(this.y) 
         + ", width:" + Quote(this.width) 
         + ", height:" + Quote(this.height) + "}";
  
  return result;
}

// create a new ScreenRect by trimming off a border from each edge

ScreenRect.prototype.trim =
function(thickness) {
  //DebugCall("ScreenRect.trim", [thickness]);

  var y = this.y + thickness;
  var height = this.height - 2*thickness;
  
  var x, width;
  if (this.width == 0) {
    x = this.x;
    width = 0;
  } else {
    x = this.x + thickness;
    width = this.width - 2*thickness;
  }
  
  var new_rect = new ScreenRect(x, y, width, height);
    
  return new_rect; 
}

// blit image to middle of upper edge
ScreenRect.prototype.topBlit =
function (image) {
  //DebugCall("ScreenRect.topBlit", [image]);
 
  var x = this.x + (this.width - image.width)/2;
  var y = this.y - image.height/2;
  
  image.blit(x, y);
}

// blit image to upper left-hand corner
ScreenRect.prototype.ulcBlit =
function (image) {
  //DebugCall("ScreenRect.ulcBlit", [image]);
 
  var x = this.x - image.width/2;
  var y = this.y - image.height/2;
  
  image.blit(x, y);
}

// blit image to upper right-hand corner
ScreenRect.prototype.urcBlit =
function (image) {
  //DebugCall("ScreenRect.urcBlit", [image]);
 
  var x = this.x + this.width - image.width/2;
  var y = this.y - image.height/2;
  
  image.blit(x, y);
}