// scaled_font.js
// ScaledFont class for Sphere

/*  Copyright (C) 2008  Stephen R. Gold

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/

// A ScaledFont is a Sphere font with built-in scaling.

RequireScript("utilities/graphics.js");

// constructor
function ScaledFont(min_height, base_font) {
  if (this instanceof ScaledFont == false) {
    return new ScaledFont(min_height, base_font);
  }
  //DebugCall("ScaledFont", [min_height, base_font]);
    
  if (base_font == undefined) {
    base_font = DefaultFont;
  } else if (typeof(base_font) == "string") {
    base_font = LoadFont(base_font + ".rfn");
  } else if (typeof(base_font) != "object") {
    Abort("Invalid base_font in ScaledFont()");
  }

  var scale;
  if (min_height == undefined) {
    scale = 1;
  } else if (typeof(min_height) == "number" && min_height > 0) {
    scale = min_height / base_font.getHeight();
  } else {
    Abort("Invalid min_height in ScaledFont(): " + Quote(min_height));
  }
  
  this.baseFont = base_font;
  this.scale = scale;
  
  //DebugLog.write(" scale=" + Quote(this.scale));
}

ScaledFont.prototype.clone = 
function () {
  //DebugCall("ScaledFont.clone");

  var base_font = this.baseFont.clone();
  var min_height = this.getHeight();
  
  return new ScaledFont(min_height, base_font);
}

ScaledFont.prototype.drawText = 
function (x, y, text) {
  //DebugCall("ScaledFont.drawText", [x, y, text]);

  var scale = this.scale;  
  var result = this.baseFont.drawZoomedText(x, y, scale, text);

  return result;
}

ScaledFont.prototype.drawTextBox =
function (x, y, width, height, offset, text) {
  //DebugCall("ScaledFont.drawTextBox", [x, y, width, height, offset, "text"]);
  
  if (this.lastBox == undefined
   || this.lastBox.width != width
   || this.lastBox.text != text) {
    //DebugLog.write(" can't use cached surface -- create new one");
   
    this.lastBox = new Object;
    this.lastBox.width = width;
    this.lastBox.text = text;

    var scaled_width = width/this.scale;
	  var h = this.baseFont.getStringHeight(text, scaled_width);
    var transparent = CreateColor(0, 0, 0, 0);
    var surface = CreateSurface(scaled_width, h, transparent);
	  surface.drawTextBox(this.baseFont, 0, 0, scaled_width, h, 0, text);
    surface.rescale(width, h * this.scale);
    this.lastBox.surface = surface;
  }

  var old_clip = GetClippingRectangle();
  SetClippingRectangle(x, y, width, height);
  this.lastBox.surface.blit(x, y + offset);
  SetClippingRectangle(old_clip.x, old_clip.y, old_clip.width, old_clip.height);
}

ScaledFont.prototype.fitString =
function (string, width) {
  //DebugCall("ScaledFont.fitString", [string, width]);
  
  var ww = this.getStringWidth(string);
  if (ww > width) {
    var new_scale = this.scale * width/ww;
    this.setScale(new_scale); 
  }
  //DebugLog.write(" new scale=" + Quote(this.scale));
}

ScaledFont.prototype.getColorMask = 
function () {
  //DebugCall("ScaledFont.getColorMask");

  var result = this.baseFont.getColorMask();
  
  return result;
}

ScaledFont.prototype.getHeight = 
function () {
  //DebugCall("ScaledFont.getHeight");

  var height = this.baseFont.getHeight();
  var result = height * this.scale;
  
  return result;
}

ScaledFont.prototype.getStringHeight =
function (string, width) {
  //DebugCall("ScaledFont.getStringHeight", [string, width]);

  var scaled_width = width/this.scale;
  var height = this.baseFont.getStringHeight(string, scaled_width);
  var result = height * this.scale;
  
  return result;
}
 
ScaledFont.prototype.getStringWidth = 
function (string) {
  //DebugCall("ScaledFont.getStringWidth", [string]);

  var width = this.baseFont.getStringWidth(string);
  var result = width * this.scale;
  
  return result;
}

ScaledFont.prototype.quote = 
function () {
  var result = "ScaledFont{scale:" + Quote(this.scale) + "}";
  
  return result;
}

ScaledFont.prototype.setColorMask = 
function (color) {
  //DebugCall("ScaledFont.setColorMask", [color]);

  var result = this.baseFont.setColorMask(color);
  
  return result;
}

ScaledFont.prototype.setScale = 
function (new_scale) {
  //DebugCall("ScaledFont.setScale", [new_scale]);

  if (new_scale == undefined || new_scale <= 0) {
    Abort("Invalid new_scale in ScaledFont.setScale(): " + Quote(new_scale));
  }
  
  this.scale = new_scale;
  this.lastBox = undefined;
}