// menu.js
// Menu class for Sphere

/*  Copyright (C) 2008  Stephen R. Gold

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
    
// A Menu is a simple scrolling menu in a ScreenRect.

// typical usage:
//
//RequireScript("library/menu.js");
//...
//  var menu = new Menu(window, font);
//  
//  menu.update = function() {
//
//    // draw rest of screen
//    Screen.clip();
//    Screen.fill(Black);
//
//    // return list of choices
//    return [ "1 large", "2 small" ];
//  };
//  
//  menu.bindKey(KEY_1, "ActiveMenu.index = 0"); // shortcut
//  menu.bindKey(KEY_2, "ActiveMenu.index = 1"); // shortcut
//
//  var choice = menu.execute(); 

RequireScript("ui/mouse.js");
RequireScript("ui/text_line.js");

ActiveMenu = undefined;

// constructor

function Menu(rect, font) {
  if (this instanceof Menu == false) {
    return new Menu(rect, font);
  }
  //DebugCall("Menu", [rect, font]);
  
  if (!(rect instanceof ScreenRect)) {
    Abort("Invalid rect in Menu()");
  }
  if (!(font instanceof ScaledFont)) {
    Abort("Invalid font in Menu()");
  }
  
  this.font = font;
  var line_height = font.getHeight();

  // default colors
  this.normalTextColor = CreateColor(128, 128, 128);     // grey
  this.normalBackgroundColor = CreateColor(40, 0, 0);    // dark red
  this.selectionTextColor = CreateColor(255, 255, 255);  // white
  this.selectionBackgroundColor = CreateColor(0, 64, 0); // medium green
 
  // make a ScreenRect for each line in the menu area
  this.textLines = [];
  var copy_rect = rect.clone();
  while (line_height <= copy_rect.height) {
    // carve out another line
    var line = copy_rect.carve("top", 0, line_height);
    //DebugLog.write("line=" + Quote(line));
    
    var text_line = new TextLine(line, undefined, undefined, "", "left");
    this.textLines.push(text_line);
    ActiveWindows.push(text_line);
  }

  // assign actions to keys that are pressed
  this.keyAction = [];
  this.bindKey(KEY_UP, "ActiveMenu.moveDown(-1)");
  this.bindKey(KEY_DOWN, "ActiveMenu.moveDown(1)");
  this.bindKey(KEY_PAGEUP, "ActiveMenu.pageDown(-1)");
  this.bindKey(KEY_PAGEDOWN, "ActiveMenu.pageDown(1)");  
  this.bindKey(KEY_ENTER, "ActiveMenu.returnFlag = true");
  this.bindKey(KEY_ESCAPE, "ActiveMenu.returnFlag = true");    

  //DebugLog.write("Menu() returns " + Quote(this));
}

Menu.prototype.bindKey =
function (key, action) {
  //DebugCall("Menu.bindKey", [key, action]);

  this.keyAction[key] = action;
}

Menu.prototype.drawLine = 
function (line) {
  //DebugCall("Menu.drawLine", [line]);

  if (!(line instanceof TextLine)) {
    Abort("Invalid line in Menu.drawLine(): " + Quote(line));
  }

  if (line.choice == this.choice) {
    line.draw(line.choice, 
      this.selectionTextColor, this.selectionBackgroundColor);    
  } else {
    line.draw(line.choice, 
      this.normalTextColor, this.normalBackgroundColor);    
  }
  
  //DebugLog.write("Menu.drawLine() returns");
}

// Returns the selected choice and sets this.lastKey to 
//    the key which caused the function to return, such 
//    as KEY_ENTER or KEY_ESCAPE).

Menu.prototype.execute =
function (choice) {
  //DebugCall("Menu.execute", [choice]);
  
  var save_menu = ActiveMenu;
  ActiveMenu = this;
  
  var text_lines = this.textLines
  var num_lines = text_lines.length;

  this.choice = choice;
  this.lastKey = undefined;
  this.returnFlag = false;

  while (!this.returnFlag) {

    // update the rest of screen and get list of choices
    var choices = this.update();
    this.choices = choices;

    // update current choice and its index
    this.index = 0;
    for (var i in choices) {
      if (choices[i] == this.choice) {
        this.index = Number(i);
        break;
      }
    }
    this.choice = choices[this.index];
    
    // adjust top so that selection is near the middle of the window
    this.topIndex = this.index - Math.round(num_lines/2);
    if (this.topIndex + num_lines > choices.length) {
      this.topIndex = choices.length - num_lines;
    }
    if (this.topIndex < 0) {
      this.topIndex = 0;
    }
    
    // display some choices
    var next_line = 0;
    for (var i in choices) {
      var choice = choices[i];
    
      if (i >= this.topIndex) {
        var text_line = text_lines[next_line];
        next_line++;
        if (text_line != undefined) {
          // the choice is visible
          text_line.choice = choice;
          text_line.click = MenuLineClick;
          this.drawLine(text_line);
        }
      }
    }
    
    for (var i = next_line; i < num_lines; i++) {
      var text_line = text_lines[i];
      text_line.click = EatClick;
    }
    
    UpdateMouse(DefaultPointerImage);
    FlipScreen();
    
    while (AreKeysLeft()) {
      var key = GetKey();
      eval(this.keyAction[key]);
      this.lastKey = key;
    }
  }
  
  ActiveMenu = save_menu;

  //DebugLog.write(" Menu.execute() returns " + this.choice); 
  return this.choice;
}

// move selection down (or up)
Menu.prototype.moveDown =
function (number_of_lines) {
  //DebugCall("Menu.moveDown", [number_of_lines]);

  var choices = this.choices;
  
  this.index += number_of_lines;
  if (this.index >= choices.length) {
    this.index = choices.length - 1;
  }
  if (this.index < 0) {
    this.index = 0;
  }
  //DebugLog.write("new index = " + Quote(this.index));
  
  this.choice = choices[this.index];
}

Menu.prototype.pageDown =
function (number_of_pages) {
  //DebugCall("Menu.pageDown", [number_of_pages]);

  var page_size = Math.ceil(Math.sqrt(this.choices.length));
  var number_of_lines = number_of_pages * page_size;
  
  this.moveDown(number_of_lines);  
}

function MenuLineClick(button, screen_x, screen_y) {
  //DebugCall("MenuLineClick", [button, screen_x, screen_y]);

  if (this.choice != undefined) {
    ActiveMenu.choice = this.choice;
  }
}

function MenuEnter() {
  var action = ActiveMenu.keyAction[KEY_ENTER];
  eval(action);
  ActiveMenu.lastKey = KEY_ENTER;
}

function MenuEscape() {
  var action = ActiveMenu.keyAction[KEY_ESCAPE];
  eval(action);
  ActiveMenu.lastKey = KEY_ESCAPE;
}