// icon_grid.js
// IconGrid class for Sphere

/*  Copyright (C) 2008  Stephen R. Gold

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/

// An IconGrid is a Window that displays a rectangular grid of Icons.

RequireScript("ui/icon.js");
RequireScript("ui/window.js");

// constructor

IconGrid.prototype = new Window();
IconGrid.prototype.constructor = IconGrid;

// Icons are normally topleft-aligned in the IconGrid.
// Specify negative num_icons to center the icons.

function IconGrid(parent, edge, num_icons, bg_color, icon_width, icon_height, icon_border) {
  if (this instanceof IconGrid == false) {
    return new IconGrid(parent, edge, num_icons, bg_color, icon_width, icon_height, icon_border);
  }
  //DebugCall("IconGrid", [parent, edge, num_icons, bg_color, icon_width, icon_height, icon_border]);

  if (parent == undefined) {
    parent = Screen.clone();
  }
  if (bg_color == undefined) {  
    bg_color = CreateColor(30, 30, 30); // dark grey
  }
  if (icon_width == undefined) {  
    icon_width = Math.floor(Screen.width/12) - 10; // in pixels
  }
  if (icon_height == undefined) {
    icon_height = icon_width;
  }
  if (icon_border == undefined) {
    icon_border = 5; // in pixels
  }

  var padded_height = icon_height + 2*icon_border;
  var padded_width = icon_width + 2*icon_border;
  
  var h_center_flag = false;
  var v_center_flag = false;
  
  var icon_rows, icon_columns;
  if (edge == "right") {
    icon_rows = Math.floor(parent.height / padded_height);
    if (num_icons < 0) {      
      v_center_flag = true;
      num_icons = -num_icons;
    }
    if (icon_rows > num_icons) {
      icon_rows = num_icons;
    }
    icon_columns = Math.ceil(num_icons / icon_rows);
    
    // even up the columns
    icon_rows = Math.ceil(num_icons / icon_columns);

  } else if (edge == "bottom") {
    icon_columns = Math.floor(parent.width / padded_width);
    if (num_icons < 0) {      
      h_center_flag = true;
      num_icons = -num_icons;
    }
    if (icon_columns > num_icons) {
      icon_columns = num_icons;
    }
    icon_rows = Math.ceil(num_icons / icon_columns);
    
    // even up the rows
    icon_columns = Math.ceil(num_icons / icon_rows);
    
  } else {
    Abort("Invalid edge in IconGrid(): " + Quote(edge));
  }
  
  var min_width = icon_columns * padded_width;
  var min_height = icon_rows * padded_height;  
 
  var rect = parent.carve(edge, min_width, min_height);

  // initialize properties
  Window.apply(this, [rect, bg_color]);

  var x0 = this.x + icon_border;
  if (h_center_flag) {
    x0 += (this.width - min_width)/2;
  }
  var y0 = this.y + icon_border;
  if (v_center_flag) {
    var h = icon_rows * padded_height;
    y0 += (this.height - min_height)/2;
  }
 
  var i = 0;
  this.icons = [];
  
  if (icon_columns > icon_rows) {
    for (var col = 0; col < icon_columns; col++) {
      var x = x0 + col*padded_width;
      for (var row = 0; row < icon_rows; row++) {
        var y = y0 + row*padded_height;
        if (i < num_icons) {
          var rect = new ScreenRect(x, y, icon_width, icon_height);
          this.icons[i] = new Icon(rect);
          i++;
        }
      }
    }
  } else {
    for (var row = 0; row < icon_rows; row++) {
      var y = y0 + row*padded_height;
      for (var col = 0; col < icon_columns; col++) {
        var x = x0 + col*padded_width;
        if (i < num_icons) {
          var rect = new ScreenRect(x, y, icon_width, icon_height);
          this.icons[i] = new Icon(rect);
          i++;
        }
      }
    }
  }
  
  this.numIcons = num_icons;  
}  

IconGrid.prototype.disable =
function() {
  //DebugCall("IconGrid.disable");

  for (var i in this.icons) {
    var icon = this.icons[i];
    if (icon != undefined) {
      icon.disable();
    }
  }
}

IconGrid.prototype.draw =
function() {
  //DebugCall("IconGrid.draw");

  var saved_clip = new ScreenRect();
  
  this.clip();  
  this.fill(this.backgroundColor);
  
  for (var i in this.icons) {
    var icon = this.icons[i];
    if (icon != undefined) {
      icon.draw();
    }
  }
  
  saved_clip.clip();
}