// roster.js
// Roster class for battle simulation (Sphere)

/*  Copyright (C) 2008-2009  Stephen R. Gold

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/

//  A Roster is a set of Entities indexed by name.

RequireScript("entities/entity.js");

// creation functions

function Roster(name) {
  if (this instanceof Roster == false) {
    return new Roster(name);
  }
  //DebugCall("Roster", [name]);

  this.members = new Object;
  this.name = name;
  
  return this;
}

Roster.prototype.add =
function(entity) {
  //DebugCall("Roster.add", [entity], Quote(this));

  if (entity instanceof Entity == false) {
    Abort("Invalid entity: " + Quote(entity));
  }
  var name = entity.name;
  if (this.members[name] != undefined) {
    Abort("Duplicate name: " + Quote(name));
  }
  this.members[name] = entity;
}

Roster.prototype.count =
function() {
  //DebugCall("Roster.count", [], Quote(this));
  
  var result = 0;
  
  for (var name in this.members) {
    if (this.members[name] != undefined) {
      result++;
    }
  }

  //DebugLog.write(" Roster.count() returns " + Quote(result));
  return result;
}

Roster.prototype.describe =
function() {
  //DebugCall("Roster.describe", [], Quote(this));

  var len = this.count();
  if (len == 0) {
    return "nobody";
  } else if (len == 1) {
    return this.getFirst().name;
  }
  
  var name_list = [];
  for (var name in this.members) {
    var entity = this.members[name];
    if (entity != undefined) {
      name_list.push(name);
    }
  }
  var last = name_list.pop();
  return name_list.join(", ") + " and " + last;
}

Roster.prototype.getFirst =
function() {
  //DebugCall("Roster.getFirst");
  
  for (var name in this.members) {
    if (this.members[name] != undefined) {
      return this.members[name];
    }
  }

  return undefined;
}

Roster.prototype.isEmpty =
function() {
  //DebugCall("Roster.isEmpty");
  
  for (var name in this.members) {
    if (this.members[name] != undefined) {
      return false;
    }
  }

  return true;
}

Roster.prototype.remove =
function(entity) {
  //DebugCall("Roster.remove", [entity], Quote(this));

  if (entity instanceof Entity == false) {
    Abort("Invalid entity in Roster.remove(): " + Quote(entity));
  }
  var name = entity.name;
  
  if (this.members[name] != entity) {
    Abort("Entity not found in Roster.remove(): " + Quote(entity));
  }
  delete this.members[name];
}

Roster.prototype.quote =
function() {
  var result = "Roster{name:" + Quote(this.name) + "}";
  return result;
}

// search functions

// find an Entity by name
Roster.prototype.find =
function(name) {
  //DebugCall("Roster.find", [name], Quote(this));
  
  if (name == undefined) {
    return undefined;
  }
  if (typeof(name) != "string") {
    Abort("Invalid name: " + Quote(name));
  }
  
  var entity = this.members[name];
  
  return entity;
}

// find the best entity based on dist2: 
//   squared screen distance from the frame center

Roster.prototype.best =
function(map_xy, best_entity) {
  //DebugCall("Roster.best", [map_xy, best_entity]);

  var best_dist2;
  if (best_entity != undefined) {
    best_dist2 = best_entity.dist2(map_xy);
  }
  
  for (var name in this.members) {
    var entity = this.members[name];
    if (entity != undefined) {
      var dist2 = entity.dist2(map_xy);
      if (best_dist2 == undefined || best_dist2 > dist2) {
        best_dist2 = dist2;
        best_entity = entity;
      }
    }
  }

  if (best_entity != undefined && best_dist2 > 1) {
    best_entity = undefined;
  }
  return best_entity;
}

// find the best crusher based on dist2: 
//   squared screen distance from the frame center

Roster.prototype.bestCrusher =
function(map_xy, best_crusher) {
  //DebugCall("Roster.bestCrusher", [map_xy, best_crusher]);

  var best_dist2;
  if (best_crusher != undefined) {
    best_dist2 = best_crusher.dist2(map_xy);
  }
  
  for (var name in this.members) {
    var entity = this.members[name];
    if (entity != undefined && entity.isCrusher()) {
      var dist2 = entity.dist2(map_xy);
      if (best_dist2 == undefined || best_dist2 > dist2) {
        best_dist2 = dist2;
        best_crusher = entity;
      }
    }
  }

  if (best_crusher != undefined && best_dist2 > 1) {
    best_crusher = undefined;
  }
  return best_crusher;
}

// find the best visible entity based on dist2: 
//    squared screen distance from the frame center

Roster.prototype.bestVisible =
function(map_xy, best_visible) {
  //DebugCall("Roster.bestVisible", [map_xy, best_visible], Quote(this));

  var best_dist2;
  if (best_visible != undefined) {
    best_dist2 = best_visible.dist2(map_xy);
  }
  
  for (var name in this.members) {
    var entity = this.members[name];
    if (entity != undefined && entity.isVisible()) {
      var dist2 = entity.dist2(map_xy);
      if (best_dist2 == undefined || best_dist2 > dist2) {
        best_dist2 = dist2;
        best_visible = entity;
      }
    }
  }

  if (best_visible != undefined && best_dist2 > 1) {
    best_visible = undefined;
  }

  //DebugLog.write(" Roster.bestVisible() returns " + Quote(best_visible));
  return best_visible;
}


// find the nearest entity based on cm2:
//   "squared physical" distance from the base center)

Roster.prototype.nearest =
function(es_cm, nearest_entity, min_cm2) {
  //DebugCall("Roster.nearest", [es_cm, nearest_entity, min_cm2]);
  
  if (min_cm2 == undefined) {
    // default to no distance threshold
    min_cm2 = 0;
  }

  var nearest_cm2;
  if (nearest_entity != undefined) {
    nearest_cm2 = nearest_entity.cm2(es_cm);
  }
  
  for (var name in this.members) {
    var entity = this.members[name];
    if (entity != undefined) {
      var cm2 = entity.cm2(es_cm);
      if ((nearest_cm2 == undefined || nearest_cm2 > cm2) && cm2 >= min_cm2) {
        nearest_cm2 = cm2;
        nearest_entity = entity;
      } 
    }
  }

  return nearest_entity;
}

Roster.prototype.nearestThawed =
function(es_cm, nearest_thawed) {
  //DebugCall("Roster.nearestThawed", [es_cm, nearest_thawed]);
  
  var nearest_cm2;
  if (nearest_thawed != undefined) {
    nearest_cm2 = nearest_thawed.cm2(es_cm);
  }
  
  for (var name in this.members) {
    var entity = this.members[name];
    if (entity != undefined && !entity.isFrozen()) {
      var cm2 = entity.cm2(es_cm);
      if (nearest_cm2 == undefined || nearest_cm2 > cm2) {
        nearest_cm2 = cm2;
        nearest_thawed = entity;
      }
    }
  }

  return nearest_thawed;
}