// clock.js
// clock routines for battle simulation (Sphere)

/*  Copyright (C) 2008  Stephen R. Gold

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/

// Three clocks are displayed as overlays on the map window:
//  TimeClock       local time of day 
//  SimClock        simulation time elapsed or remaining
//  RealClock       elapsed real-time since simulation started

// configuration

TimeClockFlag = PlayerFlag("TimeClock", true);
SimClockFlag = PlayerFlag("SimClock", true);
RealClockFlag = PlayerFlag("RealClock", false);

ShowTenthsFlag = PlayerFlag("ShowTenths", false); // affects all except TimeClock
CountdownFlag = Flag("Countdown", false);
CountdownSeconds = OptionalEnum("CountdownSeconds", 0, [0, 1, 30, 60, 120, 180, 300], "CountdownFlag");

// colors

TimeClockColor = CreateColor(190, 180, 230);      // violet
SimClockColor = CreateColor(255, 200, 64);        // egg yolk
RealClockColor = CreateColor(255, 255, 255);      // white

ClockBackgroundColor = CreateColor(0, 0, 0, 170); // shadow

// window styles

ClockStyle = LoadWindowStyle("rounded6.rws");
ClockStyle.thickness = 6;


RequireScript("ui/text_line.js");

// initialization

function CreateClocks(rects) {
  //DebugCall("CreateClocks", [rects]);

  TimeClock = new TextLine(rects[0], ClockBackgroundColor, 
    TimeClockColor, undefined, "left");
  TimeClock.setStyle(ClockStyle);
  
  SimClock = new TextLine(rects[1], ClockBackgroundColor, 
    SimClockColor, undefined, "left");
  SimClock.setStyle(ClockStyle);
  
  RealClock = new TextLine(rects[2], ClockBackgroundColor, 
    RealClockColor, undefined, "left");
  RealClock.setStyle(ClockStyle);
  
  // don't start clocks until first render is complete
  StartMs = undefined;
}

// rendering functions (to be called via SetRenderScript())

function DrawClocks() {
  //DebugCall("DrawClocks");

  if (TimeClockFlag) {
    DrawTimeClock();
  }
  if (SimClockFlag) {
    DrawSimClock();
  }
  if (RealClockFlag) {
    DrawRealClock();
  }
}

//  display local time of day 
//   format is "h:mm"
function DrawTimeClock() {
  //DebugCall("TimeClock");
  
  var now = new Date();

  var mm = String(now.getMinutes());
  if (mm.length == 1) {
    mm = "0" + mm;
  }
  var h = now.getHours(); // 24-hour notation

  // convert to 12-hour notation
  var ampm = "am";
  if (h >= 12) {
    ampm = "pm";
  }
  if (h < 1) {
    h += 12;
  } else if (h > 12) {
    h -= 12;
  }

  var text = String(h) + ":" + mm + ampm;
  
  TimeClock.draw(text);
}

//  display simulation time elapsed (or remaining, if CountdownFlag==true)
//    note: this clock shows simulated time, not real time
function DrawSimClock() {
  //DebugCall("DrawSimClock");

  var elapsed_msecs = Ticks*1000/TicksPerSecond;
  
  var msecs;
  if (CountdownFlag) {
    msecs = CountdownSeconds*1000 - elapsed_msecs;
  } else {
    msecs = elapsed_msecs;
  }
  
  var text = FormatTime(msecs);
  
  SimClock.draw(text);
}

//  display elapsed real-time since simulation started
function DrawRealClock() {
  //DebugCall("DrawRealClock");
  
  var msecs = GetTime() - StartMs;
  var text = FormatTime(msecs);

  RealClock.draw(text);
}


// Convert time (in milliseconds) to a string, for display.
//   format is "m:ss" (or "m:ss.s" if ShowTenthsFlag==true)
function FormatTime(msecs) {
  //DebugCall("FormatTime", [tenths]);

  var tenths = Math.floor(msecs/100);
  var seconds = Math.floor(tenths/10);
  var minutes = Math.floor(seconds/60);
  
  var tth = String(tenths - 10*seconds);
  var ss = String(seconds - 60*minutes);
  if (ss.length == 1) {
    ss = "0" + ss;
  }
  var m = String(minutes);   

  if (ShowTenthsFlag) {
    return m + ":" + ss + "." + tth;
  } else {
    return m + ":" + ss;
  }
}