// main.js
// Fantasy Battle Simulation, for Sphere

/*  Copyright (C) 2008-2009  Stephen R. Gold

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/

RequireScript("utilities/add_parms.js");

EditableScenarioFlag = Flag("EditableScenario", true);

// which teams the player can command
CommandBlueFlag = Flag("CommandBlue", true);
CommandRedFlag = Flag("CommandRed", true);

// map view
MapView = PlayerEnum("MapView", "wide", ["wide", "normal", "closeup", "extreme closeup"]);

// sim seconds per real second
Speedup = PlayerEnum("Speedup", 0.8, [0.1, 0.2, 0.3, 0.5, 0.8, 1, 1.5, 2]);

// max update rate, in updates per (simulated) second
TicksPerSecond = Enum("TicksPerSecond", 20, [5, 10, 15, 20, 25, 30, 40, 50]);

AllMaps = ["field", "river", "snow", "tiny"];

// name of map
Map = Enum("Map", "field", AllMaps);

CheckForVictoryFlag = Flag("CheckForVictory", true);

RequireScript("bind.js");
RequireScript("coords.js");
RequireScript("draw.js");
RequireScript("get_scenario.js");
RequireScript("layout.js");
RequireScript("team.js");
RequireScript("utilities/random.js");
RequireScript("zoom.js");

CreateEntitiesFlag = true;  // tells EnterMap() whether to create entities

function game() {
  // start here

  RequireVersion(1.13);
  RequireWindowsize(800, 600);
  
  Randomize();
  
  GetScenario();

  if (BriefingFlag) {
    GiveBriefing();
  }
  
  CacheAllSoundFx();
  InitDraw();
  BindGameIcons();
  
  GroundLayer = 0;
  ObstacleLayer = 1;
  MissileLayer = 2;

  DoneFunction = Exit;
  BasesFile = OpenFile("bases");

  SetDefaultMapScript(SCRIPT_ON_ENTER_MAP, "EnterMap()");

  var map_file = String(ResolutionCm) + "cm/" + Map + ".rmp";
  MapEngine(map_file, TicksPerSecond*Speedup);
     // Control should pass to EnterMap() below
     // and return here after the simulation ends.

  Screen.fill(CreateColor(160, 160, 160));
  var im = LoadScaledImage("'Please wait ...", 400, 200);
  Screen.centerBlit(im);
  FlipScreen();
     
  Cleanup();
  
  DoneFunction();
  GarbageCollect();
  RestartGame();
}

// verify minimum Sphere version
function RequireVersion(min_version) {
  //DebugCall("RequireVersion", [min_version]);

  var version = GetVersion();
  
  if (version < min_version) {
    Abort("Running Sphere version " + Quote(version) + ". " 
    + "This game requires version " + Quote(min_version) + " or higher.");
  }
}

// verify minimum window size
function RequireWindowsize(min_width, min_height) {
  //DebugCall("RequireWindowsize", [min_width, min_height]);

  if (Screen.width < min_width || Screen.height < min_height) {
    Abort("Running in a " + Quote(Screen.width) + "x" + Quote(Screen.height) + " window. " 
  + "This game requires " + Quote(min_width) + "x" + Quote(min_height)+ " or larger.");
  }
}

function GetImageBaseX(image_name) {
  //DebugCall("GetImageBaseX", [image_name]);
  
  var x = BasesFile.read(image_name + ":x", -1);
  if (x == -1) {
    Abort("Invalid image_name in GetImageBaseX(): " + Quote(image_name));
  }
  return x;
}

function GetImageBaseY(image_name) {
  //DebugCall("GetImageBaseY", [image_name]);
  
  var y = BasesFile.read(image_name + ":y", -1);
  if (y == -1) {
    Abort("Invalid image_name in GetImageBaseY(): " + Quote(image_name));
  }
  return y;
}

// finish initialization after the map engine has started
function EnterMap() {
  //DebugLog.write("EnterMap() MapName=" + Quote(MapName) + " CreatePeopleFlag=" + Quote(CreatePeopleFlag));
  
  var default_layer = 0;
  var viewing_angle_degrees = 60;
  InitCoords(default_layer, viewing_angle_degrees);
  
  if (CreateEntitiesFlag) {
    CreateEntities();
  }
  
  // center the map
  CenterMap();
  
  // decide whether to fill the map window
  if (MapWindow.height > GetLayerHeight(Layer)*GetTileHeight()
   || MapWindow.width > GetLayerWidth(Layer)*GetTileWidth()) {
    FillMapWindowFlag = true;
  } else {
    FillMapWindowFlag = false;
  }
  //DebugLog.write("FillMapWindowFlag = " + Quote(FillMapWindowFlag));
  
  // call every tick after all the people are updated
  SetUpdateScript("SimUpdate()")

  // call every frame after ground layer is rendered:
  SetLayerRenderer(GroundLayer, "DrawGroundOverlays()"); 
                      
  // call every frame after obstacle layer is rendered:
  SetLayerRenderer(ObstacleLayer, "DrawObstacleOverlays()"); 

  // call every frame after missile layer is rendered:
  SetLayerRenderer(MissileLayer, "DrawMissileOverlays()"); 

  // call every frame after all layers are rendered
  SetRenderScript("DrawAll()"); 
  
  //DebugLog.write("-->leaving EnterMap()");
}

function CheckForVictory() {
  //DebugCall("CheckForVictory");
  
  if (!CheckForVictoryFlag) {
    return;
  }
  
  // check for "total victory"
  var blue_is_defeated = Blue.isDefeated();
  var red_is_defeated = Red.isDefeated();
  if (blue_is_defeated || red_is_defeated) {
    if (blue_is_defeated && red_is_defeated) {
      GameOver("Inconclusive battle.");
    } else if (blue_is_defeated) {
      if (CommandBlueFlag && !CommandRedFlag) {
        GameOver("You lost the battle.");
      } else if (!CommandBlueFlag && CommandRedFlag) {
        GameOver("You won the battle.");
      } else {
        GameOver("Red won the battle.");
      }
    } else if (red_is_defeated) {
      if (CommandBlueFlag && !CommandRedFlag) {
        GameOver("You won the battle.");
      } else if (!CommandBlueFlag && CommandRedFlag) {
        GameOver("You lost the battle.");
      } else {
        GameOver("Blue won the battle.");
      }
    }
  }
}

function Cleanup() {
  // work around for Restart() bug
  
  AllBanners = undefined;
  AllEntities = undefined;
  AllObstacles = undefined;
  
  Blue.allPortals = undefined;
  Blue.upCharacters = undefined;
  Blue = undefined;
  
  Red.allPortals = undefined;
  Red.upCharacters = undefined;
  Red = undefined;
  
  ImageCache = undefined;
  SoundFxCache = undefined;
  Zaps = undefined;
  
  GarbageCollect();
}

function GameOver(message) {
  //DebugCall("GameOver", [message]);
 
  PauseMessage = message;

  // pause without any possibility of resuming
  PauseFlag = true; 
  ResumeIcon.disable();    // disable the resume icon 

  // disable all commander and god icons
  if (CommandGrid != undefined) {
    CommandGrid.disable();
  }
  if (GodPowersGrid != undefined) {
    GodPowersGrid.disable();
  }
  
  // The MapEngine continues, but the simulation is 
  // permanently paused, waiting for the player to 
  // press Q or Esc or click on a restart/menu/quit icon.
  // Any of these actions will invoke the Quit() functions below.
}

function QuitToMenu() {
  //DebugCall("QuitToMenu");

  DoneFunction = DoneRestartToMenu;
  Quit();
}

function QuitToRestart() {
  //DebugCall("QuitToRestart");
  
  DoneFunction = DoneRestartSameScenario;
  Quit();
}

function Quit() {
  //DebugCall("Quit");
  
  SetUpdateScript("");
  SetRenderScript("");

  ExitMapEngine();
    // After the map engine exits, control should 
    // eventually pass to Done(), which was configured above.
}

function DoneRestartToMenu() {
  //DebugCall("DoneRestartToMenu");
  
  var file = OpenFile("auto");
  file.write("LoadScenarioFlag", false);
  file.close();
}

function DoneRestartSameScenario() {
  //DebugCall("DoneRestartSameScenario");
  
  var file = OpenFile("auto");
  file.write("LoadScenarioFlag", true);
  file.write("SkipBriefingFlag", true);
  file.write("SkipEditingFlag", true);
  file.close();
}