// locus.js
// locus routines for battle simulation (Sphere)

/*  Copyright (C) 2008-2009  Stephen R. Gold

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
    
// A "locus" is a string describing a "person", a physical location,
//   or a physical offset from a person.
// Places serve as a facepoints, goals, or targets for characters.

RequireScript("entities/entity.js");

LocusDefaultThicknessCm = 2*30;
LocusDivider = " ";

// define an attracting locus centered on a base person
function AttractingLocus(base, tolerance_cm) {
  //DebugCall("AttractingLocus", [base, tolerance_cm]);

  if (base instanceof Entity) {
    base = base.name;
  }
  if (typeof(base) != "string") {
    Abort("Invalid base: " + Quote(base));
  }

  if (tolerance_cm == undefined) {
    tolerance_cm = 0;
  } else {
    tolerance_cm = Math.ceil(tolerance_cm);
  }
  
  var words = [base, "0", "0", "0", String(tolerance_cm)];
  var locus = words.join(LocusDivider);
  
  return locus;
}

// define a locus around a fixed point on the map
function FixedLocus(es_cm, inner_cm, outer_cm) {
  //DebugCall("FixedLocus", [es_cm, inner_cm, outer_cm]);

  if (typeof(es_cm[0]) != "number" || typeof(es_cm[1]) != "number") {
    Abort("Invalid es_cm: " + QuoteList(es_cm));
  }

  return OffsetLocus("origin", es_cm, inner_cm, outer_cm);
}

// define a locus relative to a base person
function OffsetLocus(base, es_cm, inner_cm, outer_cm) {
  //DebugCall("OffsetLocus", [base, es_cm, inner_cm, outer_cm]);

  if (base instanceof Entity) {
    base = base.name;
  }
  if (es_cm == undefined) {
    es_cm = [0, 0];
  }
  if (inner_cm == undefined) {
    inner_cm = 0;
  } else {
    inner_cm = Math.ceiling(inner_cm);
  }
  if (outer_cm == undefined) {
    outer_cm = inner_cm + LocusDefaultThicknessCm;
  } else {
    outer_cm = Math.ceiling(outer_cm);
  }
  
  if (typeof(base) != "string") {
    Abort("Invalid base: " + Quote(base));
  }
  if (es_cm instanceof Array == false
   || typeof(es_cm[0]) != "number" 
   || typeof(es_cm[1]) != "number") {
    Abort("Invalid es_cm: " + QuoteList(es_cm));
  }
  if (typeof(inner_cm) != "number" || inner_cm < 0) {
    Abort("Invalid inner_cm: " + Quote(inner_cm));
  }
  if (typeof(outer_cm) != "number" || outer_cm < inner_cm) {
    Abort("Invalid outer_cm: " + Quote(outer_cm));
  }

  var east_mm = Math.round(10*es_cm[0]);
  var south_mm = Math.round(10*es_cm[1]);
  
  var words = [base, String(east_mm), String(south_mm), String(inner_cm), String(outer_cm)];
  var locus = words.join(LocusDivider);
  
  return locus;
}

// define a repelling locus centered on a person
function RepelLocus(base, range_cm) {
  //DebugCall(RepelLocus)
  
  if (typeof(base) != "string") {
    Abort("Invalid base: " + Quote(base));
  }

  if (range_cm == undefined) {
    range_cm = 5e5 - LocusDefaultThicknessCm;
  }
  if (range_cm <= 0) {
    Abort("Invalid range_cm: " + Quote(range_cm));
  }
  
  var words = [base, "0", "0", String(range_cm), "5e5"];
  var locus = words.join(LocusDivider);
  
  return locus;
}


// test whether a character is repelled by (too close to) a locus
function IsRepelled(character, locus) {
  //DebugCall("IsRepelled", [character, locus]);
  
  if (typeof(character) != "string" || !IsCharacter(character)) {
    Abort("Invalid character: " + Quote(character));
  }
  if (typeof(locus) != "string") {
    Abort("Invalid locus: " + Quote(locus));
  }
  
  var words = locus.split(LocusDivider);
  if (words[3] == undefined || words[3] == "" || words[3] == "0") {
    // a non-repelling locus
    return false;
  }

  var es_cm = OffsetEsCm(character, locus);
  var dist_cm = NormV(es_cm);
  var inner_cm = Number(words[3]);
  if (dist_cm < inner_cm) {
    return true;
  } else {
    return false;
  }
}

// get the base of a locus
function LocusBase(locus) {
  //DebugCall("LocusBase", [locus]);
  
  if (typeof(locus) != "string") {
    Abort("Invalid locus: " + Quote(locus));
  }
  
  var words = locus.split(LocusDivider);
  var base = words[0];
  
  return base;
}

// find the direction and distance of the nearest point in a locus
function LocusNearestDD(character, locus) {
  //DebugCall("LocusNearestDD", [character, locus]);
  
  if (typeof(character) == "string") {
    character = AllEntities.find(character);
  }
  if (character instanceof Entity == false) {
    Abort("Invalid character: " + Quote(character));
  }
  if (typeof(locus) != "string") {
    Abort("Invalid locus: " + Quote(locus));
  }
   
  var center_es_cm = character.offsetEsCm(locus);
  var center_cm = NormV(center_es_cm);

  var io = LocusRadiiCm(locus);
  var inner_cm = io[0];
  var outer_cm = io[1];
  
  //DebugLog.write(" " + character + ":" + Quote(locus) + " in=" + Quote(inner_cm) + " cent=" + Quote(center_cm) + " out=" + Quote(outer_cm));
  
  var distance_cm, direction_es;
  if (center_cm == 0) {
    // at precise center
    direction_es = undefined;
    distance_cm = inner_cm;

  } else if (center_cm > outer_cm) {
    // outside, approaching
    direction_es = center_es_cm;
    distance_cm = center_cm - outer_cm;

  } else if (center_cm < inner_cm) {
    // too close, trying to get away
    direction_es = [ -center_es_cm[0], -center_es_cm[1] ];
    distance_cm = inner_cm - center_cm;

  } else {
    // already in the goal locus
    direction_es = undefined;
    distance_cm = 0;
  }
  
  return [direction_es, distance_cm];
} 

// get the absolute east-south coordinates of the center of a locus
function LocusEsCm(locus) {
  //DebugCall("LocusEsCm", [locus]);

  if (typeof(locus) != "string") {
    Abort("Invalid locus: " + Quote(locus));
  }
    
  var words = locus.split(LocusDivider);
  var base = words[0];
  
  var es_cm;
  if (base == "origin") {
    es_cm = [0, 0];
  } else {
    var base_entity = AllEntities.find(base);
    if (base_entity == undefined) {
      Abort("Invalid base: " + Quote(base));
    }
    // copy coordinates to a new Array
    var temp = base_entity.esCm;
    es_cm = [temp[0], temp[1]];
  }

  if (words[1] != undefined && words[1] != "") {
    es_cm[0] += Number(words[1])/10;
  }
  if (words[2] != undefined && words[2] != "") {
    es_cm[1] += Number(words[2])/10;
  }
  
  //DebugLog.write(" LocusEsCm() result=" + QuoteList(es_cm));
  
  return es_cm;
}

function LocusRadiiCm(locus) {
  //DebugCall("LocusRadiiCm", [locus]);

  if (typeof(locus) != "string") {
    Abort("Invalid locus in LocusOuterCm(): " + Quote(locus));
  }
  
  var words = locus.split(LocusDivider);
  var inner_cm;
  if (words[3] == undefined || words[3] == "") {
    inner_cm = 0;
  } else {
    inner_cm = Number(words[3]);
  }
  var outer_cm;
  if (words[4] == undefined || words[4] == "") {
    outer_cm = inner_cm + LocusDefaultThicknessCm;
  } else {
    outer_cm = Number(words[4]);
  }

  return [inner_cm, outer_cm];
}