// layout.js
// battlefield layouts for battle simulation (Sphere)

/*  Copyright (C) 2008-2009  Stephen R. Gold

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/

RequireScript("team.js");
RequireScript("utilities/list.js");
    
// general layout of battlefield
Layout = OptionalEnum("Layout", "red on left", 
     ["red on left", "blue on left",
      "red in center", "blue in center",
      "red on top", "red on bottom" ], 
     "Map != 'river'");

// initial number of obstacles
Obstacles = OptionalEnum("Obstacles", 0, ListIntegers(0,12), "Map != 'river'");

// default obstacle model
ObstacleModel = OptionalEnum("ObstacleModel", "pillar",
     ["pillar", "tree-bare", "tree-leafy"], "Obstacles > 0");
     
// initial number of characters of each type
Blue.archers = Enum("Blue.archers", 1, ListIntegers(0,9));
Red.archers = Enum("Red.archers", 1, ListIntegers(0,9));

Blue.elfs = Enum("Blue.elfs", 1, ListIntegers(0,9));
Red.elfs = Enum("Red.elfs", 1, ListIntegers(0,9));

Blue.knights = Enum("Blue.knights", 1, ListIntegers(0,9));
Red.knights = Enum("Red.knights", 1, ListIntegers(0,9));

Blue.sands = Enum("Blue.sands", 1, ListIntegers(0,9));
Red.sands = Enum("Red.sands", 1, ListIntegers(0,9));

// initial number of portals
Blue.portals = Enum("Blue.portals", 3, ListIntegers(0,6));
Red.portals = Enum("Red.portals", 3, ListIntegers(0,6));

// initial portal settings
Blue.product = OptionalEnum("Blue.product", "alternate", 
   SampleProducts, "Blue.portals > 0");
Red.product = OptionalEnum("Red.product", "alternate", 
   SampleProducts, "Red.portals > 0");

RequireScript("coords.js");
RequireScript("entities/character.js");
RequireScript("entities/missile.js");
RequireScript("entities/obstacle.js");
RequireScript("entities/portal.js");
RequireScript("utilities/random.js");

// create entities on the map

function CreateEntities() {
  //DebugCall("CreateEntities");
  
  // reset bookkeeping
  AllBanners = new Roster("AllBanners");
  AllEntities = new Roster("AllEntities");
  AllObstacles = new Roster("AllObstacles");
  Blue.reset();
  Blue.setOpposite(Red);
  Red.reset();
  Red.setOpposite(Blue);
  
  // a permanent invisible entity to track simulation time
  NewTimeKeeper();

  // a permanent invisible entity to mark the center of the map
  NewOrigin();
       
  if (Layout == "red in center") {
    CreatePeopleConcentric(Red);
  } else if (Layout == "blue in center") {
    CreatePeopleConcentric(Blue);
  } else if (Layout == "red on left") {
    CreatePeopleLines(Red, false);
  } else if (Layout == "blue on left") {
    CreatePeopleLines(Blue, false);
  } else if (Layout == "red on top") {
    CreatePeopleLines(Red, true);
  } else if (Layout == "blue on bottom") {
    CreatePeopleLines(Blue, true);
  } else {
    Abort("Unknown layout in CreatePeople(): " + Quote(Layout));
  }
  //DebugLog.write("-->leaving CreatePeople()");
}

// a permanent invisible entity to track simulation time
function NewTimeKeeper() {
  //DebugCall("NewTimeKeeper");

  var name = " "; // This seems to give its command generator priority.  
  NameExists[name] = true;
  CreatePerson(name, "footprint.rss", false);

  IgnorePersonObstructions(name, true);  
  SetPersonScript(name, SCRIPT_COMMAND_GENERATOR, "UpdateTimekeeper()");
  SetPersonVisible(name, false);
    
  Ticks = 0;
}

function UpdateTimekeeper() {
  if (IsPaused()) {
    return;
  }
  //DebugCall("UpdateTimekeeper", [], Quote(Ticks+1));
  
  Ticks++;  // update global simulation time
  
  if (CountdownFlag && Ticks >= CountdownSeconds * TicksPerSecond) {
    GameOver("Your time ran out.");
  }
  
  for (var name in AllEntities.members) {
    var entity = AllEntities.members[name];
    if (entity != undefined && entity.update != undefined) {
      entity.update();
    }
  }
}

function IsPaused() {
  return PauseFlag;
}

function CreatePeopleConcentric(inside_team) {
  //DebugCall("CreatePeopleConcentric", [inside_team]);
  
  if (Map == "river") {
    Abort("Invalid Map in CreatePeopleConcentric()");
  }

  var outside_team = inside_team.opposite;

  var map_ew_cm = GetLayerWidth(Layer)*12;
  var create_area_ew_cm = 105*30;
  if (create_area_ew_cm > map_ew_cm/2) {
    create_area_ew_cm = map_ew_cm/2;
  }
  
  var map_ns_cm = GetLayerHeight(Layer)*12;
  var create_area_ns_cm = 105*30;
  if (create_area_ns_cm > map_ns_cm/2) {
    create_area_ns_cm = map_ns_cm/2;
  }

  var TwoPi = 2 * Math.PI;

  var entity, east_cm, south_cm, z;
  
  // inside portals
  var team = inside_team;
  for (var i = 0; i < team.portals; i++) {
    entity = new Portal(team);
    if (team.portals == 1) {
      east_cm = 0;
      south_cm = 0;
    } else {
      z = TwoPi * (i + 0.5)/team.portals;
      east_cm = 0.12 * Math.sin(z) * create_area_ew_cm;
      south_cm = 0.12 * Math.cos(z) * create_area_ns_cm;
    }
    if (entity.place([east_cm, south_cm])) {
      Abort("can't place " + team.color + " portal");
    }
    entity.changeDirection([east_cm, south_cm]);
  } 

  // inside characters
  for (var i = 0; i < team.elfs; i++) {
    entity = new Character(team, "elf");
    do {
      z = Uniform(0, TwoPi);
      east_cm = 0.2 * Math.sin(z) * create_area_ew_cm;
      south_cm = 0.2 * Math.cos(z) * create_area_ns_cm;
    } while (entity.place([east_cm, south_cm]));
    entity.changeDirection([east_cm, south_cm]);
  }
  for (var i = 0; i < team.archers; i++) {
    entity = new Character(team, "archer");
    do {
      z = Uniform(0, TwoPi);
      east_cm = 0.27 * Math.sin(z) * create_area_ew_cm;
      south_cm = 0.27 * Math.cos(z) * create_area_ns_cm;
    } while (entity.place([east_cm, south_cm]));
    entity.changeDirection([east_cm, south_cm]);
  }
  for (var i = 0; i < team.sands; i++) {
    entity = new Character(team, "sands");
    do {
      z = Uniform(0, TwoPi);
      east_cm = 0.34 * Math.sin(z) * create_area_ew_cm;
      south_cm = 0.34 * Math.cos(z) * create_area_ns_cm;
    } while (entity.place([east_cm, south_cm]));
    entity.changeDirection([east_cm, south_cm]);
  }
  for (var i = 0; i < team.knights; i++) {
    entity = new Character(team, "knight");
    do {
      z = Uniform(0, TwoPi);
      east_cm = 0.37 * Math.sin(z) * create_area_ew_cm;
      south_cm = 0.37 * Math.cos(z) * create_area_ns_cm;
    } while (entity.place([east_cm, south_cm]));
    entity.changeDirection([east_cm, south_cm]);
  }

  // obstacles
  for (var i = 0; i < Obstacles; i++) {
    z = TwoPi * (i + 0.5)/Obstacles;
    entity = new Obstacle(ObstacleModel);
    east_cm = 0.4 * Math.sin(z) * create_area_ew_cm;
    south_cm = 0.4 * Math.cos(z) * create_area_ns_cm;
    if (entity.place([east_cm, south_cm])) {
      Abort("can't place " + entity.model);
    }
  }

  // outside characters
  team = outside_team;
  for (var i = 0; i < team.knights; i++) {
    entity = new Character(team, "knight");
    do {
      z = Uniform(0, TwoPi);
      east_cm = 0.43 * Math.sin(z) * create_area_ew_cm;
      south_cm = 0.43 * Math.cos(z) * create_area_ns_cm;
    } while (entity.place([east_cm, south_cm]));
    entity.changeDirection([-east_cm, -south_cm]);
  }
  for (var i = 0; i < team.sands; i++) {
    entity = new Character(team, "sands");
    do {
      z = Uniform(0, TwoPi);
      east_cm = 0.52 * Math.sin(z) * create_area_ew_cm;
      south_cm = 0.52 * Math.cos(z) * create_area_ns_cm;
    } while (entity.place([east_cm, south_cm]));
    entity.changeDirection([-east_cm, -south_cm]);
  }
  for (var i = 0; i < team.archers; i++) {
    entity = new Character(team, "archer");
    do {
      z = Uniform(0, TwoPi);
      east_cm = 0.72 * Math.sin(z) * create_area_ew_cm;
      south_cm = 0.72 * Math.cos(z) * create_area_ns_cm;
    } while (entity.place([east_cm, south_cm]));
    entity.changeDirection([-east_cm, -south_cm]);
  }
  for (var i = 0; i < team.elfs; i++) {
    entity = new Character(team, "elf");
    do {
      z = Uniform(0, TwoPi);
      east_cm = 0.8 * Math.sin(z) * create_area_ew_cm;
      south_cm = 0.8 * Math.cos(z) * create_area_ns_cm;
    } while (entity.place([east_cm, south_cm]));
    entity.changeDirection([-east_cm, -south_cm]);
  }
  
  // outside portals
  for (var i = 0; i < team.portals; i++) {
    z = TwoPi * i/team.portals;
    entity = new Portal(team);
    east_cm = 0.86 * Math.sin(z) * create_area_ew_cm;
    south_cm = 0.86 * Math.cos(z) * create_area_ns_cm;
    if (entity.place([east_cm, south_cm])) {
      Abort("can't place " + team.color + "portal");
    }
    entity.changeDirection([-east_cm, -south_cm]);
  }

  //DebugLog.write("-->leaving CreatePeopleConcentric()");
}

function CreatePeopleLines(left_team, top_flag) {
  //DebugCall("CreatePeopleLines", [left_team, top_flag]);
  
  if (Map == "river" && (top_flag || Obstacles > 0)) {
    Abort("Invalid configuration in CreatePeopleLines(): Obstacles=" + Quote(Obstacles));
  }

  var right_team = left_team.opposite;
  
  var map_ew_cm = GetLayerWidth(Layer)*12;
  var create_area_ew_cm = 105*30;
  if (create_area_ew_cm > map_ew_cm) {
    create_area_ew_cm = map_ew_cm;
  }
  
  var map_ns_cm = GetLayerHeight(Layer)*12;
  var create_area_ns_cm = 105*30;
  if (create_area_ns_cm > map_ns_cm) {
    create_area_ns_cm = map_ns_cm;
  }
    
  var entity, east_cm, south_cm, z;
  
  // left/top portals
  var team = left_team;
  for (var i = 0; i < team.portals; i++) {
    entity = new Portal(team, [1, 0]);
    z = -0.5 + (i + 0.5)/team.portals;
    if (top_flag) {
      east_cm = z * create_area_ew_cm;
      south_cm = -0.45 * create_area_ns_cm;
    } else {
      east_cm = -0.45 * create_area_ew_cm;
      south_cm = z * create_area_ns_cm;
    }
    if (entity.place([east_cm, south_cm])) {
      Abort("can't place " + team.color + " portal");
    }
  } 

  // left/top characters
  for (var i = 0; i < team.elfs; i++) {
    entity = new Character(team, "elf");
    do {
      if (top_flag) {
        east_cm = Uniform(-0.5, 0.5) * create_area_ew_cm;
        south_cm = Uniform(-0.6, -0.5) * create_area_ns_cm;
      } else {
        east_cm = Uniform(-0.6, -0.5) * create_area_ew_cm;
        south_cm = Uniform(-0.5, 0.5) * create_area_ns_cm;
      }
    } while (entity.place([east_cm, south_cm]));
    entity.changeDirection([1, 0]);
  }
  for (var i = 0; i < team.archers; i++) {
    entity = new Character(team, "archer");
    do {
      if (top_flag) {
        east_cm = Uniform(-0.5, 0.5) * create_area_ew_cm;
        south_cm = Uniform(-0.4, -0.3) * create_area_ns_cm;
      } else {
        east_cm = Uniform(-0.4, -0.3) * create_area_ew_cm;
        south_cm = Uniform(-0.5, 0.5) * create_area_ns_cm;
      }
    } while (entity.place([east_cm, south_cm]));
    entity.changeDirection([1, 0]);
  }
  for (var i = 0; i < team.sands; i++) {
    entity = new Character(team, "sands");
    do {
      if (top_flag) {
        east_cm = Uniform(-0.5, 0.5) * create_area_ew_cm;
        south_cm = Uniform(-0.3, -0.2) * create_area_ns_cm;
      } else {
        east_cm = Uniform(-0.3, -0.2) * create_area_ew_cm;
        south_cm = Uniform(-0.5, 0.5) * create_area_ns_cm;
      }
    } while (entity.place([east_cm, south_cm]));
    entity.changeDirection([1, 0]);
  }
  for (var i = 0; i < team.knights; i++) {
    entity = new Character(team, "knight");
    do {
      if (top_flag) {
        east_cm = Uniform(-0.5, 0.5) * create_area_ew_cm;
        south_cm = Uniform(-0.2, -0.17) * create_area_ns_cm;
      } else {
        east_cm = Uniform(-0.2, -0.17) * create_area_ew_cm;
        south_cm = Uniform(-0.5, 0.5) * create_area_ns_cm;
      }
    } while (entity.place([east_cm, south_cm]));
    entity.changeDirection([1, 0]);
  }

  // obstacles
  for (var i = 0; i < Obstacles; i++) {
    entity = new Obstacle(ObstacleModel);
    z = -0.5 + (i + 0.5)/Obstacles;
    if (top_flag) {
      east_cm = z * create_area_ew_cm;
      south_cm = z/6 * create_area_ns_cm;
    } else {
      east_cm = z/6 * create_area_ew_cm;
      south_cm = z * create_area_ns_cm;
    }
    if (entity.place([east_cm, south_cm])) {
      Abort("can't place " + entity.model);
    }
  }

  // right/bottom characters
  var team = right_team;
  for (var i = 0; i < team.knights; i++) {
    entity = new Character(team, "knight");
    do {
      if (top_flag) {
        east_cm = Uniform(-0.5, 0.5) * create_area_ew_cm;
        south_cm = Uniform(0.17, 0.2) * create_area_ns_cm;
      } else {
        east_cm = Uniform(0.17, 0.2) * create_area_ew_cm;
        south_cm = Uniform(-0.5, 0.5) * create_area_ns_cm;
      }
    } while (entity.place([east_cm, south_cm]));
    entity.changeDirection([-1, 0]);
  }
  for (var i = 0; i < team.sands; i++) {
    entity = new Character(team, "sands");
    do {
      if (top_flag) {
        east_cm = Uniform(-0.5, 0.5) * create_area_ew_cm;
        south_cm = Uniform(0.2, 0.3) * create_area_ns_cm;
      } else {
        east_cm = Uniform(0.2, 0.3) * create_area_ew_cm;
        south_cm = Uniform(-0.5, 0.5) * create_area_ns_cm;
      }
    } while (entity.place([east_cm, south_cm]));
    entity.changeDirection([-1, 0]);
  }
  for (var i = 0; i < team.archers; i++) {
    entity = new Character(team, "archer");
    do {
      if (top_flag) {
        east_cm = Uniform(-0.5, 0.5) * create_area_ew_cm;
        south_cm = Uniform(0.3, 0.4) * create_area_ns_cm;
      } else {
        east_cm = Uniform(0.3, 0.4) * create_area_ew_cm;
        south_cm = Uniform(-0.5, 0.5) * create_area_ns_cm;
      }
    } while (entity.place([east_cm, south_cm]));
    entity.changeDirection([-1, 0]);
  }
  for (var i = 0; i < team.elfs; i++) {
    entity = new Character(team, "elf");
    do {
      if (top_flag) {
        east_cm = Uniform(-0.5, 0.5) * create_area_ew_cm;
        south_cm = Uniform(0.5, 0.6) * create_area_ns_cm;
      } else {
        east_cm = Uniform(0.5, 0.6) * create_area_ew_cm;
        south_cm = Uniform(-0.5, 0.5) * create_area_ns_cm;
      }
    } while (entity.place([east_cm, south_cm]));
    entity.changeDirection([-1, 0]);
  }
  
  // right/bottom portals
  for (var i = 0; i < team.portals; i++) {
    z = -0.5 + (i + 0.5)/team.portals;
    entity = new Portal(team, [-1, 0]);
    if (top_flag) {
      east_cm = z * create_area_ew_cm;
      south_cm = 0.45 * create_area_ns_cm;
    } else {
      east_cm = 0.45 * create_area_ew_cm;
      south_cm = z * create_area_ns_cm;
    }
    if (entity.place([east_cm, south_cm])) {
      Abort("can't place " + team.color + " portal");
    }
  }

  //DebugLog.write("-->leaving CreatePeopleLines()");
}