// team_entity.js
// TeamEntity class for battle simulation (Sphere)

/*  Copyright (C) 2008-2009  Stephen R. Gold

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
    
// A TeamEntity is an Entity associated with a Team.
// There are seven TeamEntity models: {arrow, gonfalon, portal} + 4 characters

// A TeamEntity has the following properties:
//  + es           normalized direction vector
//  + hitPoints    number of hits until it's disabled
//  + maxHitPoints
//  + team         {Red, Blue}
//    + all the properties of an Entity (see entity.js).


RequireScript("entities/entity.js");
RequireScript("soundfx.js");
RequireScript("team.js");
RequireScript("utilities/direction.js");
RequireScript("utilities/random.js");


// creation functions

function TeamEntity(team, model, status, version) {
  if (this instanceof TeamEntity == false) {
    return new TeamEntity(team, model, status, version);
  }
  //DebugCall("TeamEntity", [team, model, status, version]);
  if (team == undefined) {
    return this;
  }
  
  Entity.apply(this, [model, status, version]);
    
  this.armor = [];
  this.setTeam(team);
  
  return this;
}
TeamEntity.prototype = new Entity();
TeamEntity.prototype.constructor = TeamEntity;

// update functions

TeamEntity.prototype.hitBy =
function(hitter) {
  //DebugCall("TeamEntity.hitBy", [hitter], Quote(this));

  var armor_effect = 1;
  if (this.isFrozen()) {
    if (hitter.model == "arrow") {
      // frozen armor is 50% effective against arrows
      armor_effect = 0.5;
    } else if (hitter.isCrusher()) {
      // frozen armor is 10% effective against crushers
      armor_effect = 0.1;
    } else {
      Abort("Unexpected hitter: " + Quote(hitter));
    }
  }
  
  if (armor_effect > 0) {
    var vulnerability = this.vulnerability(hitter);

    vulnerability /= armor_effect;
    if (vulnerability < 1 && Math.random() > vulnerability) {
      // the attack is parried (or deflected by armor) and does no damage
      PlaySoundFx("bounce " + this.model, this);
      return false;
    }
  }
  
  // The attack does one point of damage.
  var disable_flag = this.damage(hitter.name, 1);
  
  return disable_flag;
}

TeamEntity.prototype.damage =
function(cause, points) {
  //DebugCall("TeamEntity.damage", [cause, points], Quote(this));
  
  if (typeof(cause) != "string") {
    Abort("Unexpected cause: " + Quote(cause));
  }
  if (typeof(points) != "number"
   || points < 1) {
    Abort("Invalid points: " + Quote(points));
  }
  
  if (this.isCharacter()) {
    PlaySoundFx("damage character", this);
    
  } else if (this.isPortal()) {
    PlaySoundFx("damage portal", this);
    
  } else {
    Abort("Unexpected model in TeamEntity.damage(): " + Quote(this.model));
  }

  if (this.hitPoints == undefined) {
    Abort("Hit points not defined in TeamEntity.damage()");
  }
  
  this.hitPoints -= points;
  if (this.hitPoints <= 0) {
    this.disable(cause);
    return true;
  }
  
  // target isn't disabled yet
  return false
}

TeamEntity.prototype.disable =
function(cause) {
  //DebugCall("TeamEntity.disable", [cause]);
  
  Abort("Unexpected call to TeamEntity.disable(): model=" + Quote(this.model));
}

TeamEntity.prototype.changeDirection =
function(es) {
  //DebugLog("TeamEntity.changeDirection", [es]);

  if (typeof(es) != "object" 
   || typeof(es[0]) != "number" 
   || typeof(es[1]) != "number") {
    Abort("Invalid frame in Entity.changeFrame(): " + Quote(frame));
  }

  if (es[0] == 0 && es[1] == 0) {
    // if in doubt, face north
    es = [0, -1];
  }

  // normalize to unit length and store vector  
  var magnitude = NormV(es);
  var east = es[0]/magnitude;
  var south = es[1]/magnitude;
  this.es = [east, south];
}

TeamEntity.prototype.setMaxHitPoints =
function(points) {
  //DebugCall("TeamEntity.setMaxHitPoints", [points]);

  this.maxHitPoints = points;
  
  if (this.hitPoints == undefined || this.hitPoints > points) {
    this.hitPoints = points;
  }
}

TeamEntity.prototype.setTeam =
function(team) {
  //DebugCall("TeamEntity.setTeam", [team]);
  
  if (typeof(team) != "object"
   || typeof(team.name) != "string") {
    Abort("Invalid team: " + Quote(team));
  }

  this.team = team;
}

// read-only functions

TeamEntity.prototype.getDirectionName =
function() {
  //DebugCall("TeamEntity.getDirectionName", [], Quote(this));

  var num_directions = this.numDirections();
  var direction_name = DirectionN(this.es[0], this.es[1], num_directions);
  
  return direction_name  
}

TeamEntity.prototype.vulnerability =
function(attacker) {
  var a_east = attacker.es[0];
  var a_south = attacker.es[1];
  var t_east = this.es[0];
  var t_south = this.es[1];
  
  var m_cos = t_east * a_east + t_south * a_south; 
  var m_sin = t_east * a_south - t_south * a_east;
  var scale = Math.abs(m_cos) + Math.abs(m_sin);
  //DebugLog.write(" m_cos=" + Quote(m_cos) + ", m_sin=" + Quote(m_sin) + ", scale=" + String(scale));

  var vulnerability = 1;
  if (m_cos > 0) {
    vulnerability -= m_cos * this.armor["rear"] / scale;
  } else if (m_cos < 0) {
    vulnerability += m_cos * this.armor["front"] / scale;
  }
  if (m_sin > 0) {
    vulnerability -= m_sin * this.armor["left"] / scale;
  } else if (m_sin < 0) {
    vulnerability += m_sin * this.armor["right"] / scale;
  }
  
  return vulnerability;
}