// portal.js
// Portal class for battle simulation (Sphere)

/*  Copyright (C) 2008-2009  Stephen R. Gold

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/

// A Portal is an Entity which spawns new Characters.

// cap on number of able characters per team
MaxRoster = Enum("MaxRoster", 10, [2, 3, 4, 5, 6, 8, 10, 12, 15, 20, 30]);

// number of hits to destroy portal
PortalHits = OptionalEnum("PortalHits", 1, [1, 2, 3, 4, 6, 8, 10, 999999],
    "Blue.portals + Red.portals > 0");

SampleProducts = ["alternate", "archer", "elf", "none", "mix 30%archer 70%sands", "sands"];

RequireScript("entities/team_entity.js");
RequireScript("overlays/bolt.js");
RequireScript("soundfx.js");
RequireScript("utilities/random.js");

ProductBoltColor = CreateColor(255, 255, 255); // white

Portal.prototype = new TeamEntity();
Portal.prototype.constructor = Portal;

// create

function Portal(team, es) {
  if (this instanceof Portal == false) {
    return new Portal(team, es);
  }
  //DebugCall("Portal", [team, es]);
  if (team == undefined) {
    return this;
  }
  
  var model = "portal";
  var status = "";
  TeamEntity.apply(this, [team, model, status]);

  if (es != undefined) {
    this.changeDirection(es);
  }
  this.product = team.product;      // model(s) which the portal will spawn
  this.setLayer(GroundLayer);
  this.setMaxHitPoints(PortalHits); // hits to destroy portal
  this.setTickCount(0);             // number of ticks till first spawn
  
  //bookkeeping
  team.allPortals.add(this);
  
  return this;
}

// update

Portal.prototype.update =
function() {
  //DebugCall("Portal.update", [], Quote(this));

  // update portal opacity
  var opacity = 0.6 + 0.4*(this.hitPoints/this.maxHitPoints);
  this.changeOpacity(opacity);

  var tick_count = this.tickCount;
  tick_count--;

  var spawn = this.spawn;
  var team_color = this.team.name;
  if (spawn != undefined) {
    var max_ticks = 0.3 * spawn.frameDuration["spa0"];

    if (tick_count < max_ticks 
     && this.team.upCharacters.count() >= MaxRoster) {
      // no room for spawn in roster - hold at 70% complete
      tick_count = max_ticks;
      spawn.setTickCount(tick_count);
    }
  }
  
  if (tick_count >= 0) {
    this.setTickCount(tick_count);
    return;
  }
  
  if (spawn != undefined) {
    //DebugLog.write(" solidify " + Quote(spawn));
    PlaySoundFx("spawn " + spawn.model, spawn);
    spawn.changeActiveStatus("adv");
    spawn.setLayer(ObstacleLayer);
    spawn.changeOpacity(1);
    spawn.team.upCharacters.add(spawn);
    if (spawn.isCrusher()) {
      spawn.team.numCrushers++;
    }
    this.lastModel = spawn.model;
  }
  
  // spawn next character
  var product = this.nextProduct();
  if (product == undefined) {
    return;
  }
  spawn = new Character(this.team, product, "spa");

  if (spawn.place(this.esCm)) {
    // the portal is blocked
    spawn.deleteEntity();
    //DebugLog.write(" " + spawn.name + " aborted because " + portal.name + " was blocked");
    this.setTickCount(TicksPerSecond/4); // wait 1/4 sec before retrying
    this.spawn = undefined;

  } else {
    spawn.changeDirection(this.es);
    spawn.changeFrame(0);
    
    this.setTickCount(spawn.tickCount);
    this.spawn = spawn;
  }
}

function ClickChangeProductAlternate(screen_x, screen_y) {
  //DebugCall("ClickChangeProductAlternate", [screen_x, screen_y]);
  var map_x = ScreenToMapX(Layer, screen_x);
  var map_y = ScreenToMapY(Layer, screen_y);

  var portal = FindPortal(map_x, map_y, CommandBlueFlag, CommandRedFlag);
  if (portal != undefined) {
    //TheBolt = Bolt(portal, screen_x, ProductBoltColor);
    portal.changeProduct("alternate");
  }
}

function ClickChangeProductArcher(screen_x, screen_y) {
  //DebugCall("ClickChangeProductArcher", [screen_x, screen_y]);
  var map_x = ScreenToMapX(Layer, screen_x);
  var map_y = ScreenToMapY(Layer, screen_y);

  var portal = FindPortal(map_x, map_y, CommandBlueFlag, CommandRedFlag);
  if (portal != undefined) {
    //TheBolt = Bolt(portal, screen_x, ProductBoltColor);
    portal.changeProduct("archer");
  }
}

function ClickChangeProductSands(screen_x, screen_y) {
  //DebugCall("ClickChangeProductSands", [screen_x, screen_y]);
  var map_x = ScreenToMapX(Layer, screen_x);
  var map_y = ScreenToMapY(Layer, screen_y);

  var portal = FindPortal(map_x, map_y, CommandBlueFlag, CommandRedFlag);
  if (portal != undefined) {
    //TheBolt = Bolt(portal, screen_x, ProductBoltColor);
    portal.changeProduct("sands");
  }
}

Portal.prototype.changeProduct = 
function(new_product) {
  //DebugCall("Portal.changeProduct", [new_product]);

  if (this.product == new_product) {
    // no effect
    return;
  }
  
  this.product = new_product; 

  if (this.spawn != undefined) {
    this.spawn.deleteEntity();
    this.spawn = undefined;
  }
  this.setTickCount(0);
}

// read-only functions

Portal.prototype.nextProduct = 
function() {
  //DebugCall("Portal.nextProduct");
 
  var product = this.product; 
  if (IsMix(product)) {
    product = RandomMix(product);
  }

  if (product == "alternate") {
    // alternating archer/sands
    if (this.lastModel == "sands") {
      product = "archer";
    } else {
      product = "sands";
    }

  } else if (product == "none") {
    product = undefined;  

  } else if (product != "archer" && product != "elf" && product != "sands") {
    Abort("Unknown product in Entity.getProduct(): " + Quote(product));
  }
  
  return product;
}

Entity.prototype.isPortal =
function() {
  //DebugCall("Entity.isPortal");
  
  var model = this.model;
  var result = (model == "portal");
  
  return result;
}

Portal.prototype.disable =
function(cause) {
  //DebugCall("Portal.disable", [cause]);
      
  // disable portal
  PlaySoundFx("disable portal", this);
  
  this.team.allPortals.remove(this);

  if (this.spawn != undefined) {
    this.spawn.deleteCharacter();
    //DebugLog.write(" " + spawn.name + " deleted because " + this.name + " was disabled");
  }

  this.deleteEntity();
  //DebugLog.write(" " + cause + " disabled " + this.name);
}

// find the portal to edit (based on screen distance)
function FindPortal(map_x, map_y, include_blue_flag, include_red_flag) {
  //DebugCall("FindPortal", [map_x, map_y, include_blue_flag, include_red_flag]);

  var map_xy = [map_x, map_y];
  
  var portal;
  if (include_blue_flag == undefined || include_blue_flag) {
    portal = Blue.allPortals.best(map_xy);
  }
  if (include_red_flag == undefined || include_red_flag) {
    portal = Red.allPortals.best(map_xy, portal);
  }
  
  return portal;
}
