// missile.js
// Missile class for battle simulation (Sphere)

/*  Copyright (C) 2008-2009  Stephen R. Gold

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/

// A Missile is a Mobile (see mobile.js) that represents a flying weapon.
// One model is defined: {arrow}.

// A Missile has the following properties:
//  + dropCm       distace over which it drops to the ground
//  + missProb     probability of missing its target
//  + remainingCm  distance remaining before it reaches the ground
//   + all the properties of a Mobile 

RequireScript("entities/mobile.js");
RequireScript("soundfx.js");
RequireScript("utilities/random.js");

// creation function

function Missile(model, source, target_name) {
  if (this instanceof Missile == false) {
    return new Missile(model, source, target_name);
  }
  //DebugCall("Missile", [model, source, target_name]);
  if (model == undefined) {
    return this;
  }
  
  if (model != "arrow") {
    Abort("Unexpected model: " + Quote(model));
  }
  if (source instanceof Entity == false) {
    Abort("Unexpected source: " + Quote(source));
  }
  if (typeof(target_name) != "string") {
    Abort("Unexpected target_name: " + Quote(target_name));
  }

  PlaySoundFx("new " + model, source);
  
  var status = "fly";
  var team = source.team;
  Mobile.apply(this, [team, model, status]);
  
  this.frame = 0;
  this.setLayer(MissileLayer);
  
  var target = AllEntities.find(target_name);

  var east_cm = target.esCm[0] - source.esCm[0];
  var south_cm = target.esCm[1] - source.esCm[1];
  this.changeDirection([east_cm, south_cm]);

  var max_range_cm = source.maxRangeCm;
  var drop_cm = 1.6 * max_range_cm;  // travels 60% further than effective range
  
  var dist_cm = source.entityCm(target);
  var miss_prob = 0.5 * dist_cm/max_range_cm; // miss probability 50% at max effective range 
  var speed_cmps = 75*30;
  var step_cm = speed_cmps / TicksPerSecond;

  this.setDropCm(drop_cm);
  this.setFrameDuration("spe", 0, 1.2*FadeSeconds*TicksPerSecond); // for slow fade (after doing damage)
  this.setMissProb(miss_prob);
  this.setSourceName(source.name);
  this.setStepCm(step_cm);
  this.setTargetName(target_name);
  
  this.place(source.esCm);

  // bookkeeping
  team.numFlys++;

  return this;
}
Missile.prototype = new Mobile();
Missile.prototype.constructor = Missile;


Missile.prototype.update =
function() {
  //DebugCall("Missile.update", [], Quote(this));
  
  var status = this.status;
  //DebugLog.write(" status = " + Quote(status));
  
  if (status == "spe") {
    // spent and fading
    this.fadeOut();
    return;
  }
  
  // status == "fly" or "pas"
  var remaining_cm = this.remainingCm - this.stepCm;
  this.remainingCm = remaining_cm;

  if (remaining_cm < 0) {
    // newly spent
    this.changeStatus("spe");
    this.changeFrame(0);
    this.setLayer(GroundLayer);
    this.team.numFlys--;
    this.setTickCount(0.7*FadeSeconds*TicksPerSecond); // fast fade after miss
    return;
  }

  // continue motion
  if (status == "fly") { // approaching target
    var target = AllEntities.find(this.targetName);
    var dist_cm = this.entityCm(target);
    //DebugLog.write(" " + missile + " target=" + target + " dist_cm=" + Quote(dist_cm) + " step_cm=" + Quote(this.stepCm));
    
    if (this.stepCm > dist_cm) {
      // at closest approach to target
      this.team.numFlys--;

      if (Math.random() > this.missProb && target.isUp()) {
        // a hit!  
        var disable_flag = target.hitBy(this);
        if (disable_flag) {
          // merged with target
          this.deleteEntity();
        
        } else {
          // rebound one meter from target
          var cm = 100;
          var east_cm = target.esCm[0] - cm*this.es[0];
          var south_cm = target.esCm[1] - cm*this.es[1];
          this.place([east_cm, south_cm]);
          
          // fade rapidly
          this.changeStatus("spe");
          this.changeFrame(0);
          this.setLayer(GroundLayer);
          this.setTickCount(0.7*FadeSeconds*TicksPerSecond);
        }
        return;
      }
      
      // miss and continue past
      status = "pas";
      this.changeStatus(status);
    }    
  }

  // update height above ground
  this.height(remaining_cm/this.dropCm);
  
  var east, south; // direction of flight
  if (status == "pas") {
    // continue in previous direction
  } else if (status == "fly") {
    // zero in on target
    east = target.esCm[0] - this.esCm[0];
    south = target.esCm[1] - this.esCm[1];
    this.changeDirection([east, south]);
  } else {
    Abort("Unknown status: " + Quote(status));
  }
    
  var east_cm = this.esCm[0] + this.stepCm*this.es[0];
  var south_cm = this.esCm[1] + this.stepCm*this.es[1];
  this.place([east_cm, south_cm]);
}

Missile.prototype.height =
function(remaining) {
  //DebugCall("Missile.height", [remaining]);

  //DebugLog.write (" " + missile + " has " + Quote(remaining) + " remaining, h=" + Quote(GetPersonFrame(missile)));

  if (remaining < 0.15) {  // less than 15% of flight distance remains
    this.frame = 3; // 37 cm above ground (16 px @ 2cm res)
  } else if (remaining < 0.3) {
    this.frame = 2; // 72 cm above ground (31 px @ 2cm res)
  } else if (remaining < 0.6) {
    this.frame = 1; // 108 cm above ground (47 px @ 2cm res)
  } 
}

Missile.prototype.setDropCm =
function(cm) {
  //DebugCall("Missile.setDropCm", [cm], Quote(this));
  
  this.dropCm = cm;
  this.remainingCm = cm;
}

Missile.prototype.setMissProb =
function(prob) {
  //DebugCall("Missile.setMissProb", [prob], Quote(this));
  
  this.missProb = prob;
}
