// entity.js
// Entity class for battle simulation (Sphere)

/*  Copyright (C) 2008-2009  Stephen R. Gold

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/

// == Taxonomy of Entity models ==
//
//  A. footprint
//  B. Obstacle (see obstacle.js)
//   B1. pillar
//   B2. tree
//  C. origin
//  D. TeamEntity (see team_entity.js)
//   D1. banner (see banner.js)
//    D1a. gonfalon
//   D2. Mobile
//    D2a. Character (see character.js)
//     D2a1. archer
//     D2a2. elf
//     D2a3. knight
//     D2a4. sands
//    D2b. missile (see missile.js)
//     D2b1. arrow
//   D3. Portal (see portal.js)

RequireScript("coords.js");
RequireScript("name.js");
RequireScript("utilities/direction.js");
RequireScript("utilities/graphics.js");
RequireScript("utilities/math.js");
RequireScript("utilities/random.js");

FadeOutFlag = PlayerFlag("FadeOut", true);

// fade time for disabled characters and spent arrows, in sim seconds
FadeSeconds = PlayerEnum("FadeSeconds", 15, [0, 1, 3, 5, 10, 15, 30, 60, 9999]);

// update inhibitor
PauseFlag = PlayerFlag("Pause", true);

// draw bounding box for each frame
BoxesFlag = PlayerFlag("Boxes", false);

BoxesOutlineColor = CreateColor(0, 0, 0, 200);  // black

// creation functions

function Entity(model, status, version) {
  if (this instanceof Entity == false) {
    return new Entity(model, status, version);
  }
  //DebugCall("Entity", [model, status, version]);
  if (model == undefined) {
    return this;
  }
  
  var name = NewName(model);
  if (AllEntities.find(name) != undefined) {
    Abort("Duplicate entity name: " + Quote(name));
  }
  
  this.frame = 0;
  this.frameDuration = [];
  this.name = name;
  this.model = model;

  this.changeStatus(status);
  var white = CreateColor(255, 255, 255);
  this.setMask(white);
  this.setVersion(version);
  // this.esCm
  // this.layer
  
  AllEntities.add(this);
  
  return this;
}

Entity.prototype.leaveFootprint =
function() {
  //DebugCall("Entity.leaveFootprint");

  var entity = new Entity("footprint", "");
  entity.setEsCm(this.esCm);  
  entity.setLayer(GroundLayer);
  entity.setSourceName(this.name);
  entity.setTickCount(FadeSeconds*TicksPerSecond);
  entity.update = entity.fadeOut;
}

// a permanent invisible entity to mark the center of the map
function NewOrigin() {
  //DebugCall("NewOrigin");

  var name = "origin";
  NameExists[name] = true;
  var entity = new Entity(name, "origin");

  entity.changeStatus("");
  entity.setEsCm([0, 0]);
  entity.setLayer(GroundLayer);
  var invisible = CreateColor(0,0,0,0);
  entity.setMask(invisible);
  
  return name;
}

// destructors

Entity.prototype.deleteEntity =
function() {
  //DebugCall("Entity.deleteEntity");
  
  if (LocusBase(EyeLocus) == this.name) {
    var es_cm = LocusEsCm(EyeLocus);
    EyeLocus = FixedLocus(es_cm);
  }
  
  AllEntities.remove(this);
}

// update functions

Entity.prototype.changeFrame =
function(frame) {
  //DebugCall("Entity.changeFrame", [frame]);

  if (typeof(frame) != "number" 
   || frame < 0
   || frame != Math.round(frame)) {
    Abort("Invalid frame in Entity.changeFrame(): " + Quote(frame));
  }

  var last_frame = this.getLastFrame();
  if (frame > last_frame) {
    //DebugLog.write(" cycle around");
    frame = frame % (last_frame + 1);
  }
  this.frame = frame;

  var duration = this.getFrameDuration();
  this.setTickCount(duration);
  //DebugLog.write(" " + Quote(this.name) + " tick_count set to " + Quote(duration));
  
  return frame;
}

Entity.prototype.changeOpacity =
function(opacity) {
  //DebugCall("Entity.changeOpacity", [opacity], Quote(this));

  if (typeof(opacity) != "number") {
    Abort("Invalid opacity: " + Quote(opacity));
  }
  
  if (opacity < 0) {
    opacity = 0;
  } else if (opacity > 1) {
    opacity = 1;
  }

  if (this.mask == undefined) {
    Abort("Entity lacks a mask: " + Quote(this.name));  
  }
  
  // nonlinear
  var alpha = Math.round(opacity * opacity * 255);
  this.mask.alpha = alpha;
}

Entity.prototype.changeStatus =
function(status, dis) {
  //DebugCall("Entity.changeStatus", [status, dis], Quote(this));

  if (typeof(status) != "string") {
    Abort("Invalid status: " + Quote(status));
  }
  if (status == this.status) {
    Abort(this.name + " is already " + status);
  }
  
  this.status = status;
  if (status == "dis") {
    this.dis = dis;
  }

  return status;
}

Entity.prototype.draw =
function() {
  //DebugCall("Entity.draw", [], Quote(this));

  if (!this.isVisible()) {
    return;
  }
  var image_name = this.getImageName();
  var map_x = this.getMapX();
  var map_y = this.getMapY();
    
  if (this.isCharacter()) {
    if (image_name == this.lastImageName) {
      map_x = this.lastMapX;
      map_y = this.lastMapY;
    } else {
      this.lastImageName = image_name;
      this.lastMapX = map_x;
      this.lastMapY = map_y;
      UnZap(this);
    }      
  }
    
  var x = MapToScreenX(Layer, map_x) - GetImageBaseX(image_name);
  var y = MapToScreenY(Layer, map_y) - GetImageBaseY(image_name);
  var image = CacheImage(image_name);
  var width = image.width;
  var height = image.height;

  if (x + width > MapWindow.x 
   && x < MapWindow.x + MapWindow.width
   && y + height > MapWindow.y
   && y < MapWindow.y + MapWindow.height) {
   
    var mask = this.mask;
    if (mask.alpha >= 255 
     && mask.red >= 255 && mask.green >= 255 && mask.blue >= 255) {
      //DebugCall("image.blit", [x, y, mask]);
      image.blit(x, y);
    } else {
      //DebugCall("image.blitMask", [x, y, mask]);
      image.blitMask(x, y, mask);
    }
    
    if (BoxesFlag) {
      OutlinedRectangle(x, y, width, height, BoxesOutlineColor);
    }
  } 
}

Entity.prototype.fadeIn =
function() {
  //DebugCall("Entity.fadeIn");

  var tick_count = this.tickCount - 1;
  
  if (tick_count > 0) {
    this.setTickCount(tick_count);
    var duration = this.getFrameDuration();
    var opacity = 1 - tick_count/duration;
    this.changeOpacity(opacity);
    return false;
  }
  
  // the fade-in is complete
  this.changeOpacity(1);
  return true;
}

Entity.prototype.fadeOut =
function() {
  //DebugCall("Entity.fadeOut", [], Quote(this));

  var tick_count = this.tickCount - 1;
  
  if (tick_count > 0) {
    this.setTickCount(tick_count);
    if (FadeOutFlag) {
      var duration = FadeSeconds*TicksPerSecond;
      var opacity = tick_count/duration;
      this.changeOpacity(opacity);
    }
    return false;
  }

  this.deleteEntity();
  //DebugLog.write(this.name + " deleted by FadeOut()");
  
  return true;
}

// Entity.place() returns true if obstructed or undefined
Entity.prototype.place =
function(es_cm) {
  //DebugCall("Entity.place", [es_cm]);

  if (es_cm == undefined) {
    return true;
  }

  if (typeof(es_cm) != "object"
   || es_cm[0] == undefined
   || es_cm[1] == undefined) {
    Abort("Invalid es_cm in Entity.place: " + Quote(es_cm));
  }

  var map_x = CmToMapX(es_cm[0]);
  var map_y = CmToMapY(es_cm[1]); 
  if (this.isObstructed(map_x, map_y)) {
    return true; 
  }
  
  this.setEsCm(es_cm);
  return false;
}

Entity.prototype.setEsCm =
function(es_cm) {
  //DebugCall("Entity.setEsCm", [es_cm]);

  if (typeof(es_cm) != "object"
   || es_cm[0] == undefined
   || es_cm[1] == undefined) {
    Abort("Invalid es_cm in Entity.setEsCm: " + Quote(es_cm));
  }

  this.esCm = es_cm;  
}

Entity.prototype.setFrameDuration =
function(status, frame, ticks) {
  //DebugCall("Entity.setFrameDuration", [status, frame, ticks]);

  if (typeof(status) != "string") {
    Abort("Invalid status: " + Quote(status));
  }
  if (typeof(frame) != "number" 
   || frame < 0 
   || frame != Math.round(frame)) { 
    Abort("Invalid frame: " + Quote(frame));
  }
  if (typeof(ticks) != "number" || ticks < 0) { 
    Abort("Invalid ticks: " + Quote(ticks));
  }

  this.frameDuration[status + String(frame)] = Math.ceil(ticks);  
}

Entity.prototype.setLayer =
function(layer) {
  //DebugCall("Entity.setLayer", [layer]);

  if (layer != GroundLayer 
   && layer != ObstacleLayer
   && layer != MissileLayer) {
    Abort("Invalid layer in Entity.setLayer: " + Quote(layer));
  }

  this.layer = layer;  
}

Entity.prototype.setMask =
function(mask) {
  //DebugCall("Entity.setMask", [mask]);
  
  if (typeof(mask) != "object"
   || mask.red == undefined
   || mask.green == undefined
   || mask.blue == undefined
   || mask.alpha == undefined) {
    Abort("Invalid mask in Entity.setMask: " + Quote(mask));
  }

  this.mask = mask;
}

Entity.prototype.setSourceName =
function(name) {
  //DebugCall("Entity.setSourceName", [name], Quote(this));
  
  if (typeof(name) != "string") {
    Abort("Invalid name: " + Quote(name));
  }

  this.sourceName = name;
}

Entity.prototype.setTickCount =
function(ticks) {
  //DebugCall("Entity.setTickCount", [ticks], Quote(this));
  
  if (typeof(ticks) != "number" || ticks < 0) {
    Abort("Invalid ticks: " + Quote(ticks));
  }

  this.tickCount = ticks;
}

Entity.prototype.setVersion =
function(version) {
  //DebugCall("Entity.setVersion", [version]);

  if (version == undefined) {
    version = "r"; // default
  }
  if (typeof(version) != "string") {
    Abort("Invalid version: " + Quote(version));
  }
  
  this.version = version;
}


// read-only functions

// distance from an entity to a fixed location
//   (based on "squared physical" distance from the base center)

Entity.prototype.cm2 = 
function(es_cm) {
  //DebugCall("Entity.cm2", [es_cm], Quote(this));

  var ew_cm = this.esCm[0] - es_cm[0];
  var ns_cm = this.esCm[1] - es_cm[1];
  
  var cm2 = Norm2(ew_cm, ns_cm);
  return cm2;
}

Entity.prototype.describe =
function() {
  //DebugCall("Entity.describe", [], Quote(this));

  var desc = "is ";
  var model = this.model;
  var model_desc = DescribeModel(model);
  var team = this.team;
  
  switch (model) {
    case "footprint":
      desc += "a " + model_desc + " left by "
           + CharacterName(this.sourceName) + ".";
      break;
      
    case "gonfalon":
      desc += "the " + team.name + " " + model_desc + " associated with "
           + this.references.describe() + ".";
      break;
      
    case "pillar":
    case "tree-bare":
    case "tree-leafy":
      desc += "a " + model_desc + ".";
      break;

    case "arrow":
      desc += "a " + team.name + " " + model_desc + " that "
           + this.sourceName
           + " shot at " + this.targetName
           + ". It is " + DescribeStatus(this.status) + ".";           
      break;
      
    case "portal":
      desc += "a " + model_desc + " for teleporting " + team.name
           + " warriors to the battlefield. It is "
           + this.describeDamage() + " and "
           + DescribeProducts(this.product) + ".";
      break;
      
    default:
      Abort("Unknown model in Entity.describe(): " + Quote(model));
  }
  
  return desc;
}

Entity.prototype.dist2 =
function(map_xy) {
  //DebugCall("Entity.dist2", map_xy, Quote(this));

  var image_name = this.getImageName();
  var image = CacheImage(image_name);

  var xr = image.width/2;
  var yr = image.height/2;

  var x = this.getMapX() - GetImageBaseX(image_name) + xr;
  var y = this.getMapY() - GetImageBaseY(image_name) + yr;
    
  var dx = (map_xy[0] - x)/xr;
  var dy = (map_xy[1] - y)/yr;
    
  var dist2 = Norm2(dx, dy);
  return dist2;
}

Entity.prototype.entityCm =
function(to) {
  //DebugCall("Entity.entityCm", [to], Quote(this));
  
  if (typeof(to) == "string") {
    to = AllEntities.find(to);
  }

  var ew_cm = to.esCm[0] - this.esCm[0];
  var ns_cm = to.esCm[1] - this.esCm[1];

  var cm = Norm(ew_cm, ns_cm);
  return cm;
}

Entity.prototype.entityDirection =
function(to, resolution) {
  //DebugCall("Entity.entityDirection", [to, resolution]);

  if (typeof(to) == "string") {
    to = AllEntities.find(to);
  }

  var east_cm = to.esCm[0] - this.esCm[0];
  var south_cm = to.esCm[1] - this.esCm[1];
  
  var direction = DirectionN(east_cm, south_cm, resolution);
    
  //DebugLog.write("Entity.entityDirection() returns " + Quote(direction));
  return direction;
}

Entity.prototype.drawObstructionBase =
function() {
  //DebugCall("Entity.drawObstructionBase");

  var r = this.getBaseCm()/2;
    
  var left_cm = this.esCm[0] - r;
  var right_cm = this.esCm[0] + r;
  var top_cm = this.esCm[1] - r;
  var bottom_cm = this.esCm[1] + r;
  
  var x1 = MapToScreenX(this.layer, CmToMapX(left_cm));
  var x2 = MapToScreenX(this.layer, CmToMapX(right_cm));
  var y1 = MapToScreenY(this.layer, CmToMapY(top_cm));
  var y2 = MapToScreenY(this.layer, CmToMapY(bottom_cm));
  var width = x2 - x1;
  var height = y2 - y1;
  
  var color = ObstructionBasesOutlineColor;
  OutlinedRectangle(x1, y1, width, height, color);
}

Entity.prototype.drawSilhouette =
function() {
  //DebugCall("Entity.drawSilhouette");

  if (!this.isVisible()) {
    return;
  }
  var image_name = this.getImageName();
  var image = CacheImage(image_name);

  var map_x = this.getMapX();
  var map_y = this.getMapY();
  if (this.isCharacter()) {
    map_x = this.lastMapX;
    map_y = this.lastMapY;
  }
    
  var x = MapToScreenX(Layer, map_x) - GetImageBaseX(image_name);
  var y = MapToScreenY(Layer, map_y) - GetImageBaseY(image_name);

  if (x + image.width > MapWindow.x 
   && x < MapWindow.x + MapWindow.width
   && y + image.height > MapWindow.y
   && y < MapWindow.y + MapWindow.height) {

    var color;
    if (this.team == Red) {
      color = CreateColor(255, 0, 0);
    } else if (this.team == Blue) {
      color = CreateColor(0, 0, 255);
    } else {
      color = CreateColor(255, 255, 255);
    }
  
    Silhouette(x, y, image, color);
  } 
}

Entity.prototype.getBaseCm =
function() {
  //DebugCall("Entity.getBaseCm");
  
  switch(this.model) {
    case "archer":
    case "elf":
    case "knight":
    case "sands":
      return (this.status == "dis") ? 0 : 56;
    case "arrow":
    case "footprint":
    case "origin":
    case "portal":
    case "gonfalon":   
      return 0;
    case "tree-bare":
    case "tree-leafy":
      return 42;
    case "pillar":
      return 62;
  }
  Abort("Unknown model: " + Quote(this.model));
}

Entity.prototype.getDebugDescription =
function() {
  //DebugCall("Entity.getDebugDescription", [], Quote(this));

  var separator = "\n";

  var text = "name=" + Quote(this.name);
  text += separator + "model=" + Quote(this.model);
  text += separator + "status=" + Quote(this.status);
  text +=            " motion=" + Quote(this.getMotion());
  text += separator + "direction=" + Quote(this.getDirectionName());
  text +=            " frame=" + Quote(this.frame);
  text +=            " tickCount=" + Quote(this.tickCount);
  text += separator + "screen x=" + Quote(this.getScreenX());
  text +=                   " y=" + Quote(this.getScreenY());
  text += separator + "map x=" + Quote(this.getMapX());
  text +=                " y=" + Quote(this.getMapY());
  text += separator + "east_cm=" + Quote(this.esCm[0]);
  text +=            " south_cm=" + Quote(this.esCm[1]);
  text +=            " layer=" + Quote(this.layer);
  text += separator + "mask=" + Quote(this.mask);
  text += separator + "isCharacter=" + Quote(this.isCharacter());
  text += separator + "isObstacle=" + Quote(this.isObstacle());
  text += separator + "isPortal=" + Quote(this.isPortal());

  return text;
}

Entity.prototype.getDirectionName =
function() {
  //DebugCall("Entity.getDirectionName", [], Quote(this));
  
  return "north"; // default direction
}

Entity.prototype.getFrameDuration =
function() {
  //DebugCall("Entity.getFrameDuration", [], Quote(this));

  var status = this.status;
  var frame = this.frame;
  //DebugLog.write(" status=" + Quote(status) + " frame=" + Quote(frame));
  
  var duration = this.frameDuration[status + String(frame)];
  
  //DebugLog.write(" Entity.getFrameDuration() returns " + Quote(duration));
  return duration;
}

Entity.prototype.getImageName =
function() {
  //DebugCall("Entity.getImageName", [], Quote(this));
  
  var direction_name = this.getDirectionName();
  var motion = this.getMotion();
  
  var directory = "frames/" + String(ResolutionCm) + "cm";
  directory += "/" + this.version;
  if (this.team != undefined) {
    directory += this.team.name + "-";
  }
  directory += this.model;
  
  var md;
  if (motion != "") {
    md = motion + "-" + direction_name;
  } else {
    md = direction_name;
  }
  
  var frame_file = md + String(this.frame) + ".png";
  var image_name = directory + "/" + frame_file;       

  return image_name;
}
 
Entity.prototype.getLastFrame =
function() {
  //DebugCall("Entity.getLastFrame", [], "name=" + this.name);

  var num_frames = this.numFrames();
  var last_frame = num_frames - 1;
  
  return last_frame;
}

Entity.prototype.getMapX =
function() {
  //DebugCall("Entity.getMapX");
  
  var map_x = CmToMapX(this.esCm[0]);
  // TODO adjust for altitude
  return map_x;
}

Entity.prototype.getMapY =
function() {
  //DebugCall("Entity.getMapY");
  
  var map_y = CmToMapY(this.esCm[1]);
  return map_y;
}


Entity.prototype.getMotion =
function() {
  //DebugCall("Entity.getMotion", [], Quote(this));

  var status = this.status;
  //DebugLog.write(" status=" + Quote(status));

  switch (status) {
    case "":     // a "person" which doesn't move (such as an obstacle or portal)
      return "";
    case "adv":  // character ADVancing (or waiting)
      return "adv";
    case "att":  // character ATTacking
      return "att";
    case "dis":  // DISabled character
      return "dis" + this.dis;
    case "fro":  // FROzen character
      return "fro";     
    case "fly":  // missile FLYing toward target
      return "";
    case "pas":  // missile continuing after PASsing target
      return "";
    case "spa":  // character being SPAwned by portal
      return "fro";
    case "spe":  // SPEnt missile lying on the ground
      return "spe";
    case "wav":  // banner WAVing
      return "";
  }
  Abort("Unexpected status: " + Quote(status));
}

Entity.prototype.getScreenX =
function() {
  //DebugCall("Entity.getScreenX");
  
  var map_x = this.getMapX();
  var screen_x = MapToScreenX(this.layer, map_x);

  return screen_x;
}

Entity.prototype.getScreenY =
function() {
  //DebugCall("Entity.getScreenY");
  
  var map_y = this.getMapY();
  var screen_y = MapToScreenY(this.layer, map_y);

  return screen_y;
}

Entity.prototype.isBanner =
function() {
  //DebugCall("Entity.isBanner");

  return false;
}

Entity.prototype.isCharacter =
function() {
  //DebugCall("Entity.isCharacter");

  return false;
}

Entity.prototype.isCrusher = 
function() {
  //DebugCall("Entity.isCrusher", [], Quote(this));

  return false;
}

Entity.prototype.isEffective =
function() {
  //DebugCall("Entity.isEffective");
  
 return false;  
}

Entity.prototype.isFrozen =
function() {
  //DebugCall("Entity.isFrozen");
  
  return false;  
}

Entity.prototype.isInArea =
function(x1, y1, x2, y2) {
  //DebugCall("Entity.isInArea", [x1, y1, x2, y2]);
  
  var x = this.getMapX();
  var y = this.getMapY();
  
  if ((x < x1) + (x < x2) == 1
   && (y < y1) + (y < y2) == 1) {
    return true;
  } else {
    return false;
  }
}

Entity.prototype.isKiller =
function() {
  //DebugCall("Entity.isKiller");
  
  return false;
}

Entity.prototype.isObstacle =
function() {
  //DebugCall("Entity.isObstacle");

  return false;
}

Entity.prototype.isObstructed =
function(map_x, map_y) {
  //DebugCall("Entity.isObstructed", [map_x, map_y], Quote(this));

  if (map_x < 1 || map_y < 1 || map_x >= MapWidth - 1 || map_y >= MapHeight - 1) {
    //DebugLog.write(this.name + " is obstructed by map-edge");
    return true;
  }
 
  if (this.layer != ObstacleLayer) {
    return false;
  }
  
  var base_cm = this.getBaseCm();
  var base_halfheight = PixelsPerNSCm * base_cm / 2;
  var base_halfwidth = PixelsPerEWCm * base_cm / 2;

  var tile_x1 = Math.floor((map_x - base_halfwidth)/GetTileWidth());
  var tile_x2 = Math.ceil((map_x + base_halfwidth)/GetTileWidth());
  var tile_y1 = Math.floor((map_y - base_halfheight)/GetTileHeight());
  var tile_y2 = Math.ceil((map_y + base_halfheight)/GetTileHeight());
  for (var y = tile_y1; y < tile_y2; y++) {
    for (var x = tile_x1; x < tile_x2; x++) {
      var tile = GetTile(x, y, ObstacleLayer);
      if (tile < 21 || tile > 100) {
        // tile obstruction
        return true;
      }
    }
  }
  
  var east_cm = MapXToCm(map_x);
  var south_cm = MapYToCm(map_y);
  
  for (var name in AllEntities.members) {
    var entity = AllEntities.members[name];
    if (entity.layer == ObstacleLayer && entity != this) {
      var entity_base_cm = entity.getBaseCm();
      var min_cm = (base_cm + entity_base_cm)/2;
      var ew_cm = Math.abs(east_cm - entity.esCm[0]);
      var ns_cm = Math.abs(south_cm - entity.esCm[1]);
      if (ew_cm < min_cm && ns_cm < min_cm) {
        // entity obstruction
        return true;
      }
    }
  } 
  return false;
}

Entity.prototype.isUp =
function() {
  //DebugCall("Entity.isUp");

  return false;
}

Entity.prototype.isVisible =
function() {
  //DebugCall("Entity.isVisible");
  
  var result = (this.mask.alpha > 0);

  return result;
}

Entity.prototype.numDirections =
function() {
  //DebugCall("Entity.numDirections", [], Quote(this));

  var model = this.model;
    
  switch (model) {
    case "archer":     return 8;
    case "arrow":      return 16;
    case "elf":        return 8;
    case "gonfalon":   return 2;
    case "knight":     return 8;
    case "pillar":     return 1;
    case "portal":     return 1;
    case "sands":      return 8;
    case "tree-bare":  return 1;
    case "tree-leafy": return 1;
  }
  
  Abort("Unexpected model: " + Quote(model));
}

Entity.prototype.numFrames =
function() {
  //DebugCall("Entity.numFrames");
  
  var model = this.model;
  var motion = this.getMotion();
    
  if (model == "archer") {
    switch (motion) {
      case "adv":  return 4;
      case "att":  return 6;
      case "dis":  return 3;
      case "disa": return 3;
      case "disb": return 3;
      case "fro":  return 1;
    }
    
  } else if (model == "arrow") {
    switch (motion) {
      case "":     return 4;
      case "spe":  return 1;
    }

  } else if (model == "elf") {
    switch (motion) {
      case "adv":  return 4;
      case "att":  return 3;
      case "dis":  return 3;
      case "disa": return 3;
      case "disb": return 3;
      case "fro":  return 1;
    }

  } else if (model == "gonfalon") {
    return 5;
    
  } else if (model == "knight") {
    switch (motion) {
      case "adv":  return 4;
      case "att":  return 8;
      case "dis":  return 5;
      case "disa": return 5;
      case "disb": return 5;
      case "fro":  return 1;
    }
    
  } else if (model == "pillar") {
    return 1;

  } else if (model == "portal") {
    return 1;

  } else if (model == "sands") {
    switch (motion) {
      case "adv":  return 4;
      case "att":  return 4;
      case "dis":  return 3;
      case "disa": return 3;
      case "disb": return 3;
      case "fro":  return 1;
    }

  } else if (model == "tree-bare" || model == "tree-leafy") {
    return 1;
    
  } else {
    Abort("Invalid model: " + Quote(model));
  }
  
  Abort("Invalid motion: " + Quote(motion) + " for model " + Quote(model));
}

// find the east-south offset in cm to the center of a locus
Entity.prototype.offsetEsCm =
function(locus) {
  //DebugCall("Entity.offsetEsCm", [locus], Quote(this));

  if (typeof(locus) != "string") {
    Abort("Invalid locus: " + Quote(locus));
  }
  
  var words = locus.split(LocusDivider);
  var base_name = words[0];

  var east_cm = 0;
  if (words[1] != undefined && words[1] != "") {
    east_cm = Number(words[1])/10;
  }
  
  var south_cm = 0;
  if (words[2] != undefined && words[2] != "") {
    south_cm = Number(words[2])/10;
  }

  if (base_name != this.name) {
    if (base_name != "" && base_name != "origin") {
      var base = AllEntities.find(base_name);
      east_cm += base.esCm[0];
      south_cm += base.esCm[1];
    }
    east_cm -= this.esCm[0];
    south_cm -= this.esCm[1];
  }    
  
  var result = [east_cm, south_cm]; 

  return result;
}

Entity.prototype.quote =
function() {
  return "Entity{name:" + Quote(this.name) + "}";
}