// character.js
// character routines for battle simulation (Sphere)

/*  Copyright (C) 2008-2009  Stephen R. Gold

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/

// A Character is a Mobile (see mobile.js) that represents a creature.
// There are four character models:  {archer, elf, heavy, sands}.

// A Character has the following properties:
//  + armor          fraction of its {front, left, rear, right} side that is protected by armor
//  + frameDuration  number of ticks that each frame of its spriteset is displayed
//  + goal           locus it wants to move to
//  + lookFlag       if "yes," update its goal and target on the next tick
//  + mission        its array of orders
//   + all the properties of a team person.

RequireScript("entities/banner.js");
RequireScript("entities/mobile.js");
RequireScript("entities/portal.js");
RequireScript("locus.js");
RequireScript("orders.js");
RequireScript("utilities/list.js");
RequireScript("utilities/math.js");
RequireScript("utilities/random.js");

// configuration

// default missions

Blue.archerMission = OptionalEnum("Blue.archerMission", 
  "defend.source || approach.portal",
     SampleMissions, "Blue.archers > 0 || Blue.portals > 0");
Blue.elfMission = OptionalEnum("Blue.elfMission", 
  "defend.source || approach.portal",
    SampleMissions, "Blue.elfs > 0 || Blue.portals > 0");
Blue.knightMission = OptionalEnum("Blue.knightMission", 
  "engage.any", 
    SampleMissions, "Blue.knights > 0 || Red.portals > 0");
Blue.sandsMission = OptionalEnum("Blue.sandsMission", 
  "engage.any", 
    SampleMissions, "Blue.sands > 0 || Blue.portals > 0");
  
Red.archerMission = OptionalEnum("Red.archerMission", 
  "defend.source || approach.portal", 
    SampleMissions, "Red.archers > 0 || Red.portals > 0");
Red.elfMission = OptionalEnum("Red.elfMission", 
  "defend.source || approach.portal", 
    SampleMissions, "Red.elfs > 0 || Red.portals > 0");
Red.knightMission = OptionalEnum("Red.knightMission", 
  "engage.any", 
    SampleMissions, "Red.knights > 0 || Red.portals > 0");
Red.sandsMission = OptionalEnum("Red.sandsMission", 
  "engage.any", 
    SampleMissions, "Red.sands > 0 || Red.portals > 0"); 

// max/initial hit points
ArcherHits = OptionalEnum("ArcherHits", 1, [1, 2, 3],
  "Red.archers > 0 || Red.portals > 0 || Blue.archers > 0 || Blue.portals > 0");
ElfHits = OptionalEnum("ElfHits", 1, [1, 2, 3],
  "Red.elfs > 0 || Red.portals > 0 || Blue.elfs > 0 || Blue.portals > 0");
KnightHits = OptionalEnum("KnightHits", 1, [1, 2, 3],
  "Red.knights > 0 || Red.portals > 0 || Blue.knights > 0 || Blue.portals > 0");
SandsHits = OptionalEnum("SandsHits", 1, [1, 2, 3],
  "Red.sands > 0 || Red.portals > 0 || Blue.sands > 0 || Blue.portals > 0");

// if true, leave a footprint after each step
FootprintsFlag = PlayerFlag("Footprints", false);

// sim seconds to keep sliding (used in obstacle avoidance)
Persistence = Enum("Persistence", 3, [0, 1, 2, 3, 4, 5, 10]);

// sim seconds before thaw
FreezeSeconds = Enum("FreezeSeconds", 4, [2, 3, 4, 6, 8, 10, 15]);

// colors

BoostBoltColor = CreateColor(150, 0, 200); // purple
FreezeBoltColor = CreateColor(0, 0, 0);    // black

AllSides = ["front", "left", "right", "rear"];

// constructor

function Character(team, model, status, mission) {
  if (this instanceof Character == false) {
    return new Character(team, model, status, mission);
  }
  //DebugCall("Character", [team, model, status, mission]);

  if (team == undefined) {
    return this;
  }
  
  if (status == undefined) {
    // default initial status is ADVancing
    status = "adv";
  }
  var version = "r"; // rendered version is the default  
  Mobile.apply(this, [team, model, "", version]);
  
  var fade_ticks = FadeSeconds*TicksPerSecond;
  var freeze_ticks = FreezeSeconds*TicksPerSecond

  var attack_speed = Uniform(0.5, 1.5)/TicksPerSecond;
  var degrees_per_second;   // max turn rate
  var cm_per_second;        // max footspeed, in cm/s
  var max_hp;               // max damage before disabled
  var max_range_cm;         // max attack range, in centimeters
  var spawn_seconds;        // time needed to spawn

  if (model == "archer") { // archer character
    if (mission == undefined) {
      mission = team.archerMission;
    }
    
    degrees_per_second = Uniform(100, 140);
    cm_per_second = Uniform(3*30, 6*30);
    max_hp = ArcherHits;
    max_range_cm = Uniform(20*30, 40*30);
    spawn_seconds = 5;

    // unarmored
    this.noArmor();

    // set ticks spent with each frame    

    this.setFrameDuration("adv", 0, 0.1*TicksPerSecond);
    this.setFrameDuration("adv", 1, 0.4*TicksPerSecond);
    this.setFrameDuration("adv", 2, 0.1*TicksPerSecond);
    this.setFrameDuration("adv", 3, 0.4*TicksPerSecond);

    this.setFrameDuration("att", 0, 0.5/attack_speed);
    this.setFrameDuration("att", 1, 0.5/attack_speed);
    this.setFrameDuration("att", 2, 0.5/attack_speed);
    this.setFrameDuration("att", 3, 0.5/attack_speed);
    this.setFrameDuration("att", 4, 0.5/attack_speed);
    this.setFrameDuration("att", 5, 1);
    
    this.setFrameDuration("dis", 0, 0.6*TicksPerSecond);
    this.setFrameDuration("dis", 1, 0.2*TicksPerSecond);
    this.setFrameDuration("dis", 2, fade_ticks);

  } else if (model == "elf") { // elf character with magic wand
    if (mission == undefined) {
      mission = team.elfMission;
    }
    
    degrees_per_second = Uniform(120, 160);
    cm_per_second = Uniform(5*30, 7*30);
    max_hp = ElfHits;
    max_range_cm = 41*30; // slightly farther than any archer
    spawn_seconds = 10;

    // unarmored
    this.noArmor();

    // set ticks spent with each frame    

    this.setFrameDuration("adv", 0, 0.1*TicksPerSecond);
    this.setFrameDuration("adv", 1, 0.4*TicksPerSecond);
    this.setFrameDuration("adv", 2, 0.1*TicksPerSecond);
    this.setFrameDuration("adv", 3, 0.4*TicksPerSecond);
    
    this.setFrameDuration("att", 0, 0.2*TicksPerSecond);
    this.setFrameDuration("att", 1, 0.6*TicksPerSecond);
    this.setFrameDuration("att", 2, 0.2*TicksPerSecond);

    this.setFrameDuration("dis", 0, 0.6*TicksPerSecond);
    this.setFrameDuration("dis", 1, 0.2*TicksPerSecond);
    this.setFrameDuration("dis", 2, fade_ticks);
    
  } else if (model == "knight") { // knight character
    if (mission == undefined) {
      mission = team.knightMission;
    }
    
    degrees_per_second = Uniform(45, 60);
    cm_per_second = Uniform(3*30, 4*30);
    max_hp = KnightHits;
    max_range_cm = Uniform(5*30, 7*30);
    spawn_seconds = 15;
    
    // well-protected by armor on all sides
    this.setArmor("front", 0.95);
    this.setArmor("left", 0.98);
    this.setArmor("right", 0.98);
    this.setArmor("rear", 0.9);

    // set ticks spent with each frame    

    this.setFrameDuration("adv", 0, 0.1*TicksPerSecond);
    this.setFrameDuration("adv", 1, 0.4*TicksPerSecond);
    this.setFrameDuration("adv", 2, 0.1*TicksPerSecond);
    this.setFrameDuration("adv", 3, 0.4*TicksPerSecond);

    this.setFrameDuration("att", 0, 0.2/attack_speed);
    this.setFrameDuration("att", 1, 0.2/attack_speed);
    this.setFrameDuration("att", 2, 0.2/attack_speed);
    this.setFrameDuration("att", 3, 0.2/attack_speed);
    this.setFrameDuration("att", 4, 0.2/attack_speed);
    this.setFrameDuration("att", 5, 0.2/attack_speed);
    this.setFrameDuration("att", 6, 0.2/attack_speed);
    this.setFrameDuration("att", 7, 0.2/attack_speed);
    
    this.setFrameDuration("dis", 0, 0.5*TicksPerSecond);
    this.setFrameDuration("dis", 1, 0.5*TicksPerSecond);
    this.setFrameDuration("dis", 2, 0.5*TicksPerSecond);
    this.setFrameDuration("dis", 3, 0.5*TicksPerSecond);
    this.setFrameDuration("dis", 4, fade_ticks);

  } else if (model == "sands") { // sword-and-shield character
    if (mission == undefined) {
      mission = team.sandsMission;
    }
    
    degrees_per_second = Uniform(90, 120);
    cm_per_second = Uniform(4*30, 6*30);
    max_hp = SandsHits;
    max_range_cm = Uniform(3*30, 5*30);
    spawn_seconds = 4;
    
    // protected by shield and helm, especially on left and front sides
    this.setArmor("front", 0.75);
    this.setArmor("left", 0.75);
    this.setArmor("right", 0.2);
    this.setArmor("rear", 0.2);

    // set ticks spent with each frame    

    this.setFrameDuration("adv", 0, 0.1*TicksPerSecond);
    this.setFrameDuration("adv", 1, 0.4*TicksPerSecond);
    this.setFrameDuration("adv", 2, 0.1*TicksPerSecond);
    this.setFrameDuration("adv", 3, 0.4*TicksPerSecond);

    this.setFrameDuration("att", 0, 0.3/attack_speed);
    this.setFrameDuration("att", 1, 0.1/attack_speed);
    this.setFrameDuration("att", 2, 0.2/attack_speed);
    this.setFrameDuration("att", 3, 0.4/attack_speed);
    
    this.setFrameDuration("dis", 0, 0.6*TicksPerSecond);
    this.setFrameDuration("dis", 1, 0.2*TicksPerSecond);
    this.setFrameDuration("dis", 2, fade_ticks);

  } else {
    Abort("Unknown model " + Quote(model) + " in NewCharacter()");
  }

  this.changeDirection([0, 0]);
  this.clearLookFlag();
  this.directionalFlag = false;
  this.setFrameDuration("fro", 0, freeze_ticks);
  this.setFrameDuration("spa", 0, spawn_seconds*TicksPerSecond);
  this.setMaxHitPoints(max_hp);
  this.setMaxRangeCm(Math.round(max_range_cm));
  
  this.facepoint = "";
  this.goal = "";
  this.setMission(mission);
  this.setTargetName("");
  
  // set distance travelled in one tick
  var step_cm = cm_per_second/TicksPerSecond;
  this.setStepCm(step_cm);

  // set cosine of max turn angle for one tick
  var max_turn_degrees = degrees_per_second/TicksPerSecond;
  if (max_turn_degrees > 180) {
    max_turn_degrees = 180;
  }
  var min_cos = CosineDegrees(max_turn_degrees);
  this.setMinCos(min_cos);
  
  // initial status, frame, tickCount, and lookFlag
  this.changeActiveStatus(status, "");

  // bookkeeping
  if (this.isUp()) {
    this.setLayer(ObstacleLayer);
    
    team.upCharacters.add(this);
    if (this.isCrusher()) {
      team.numCrushers++;
    }
    
  } else if (this.status == "spa") {
    this.changeOpacity(0);
    this.setLayer(ObstacleLayer);
  }
  
  if (this.targetName == undefined) {
    Abort();
  }
  
  //DebugLog.write(" Character() returns " + Quote(this));  
  return this;
}
Character.prototype = new Mobile();
Character.prototype.constructor = Character;

// destructor

Character.prototype.deleteCharacter =
function() {
  //DebugCall("Character.deleteCharacter", [], Quote(this));
  
  if (this.isUp()) {
    var team = character.team;

    // bookkeeping
    team.upCharacters.remove(character);
    if (this.isCrusher()) {
      team.numCrushers--;
    }
  }
  
  this.deleteEntity();
}

// update functions

Character.prototype.update =
function() {
  //DebugCall("Character.update", [], Quote(this));

  var frame = this.frame;
  var status = this.status;
  var tick_count = this.tickCount;
  
  //DebugLog.write(" frame=" + Quote(frame) + " status=" + Quote(status) + " tick=" + Quote(tick_count));

  // update tick_count and frame
  if (status == "spa") {        // character is being spawned
    this.fadeIn();
    return;
    
  } else if (status == "dis") { // character is disabled
    if (frame < this.getLastFrame()) {
      // falling
      tick_count--;
      if (tick_count > 0) {
        this.setTickCount(tick_count);
      } else {
        this.changeFrame(frame + 1);
      }
    } else {
      // fading
      this.fadeOut();
    }
    return;

   } else if (status == "fro") { // character is temporarily frozen
    tick_count--;
    if (tick_count > 0) {
      this.setTickCount(tick_count);
    } else {   
      // thaw
      this.changeActiveStatus("adv");
      this.update(); // redo
    }
    return; 
  }
  
  if (status != "adv" && status != "att") {
    Abort("unexpected status for " + Quote(this.name) + ": " + Quote(status));
  }

  var look_flag = this.clearLookFlag();

  tick_count--;
  if (tick_count > 0) {
    this.setTickCount(tick_count);
  } else {
    frame = this.changeFrame(frame + 1);
    if (frame == 0) {
      look_flag = true;
    }
  }

  // effective character -- mission determines what it will do
  this.doMission(look_flag);
  this.doTurn();

  if (this.goal == undefined) {
    Abort("Goal not set");
  }
  if (this.targetName == undefined) {
    Abort("Target not set");
  }

  if (tick_count == 0) {
    if (!this.isCrusher()) {
      look_flag = true;  // non-crushers look after every frame
    } else if (this.model == "knight" && status == "att" && frame == 4) {
      look_flag = true;
    }
  }
  
  // update status based on target
  var target_name = this.targetName;
  
  if (look_flag) {
    // update status based on target    
    if (status == "adv" && target_name != "") {
      // have target, so attack
      this.changeActiveStatus("att");
      this.update(); // redo
      return;

    } else if (status == "att" && target_name == "") {
      // no target, so advance
      this.changeActiveStatus("adv");
      this.update(); // redo
      return;
    }
  }

  if (status == "adv") {
    this.doAdvance();
    
  } else if (status == "att" && tick_count == 0) {
    this.doAttack();
    this.setSlide(undefined);
  }
}

Character.prototype.doMission =
function(look_flag) {
  //DebugCall("Character.doMission", [look_flag], Quote(this));
  
  if (this.goal == undefined) {
    Abort("Goal not set");
  }
  if (this.targetName == undefined) {
    Abort("Target not set");
  }

  var goal, target_name, facepoint;
  if (look_flag) {
    this.directionalFlag = false;
    
    // make a copy so that purges don't screw up the iterator
    var mission = RemoveIfPresent(undefined, this.mission);

    // go through the orders one by one 
    // keeping the first goal/target_name/facepoint definition
    for (var i in mission) {
      var order = mission[i];
      if (order != undefined) {

        // do a complete update
        var gtf = this.doLook(order, goal != undefined);
        /*
        if (gtf[0] != undefined && gtf[0] != "" && !this.isGoal(gtf[0])) {
          Abort("doLook() returned invalid goal: " + Quote(gtf[0]));
        }
        if (gtf[1] != undefined && gtf[1] != "" && !this.isTarget(gtf[1])) {
          Abort("doLook() returned invalid target: " + Quote(gtf[1]));
        }
        */

        if (goal == undefined && gtf[0] != undefined) {
          this.goalSetBy = order;
          goal = gtf[0];
        }
        if (target_name == undefined) {
          target_name = gtf[1];
        }
        if (facepoint == undefined) {
          facepoint = gtf[2];
        }
      }
    }
  }
  
  if (facepoint == undefined) {
    facepoint = "";
  }
      
  if (goal == undefined) {
    //DebugLog.write("  follow through with previous goal");
    goal = this.goal;
    if (!this.isGoal(goal)) {
      goal = "";
    }
  }
  
  if (target_name == undefined) {
    //DebugLog.write("  follow through with previous target");
    target_name = this.targetName;
    if (!this.isHittableTarget(target_name)) {
      if (goal != "") {
        // try goal
        target_name = LocusBase(goal);
      }
    }
    if (!this.isHittableTarget(target_name)) {
      // try nearest target, even if it's unfamiliar
      var t = this.nearestTarget("engage.any");
      if (t != undefined) {
        target_name = t.name;
      }
    }
    if (!this.isHittableTarget(target_name)) {
      // no target in range
      target_name = "";
    }    
  }
  
  //DebugLog.write(" goal=" + Quote(goal) + " target_name=" + Quote(target_name));
  
  if (goal != undefined && goal != "" && !this.isGoal(goal)) {
    Abort("Character.doMission() returned invalid goal: " + Quote(goal));
  }
  if (target_name != undefined && target_name != "" && !this.isTarget(target_name)) {
    Abort("Character.doMission() returned invalid target: " + Quote(target_name));
  }

  this.goal = goal;
  this.targetName = target_name;
  this.facepoint = facepoint;
}

// find goal and target for character based on an order
Character.prototype.doLook =
function(order, goal_is_set) {
  //DebugCall("Character.doLook", [order, goal_is_set], Quote(this));

  var goal, target_name, facepoint;
  
  var mode = OrderMode(order);
  //DebugLog.write(" mode=" + Quote(mode));
  
  if (mode == "approach.portal") {
    // approach nearest enemy portal to within striking range 
    var portal = this.nearestEnemyPortal();
    if (portal != undefined) {
      goal = AttractingLocus(portal.name, this.maxRangeCm);  
    }
  
  } else if (mode == "avoid.any") {
    // move away from nearest standing character within N cm
    
    var neighbor = this.nearestAble();
    if (neighbor != undefined) {
      var cm = OrderArg(order);
      if (cm == "") {
        cm = 100;
      } else {
        cm = Number(cm);
      }
      goal = RepelLocus(neighbor, cm);
    }

  } else if (mode == "avoid.foe") {
    // move away from nearest standing enemy
    var foe = this.nearestAbleFoe();
    if (foe != undefined) {
      goal = RepelLocus(foe);
    }
    
  } else if (mode == "defend.source" || mode == "engage.any") {
    // raw aggression -- engage nearest target
    var t = this.nearestTarget(mode);
    
    if (t != undefined) {
      goal = AttractingLocus(t.name);
      
      if (this.isHittableTarget(t.name)) {
        // it's hittable, so engage it
        target_name = t.name;
      }
    }
       
  } else if (mode == "directional") {
    // obey arrow keys until Ctrl key is pressed
    var cm = 1000;
    var east_cm = ArrowKeysEast() * cm;
    var south_cm = ArrowKeysSouth() * cm;

    if (east_cm != 0 || south_cm != 0) {
      goal = OffsetLocus(this, [east_cm, south_cm]);
      target_name = ""; // arrow-key movement suppress combat
    }
    if (IsKeyPressed(KEY_CTRL)) {
      // player gives up direct control -- purge
      this.purgeOrder(order);
    } else if (!goal_is_set) {
      this.directionalFlag = true;
    }
    this.setLookFlag();

  } else if (mode == "go") {
    // go to locus then purge this order
    
    var locus = OrderArg(order);
    if (!this.isGoal(locus)) {
      //DebugLog.write(" " + character + " lost its goal: " + Quote(locus));
      this.purgeOrder(order);
      
    } else if (this.isInLocus(locus)) {
      //DebugLog.write(" in locus -- goto is complete -- purge it");
      this.purgeOrder(order);
      var base = LocusBase(locus);
      base = AllEntities.find(base);
      if (base.isBanner()) {
        base.deleteBanner();
      }
      
    } else {
      //DebugLog.write(" go set goal=" + Quote(locus));
      goal = locus;
    }

  } else if (mode == "stand") {
    // stand right where you are -- don't move
    goal = "";

  } else if (mode == "stay") {
    // go to locus and stay there -- useful for follow the leader

    var locus = OrderArg(order);    
    if (!this.isGoal(locus)) {
      //DebugLog.write(" " + this.name + " lost its goal: " + Quote(locus));
      this.purgeOrder(order);
      
    } else if (!this.isInLocus(locus)) {
      goal = locus;  
    }
    
  } else if (mode == "wander") {
    // walk randomly, changing directions every N sim-seconds
    
    var seconds = OrderArg(order);
    if (seconds == "") {
      seconds = 1;
    } else {
      seconds = Number(seconds);
    }

    goal = this.wanderLocus;
    var wander_tick = this.wanderTick;
    var elapsed_ticks = Ticks - wander_tick;
    if (goal == undefined || elapsed_ticks > seconds*TicksPerSecond) {
      // pick new goal locus: random point 5 km away
      var radius_cm = 5e5;
      var east_cm = this.esCm[0];
      var south_cm = this.esCm[1];
      var z = Uniform(0, 2*Math.PI);
      east_cm += radius_cm * Math.sin(z);
      south_cm += radius_cm * Math.cos(z);
      goal = FixedLocus([east_cm, south_cm]);
      this.wanderLocus = goal;
      this.wanderTick = Ticks;
    }
    
  } else {
    Abort("Unknown order mode " + Quote(mode) + " in Character.doLook()");
  }

  /*DebugLog.write(" Character.doLook() returns goal=" + Quote(goal) 
      + " target_name=" + Quote(target_name) 
      + " facepoint=" + Quote(facepoint));
   */
  var gtf = [goal, target_name, facepoint];  
  return gtf;
}


Character.prototype.doTurn =
function() {
  //DebugCall("Character.doTurn");

  // face target if any
  var facepoint = this.targetName;
  if (facepoint == "") {
    // no target, so face goal
    facepoint = this.goal;
  }
  if (facepoint == "") {
    // no goal or target, so face facepoint
    facepoint = this.facepoint;
  }
  if (facepoint == "") {
    // nothing to do
    return;
  }

  // get direction and distance to nearest point in facepoint locus
  var dir_dis = LocusNearestDD(this.name, facepoint);
  var face_cm = dir_dis[1];
  if (face_cm == 0) {
    return;
  }
  
  var face_es = dir_dis[0];
  //DebugLog.write(" facepoint direction es=" + QuoteList(face_es));

  // find turn angle to face the facepoint
  var east = this.es[0];
  var south = this.es[1];
  var cos = CosBetween(face_es[0], face_es[1], east, south);
  var sin = SinBetween(face_es[0], face_es[1], east, south);
 
  //DebugLog.write(" turn sin=" + Quote(sin));
  //DebugLog.write(" turn cos=" + Quote(cos));
  
  // limit the turn angle
  var min_cos = this.minCos;
  var sin_turn, cos_turn;
  if (cos < min_cos) {
    cos_turn = min_cos;
    sin_turn = GetSign(sin) * Math.sqrt(1 - min_cos*min_cos);
  } else {
    cos_turn = cos;
    sin_turn = sin;
  }

  // set new (facing) direction
  this.turn(sin_turn, cos_turn);
}

Character.prototype.doAdvance =
function() {
  //DebugCall("Character.doAdvance", [], Quote(this));
  
  var goal = this.goal;
  if (goal == "") {
    this.changeFrame(0); // revert to resting
    this.setLookFlag();
    return;
  }
  
  // get offset and distance to center of goal locus
  var dir_dis = LocusNearestDD(this.name, goal);
  var goal_es = dir_dis[0];
  var goal_cm = dir_dis[1];
  
  if (goal_cm == 0) {
    //DebugLog.write(" " + this.name + " already inside goal locus.");
    return;
  }

  // How far can it go without overshooting?
  var io = LocusRadiiCm(goal);
  var inner_cm = io[0];
  var outer_cm = io[1];
  var thickness_cm = outer_cm - inner_cm;
  goal_cm += thickness_cm;
  //DebugLog.write(" " + this.name + " goal_cm=" + Quote(goal_cm));
  
  // adjust when not facing the goal directly
  var move_cm;
  if (goal_es == undefined) {
    // direction unimportant - send it in the direction it's facing
    goal_es = this.es;
    move_cm = goal_cm;

  } else {
    // check for forward progress
    //DebugLog.write("  " + this.name + " direction " + Location(east, south));

    var dot = Dot(this.es[0], this.es[1], goal_es[0], goal_es[1]);
    if (dot <= 0) {
      //DebugLog.write(" " + this.name + " facing wrong way -- don't move, just turn");
      return;
    }
    move_cm = goal_cm * dot/NormV(goal_es);
  }
  move_cm = Math.min(move_cm, this.stepCm);

  if (FootprintsFlag) {
    var frame = this.frame;
    var tick_count = this.tickCount;
    //DebugLog.write("  " + this.name + " footprint frame=" + Quote(frame) + ", tick_count=" + Quote(tick_count));
    if (tick_count == 1 && (frame == 0 || frame == 2)) {
      this.leaveFootprint();
    }
  }

  var east_cm = this.esCm[0];
  var south_cm = this.esCm[1];
  
  // move in the direction it's facing
  //DebugLog.write("  " + this.name + " starting from " + Location(east_cm, south_cm));

  // try moving forward the full distance
  var try_east_cm = east_cm + move_cm*this.es[0];
  var try_south_cm = south_cm + move_cm*this.es[1];
  if (!this.place([try_east_cm, try_south_cm])) {
    // success - done!
    return;
  }

  // movement is obstructed
  
  var slide = this.slide;
  if (slide != undefined) {
    // if it slid last time it met an obstruction, 
    //  then usually slide that way again
    
    //DebugLog.write(" " + Quote(this.name) + " old slide=" + Quote(slide));
    if (Persist()) {
      var try_east_cm = east_cm + move_cm*Easting(slide);
      var try_south_cm = south_cm + move_cm*Southing(slide);
      if (!this.place([try_east_cm, try_south_cm])) {
        return;
      }
    }
  }
  
  // can't move that far -- move as far as possible
  move_cm = this.unobstructedCm();
  //DebugLog.write(" obstructed move_cm = " + Quote(move_cm));
  if (move_cm <= 0) {
    east_cm += move_cm*this.es[0];
    south_cm += move_cm*this.es[1];
    this.place([east_cm, south_cm]);
  }
  
  var x = this.getMapX();
  var y = this.getMapY();

  slide = undefined;
  var dir_sequence = SlideSequence(goal_es);  
  for (var i in dir_sequence) {
    var dir = dir_sequence[i];
    
    var easting = Easting(dir);
    var southing = Southing(dir);

    var try_x = x + easting;
    var try_y = y + southing;
    if (!this.isObstructed(try_x, try_y)) {
      // can move at least one pixel in this direction
      
      // try to use up its movement allowance
      var step_cm = this.stepCm - move_cm;
      var try_x = CmToMapX(east_cm + step_cm*easting);
      var try_y = CmToMapY(south_cm + step_cm*southing);
      var move_cm = step_cm;
      if (this.isObstructed(try_x, try_y)) {
        // can't move that far -- move as far as possible  
        move_cm = this.unobstructedCm([east_cm, south_cm], 
          [easting, southing], step_cm);
      }
      //DebugLog.write(" can slide " + Quote(move_cm) + " cm " + dir);
      
      east_cm += move_cm*easting;
      south_cm += move_cm*southing;
      if (move_cm > 0) {
        slide = dir;
      }
      break;
    }
  }

  this.place([east_cm, south_cm]);
  this.setSlide(slide);
  //DebugLog.write(" new slide=" + Quote(slide)); 
}

Character.prototype.doAttack =
function() {
  //DebugCall("Character.doAttack", [], Quote(this));

  var target_name = this.targetName;
  if (target_name == undefined || target_name == "") {
    return;
  }
  var target = AllEntities.find(target_name);
  if (target == undefined) {
    Abort("Target is not an entity: " + Quote(target_name));
  }

  // attack target
  var frame = this.frame;
  var model = this.model;

  if (model == "archer" && frame == 5) {
    var missile = new Missile("arrow", this, target_name);
    
  } else if (model == "elf" && frame == 2) {
    Zap(this, target);
    this.lastImageName = this.getImageName(); 
    target.freeze();
    
  } else if (model == "knight" && (frame == 3 || frame == 6)) {
    target.hitBy(this);

  } else if (model == "sands" && frame == 1) {
    target.hitBy(this);
  }
} 

// misc update functions

// boost a character's attack speed by 2x
Character.prototype.boost =
function() {
  //DebugCall("Character.boost");
  
  for (var i = 0; i < 8; i++) {
    var key = "att" + String(i);
    var ticks = this.frameDuration[key];
    if (ticks != undefined) {
      this.frameDuration[key] = Math.ceil(ticks/2);
    }
  }
}

Character.prototype.changeActiveStatus =
function(status, dis) {
  //DebugCall("Character.changeActiveStatus", [status, dis], Quote(this));

  this.changeStatus(status, dis);

  // reset frame index
  this.changeFrame(0);

  if (status == "adv" || status == "att") {
    // force a look on the next update
    this.setLookFlag();
  }

  return status;
}

Character.prototype.clearLookFlag =
function() {
  //DebugCall("Character.clearLookFlag", [], Quote(this));

  var look_flag = this.lookFlag;  
  this.lookFlag = false;
    
  return look_flag;
}

// cause a character to defect (change to the opposite team)
Character.prototype.defect =
function() {
  //DebugCall("Character.defect");

  var team_color = this.team.color;

  if (this.isUp()) { 
    this.team.upCharacters.remove(this);
    if (this.isCrusher()) {
      this.team.numCrushers--;
    }
  }
 
  this.team = this.team.opposite;
   
  if (this.isUp()) { 
    this.team.upCharacters.add(this);
    if (this.isCrusher()) {
      this.team.numCrushers++;
    } 
  }
  
  this.updateSource();
}

Character.prototype.deleteBanners =
function() {
  //DebugCall("Character.deleteBanners", [], Quote(this));

  for (var i in this.mission) {
    var order = this.mission[i];
    
    var mode = OrderMode(order);
    if (mode == "go") {
      var locus = OrderArg(order);
      var base_name = LocusBase(locus);
      var base = AllEntities.find(base_name);
      if (base.isBanner()) {
        base.removeReference(this);
      }
    }
  }
}

Character.prototype.disable =
function(cause_name) {
  //DebugCall("Character.disable", [cause_name], Quote(this));
    
  if (!this.isUp()) {
    Abort(Quote(this) + " status=" + Quote(this.status));
  }
  
  var dis = "";
  var cause = AllEntities.find(cause_name);
  if (cause != undefined) {
    if (cause.model == "arrow") {
      dis = "a";
    }
  }
  PlaySoundFx(this.version + "dis" + dis, this);
  this.changeActiveStatus("dis", dis);

  // delete any banners associated with this character
  this.deleteBanners();
    
  this.setLayer(GroundLayer);
  
  // bookkeeping
  this.team.upCharacters.remove(this);
  if (this.isCrusher()) {
    this.team.numCrushers--;
  }
}

// freeze a character
Character.prototype.freeze =
function() {
  //DebugCall("Character.freeze", [], Quote(this));

  if (!this.isFrozen()) {
    this.changeActiveStatus("fro");
  }
}

Character.prototype.noArmor =
function() {
  //DebugCall("Character.noArmor");

  for (var i in AllSides) {
    var side = AllSides[i];
    this.setArmor(side, 0);
  }
}

// purge an order from the character's mission
//  -- doesn't attempt to delete the base person (which may be a banner)
Character.prototype.purgeOrder =
function(order) {
  //DebugCall("Character.purgeOrder", [order]);

  if (typeof(order) != "string" || order == "") {
    Abort("Unexpected order: " + Quote(order));
  }
  
  var list = Remove(order, this.mission)
  this.setMission(list);
}

Character.prototype.purgeOrderIfPresent =
function(order) {
  //DebugCall("Character.purgeOrderIfPresent", [order], Quote(this));

  if (typeof(order) != "string" || order == "") {
    Abort("Unexpected order: " + Quote(order));
  }

  var list = RemoveIfPresent(order, this.mission)
  this.setMission(list);
}

// add a new top-priority order to the character's mission
Character.prototype.pushOrder =
function(order) {
  //DebugCall("Character.pushOrder", [order], Quote(this));

  if (typeof(order) != "string" || order == "") {
    Abort("Unexpected order: " + Quote(order));
  }

  this.mission.unshift(order);
  PlaySoundFx("ack order", this)
}

Character.prototype.setArmor =
function(side, fraction) {
  //DebugCall("Character.setArmor", [side, fraction]);

  if (typeof(side) != "string") {
    Abort("Invalid side in Character.setArmor(): " + Quote(side));
  }
  if (typeof(fraction) != "number"
   || fraction < 0
   || fraction > 1) {
    Abort("Invalid fraction in Character.setArmor(): " + Quote(fraction));
  }

  this.armor[side] = fraction;
}

// set look flag
Character.prototype.setLookFlag =
function() {
  //DebugCall("Character.setLookFlag", [], Quote(this));
  
  this.lookFlag = true;
}

// set mission
Character.prototype.setMission =
function(mission) {
  //DebugCall("Character.setMission", [mission]);
  
  if (typeof(mission) == "string") {
    this.mission = mission.split(OrderListSeparator);
  } else if (mission instanceof Array) {
    this.mission = mission; 
  } else {
    Abort("Invalid mission: " + Quote(mission));
  }
}

// set slide
Character.prototype.setSlide =
function(slide) {
  //DebugCall("Character.setSlide", [slide]);
  
  if (slide != undefined 
   && typeof(slide) != "string") {
    Abort("Invalid slide: " + Quote(slide));
  }
  
  this.slide = slide;
}

Character.prototype.turn =
function(sin, cos) {
  //DebugCall("Character.turn", [sin, cos]);

  if (!(sin >= -1.01 && sin <= 1.01)) {
    Abort("Invalid sin: " + Quote(sin));
  }
  if (!(cos >= -1.01 && cos <= 1.01)) {
    Abort("Invalid cos: " + Quote(cos));
  }
  
  // trim
  if (sin > 1) { sin = 1; }
  if (sin < -1) { sin = -1; }
  if (cos > 1) { cos = 1; }
  if (cos < -1) { cos = -1; }
  
  var old_east = this.es[0];
  var old_south = this.es[1];
  //DebugLog.write("  " + this.name + " old es=" + Location(old_east, old_south));
  
  var east  = cos * old_east + sin * old_south;
  var south = cos * old_south - sin * old_east;

  this.changeDirection([east, south]);
}

Character.prototype.updateSource =
function() {
  //DebugCall("Character.updateSource");
  
  var roster = this.team.allPortals;
  var source = roster.nearest(this.esCm);

  this.sourceName = source.name;
  
  return source;
}

// read-only functions

Character.prototype.describe =
function() {
  //DebugCall("Character.describe", [], Quote(this));

  var desc = "is " + this.name + ", a " + this.team.name + " "
       + DescribeModel(this.model) + ", who is "
       + DescribeStatus(this.status) + ".";

  if (!this.isUp()) {
    return desc;
  }

  desc += " He is " + this.describeDamage() + ".\n\n";
    
  var mission = this.mission;
  var len = mission.length;
  if (len == 1) {
    desc += "His movement orders are " + DescribeOrder(mission[0], "inf") + ".";
  } else {
    desc += "His movement orders are:\n";
    for (var i = 1; i < len; i++) {
      var order = mission[i-1];
      desc += " " + String(Number(i)) + ". "  
           + DescribeOrder(order, "inf") + ", " 
           + DescribeOrderConjunction(order) + "\n";
    }
    var order = mission[len - 1];
    desc += " " + String(len) + ". "  
         + DescribeOrder(order, "inf") + ".";
  }

  return desc;
}

Character.prototype.drawTargetZone =
function(color) {
  //DebugCall("Character.drawTargetZone", [color]);
 
  var half_range_cm = this.maxRangeCm/2;

  var east_cm = this.esCm[0] + this.es[0] * half_range_cm;
  var south_cm = this.esCm[1] + this.es[1] * half_range_cm;

  var map_x = CmToMapX(east_cm);
  var map_y = CmToMapY(south_cm);
  
  var x = MapToScreenX(Layer, map_x);
  var y = MapToScreenY(Layer, map_y);
  var rx = half_range_cm * PixelsPerEWCm;
  var ry = half_range_cm * PixelsPerNSCm;  
    
  OutlinedEllipse(x, y, rx, ry, color);
}

Character.prototype.isCharacter =
function() {
  //DebugCall("Character.isCharacter", [], Quote(this));

  return true;
}

Character.prototype.isCrusher = 
function() {
  //DebugCall("Character.isCrusher", [], Quote(this));

  var model = this.model;
  var result = (model == "sands" || model == "knight");
  
  return result;
}

Character.prototype.isEffective =
function() {
  //DebugCall("Character.isEffective");
  
  var status = this.status;
  var result = (status == "adv" || status == "att");

  return result;  
}

Character.prototype.isFrozen =
function() {
  //DebugCall("Character.isFrozen");
  
  var result = (this.status == "fro");

  return result;  
}

// test whether a locus is a valid goal for character movement
Character.prototype.isGoal =
function(locus) {
  //DebugCall("Character.isGoal", [locus], Quote(this));
  
  if (locus == undefined || locus == "") {
    return false;
  }
  if (typeof(locus) != "string") {
    Abort("Invalid locus: " + Quote(locus));
  }
  
  var base = LocusBase(locus);    
  if (base == "origin" || base == this.name) {
    // fixed point or self-relative locus is okay
    return true;
  }
  base = AllEntities.find(base);
  if (base == undefined) {
    // entity no longer exists
    return false;
  }
  
  if (base.isCharacter()) {
    // characters are valid goals, provided they are not spawning or disabled
    var status = base.status;
    return (status != "spa" && status != "dis");
    
  } else if (base.isPortal()) {
    // portals are valid goals, provided they belong to the enemy
    return base.team != this.team;

  } else if (base.isBanner()) {
    // banners are valid goals, provided they belong to the team
    return base.team == this.team;
  }
  
  // other "people" (footsteps, obstacles, and missiles) are never goals
  return false;
}

Character.prototype.isHittableTarget =
function(locus) {
  //DebugCall("Character.isHittableTarget", [locus], Quote(this));
  
  if (!this.isTarget(locus)) {
    //DebugLog.write(" not a target");
    return false;
  }
  
  // quick checks to see if target is clearly out of range
  var target = LocusBase(locus);
  target = AllEntities.find(target);
  var ew_cm = target.esCm[0] - this.esCm[0];
  if (Math.abs(ew_cm) >= this.maxRangeCm) {
    // out of range
    return false;
  }
  var ns_cm = target.esCm[1] - this.esCm[1];
  if (Math.abs(ns_cm) >= this.maxRangeCm) {
    // out of range
    return false;
  }

  // new method: full circle centered at half-range
  // in the direction attacker is facing
  var half_range_cm = this.maxRangeCm/2;
  var center_east_cm = this.esCm[0] + this.es[0] * half_range_cm;
  var center_south_cm = this.esCm[1] + this.es[1] * half_range_cm;
  ew_cm = target.esCm[0] - center_east_cm;
  ns_cm = target.esCm[1] - center_south_cm;
  
  var distance_cm = Norm(ew_cm, ns_cm);
  var result = (distance_cm <= half_range_cm);
  
  return result;
}

Character.prototype.isKiller =
function() {
  //DebugCall("Character.isKiller");
  
  var model = this.model;
  var result = (model == "archer" || model == "knight" || model == "sands")
            && this.isUp();

  return result;  
}

// test whether a character is in locus
Character.prototype.isInLocus =
function(locus) {
  //DebugCall("Character.isInLocus", [locus], Quote(this));
  
  if (typeof(locus) != "string") {
    Abort("Invalid locus: " + Quote(locus));
  }
  
  var es_cm = this.offsetEsCm(locus);
  var dist_cm = NormV(es_cm);

  var words = locus.split(LocusDivider);
  var inner_cm;
  if (words[3] == undefined || words[3] == "") {
    inner_cm = 0;
  } else {
    inner_cm = Number(words[3]);
  }
  var outer_cm;
  if (words[4] == undefined || words[4] == "") {
    outer_cm = inner_cm + LocusDefaultThicknessCm;
  } else {
    outer_cm = Number(words[4]);
  }

  //DebugLog.write(" inner=" + Quote(inner_cm) + " dist=" + Quote(dist_cm) + " outer=" + Quote(outer_cm));
  
  if (dist_cm > outer_cm || dist_cm < inner_cm) {
    //DebugLog.write(" return false");
    return false;
  } else {
    //DebugLog.write(" return true");
    return true;
  }
}


Character.prototype.isTarget =
function(locus) {
  //DebugCall("Character.isTarget", [locus], Quote(this));
  
  if (locus == undefined || locus == "") {
    return false;
  }
  if (typeof(locus) != "string") {
    Abort("Invalid locus: " + Quote(locus));
  }

  var words = locus.split(LocusDivider);
  var base = words[0];
  var east_cm = words[1];
  var south_cm = words[2];
  
  // base must be the name of an entity
  var target = AllEntities.find(base);
  if (target == undefined) {
    return false;
  }
  
  // offset must be zero
  if (east_cm != undefined && east_cm != "" && Number(east_cm) != 0) {
    return false;
  }
  if (south_cm != undefined && south_cm != "" && Number(south_cm) != 0) {
    return false;
  }
  
  if (this.team == base.team) {
    // teammates are never targets
    return false;
  }
  
  if (target.isCharacter()) {
    // enemy characters are legal targets, provided they are not spawning or disabled
    // or (for elf attacks) frozen
    if (this.model == "elf") {
      return target.isEffective();
    } else {
      return target.isUp();
    }

  } else if (target.isPortal()) {
    // enemy portals are legal targets, but only for crushers
    return this.isCrusher(); 
  }
  
  // other entities are never targets
  return false;
}

Character.prototype.isUp =
function() {
  //DebugCall("Character.isUp");

  var status = this.status;
  var result = (status == "adv" || status == "att" || status == "fro");

  return result;  
}

Character.prototype.nearestAbleFoe =
function() {
  //DebugCall("Character.nearestAbleFoe", [], Quote(this));
  
  var roster = this.team.opposite.roster;  
  var foe = roster.nearest(this.esCm);
  
  return foe;
}

Character.prototype.nearestEnemyPortal =
function() {
  //DebugCall("Character.nearestEnemyPortal", [], Quote(this));

  var opposite = this.team.opposite;
  var portal = opposite.allPortals.nearest(this.esCm);
  
  return portal;
}

Character.prototype.nearestTarget =
function(mode) {
  //DebugCall("Character.nearestTarget", [mode], Quote(this));
  
  // find center of target search area
  var center = this;
  if (mode == "defend.source") {
    // source-centric mode
    var source = AllEntities.find(this.sourceName);
    if (source == undefined) {
      source = this.updateSource();
    }
    if (source != undefined) {
      center = source;
    }
  }

  var include_frozen = (this.model != "elf");
  var include_portals = this.isCrusher();
  var enemy = this.team.opposite;

  var target;
  if (include_frozen) {
    target = enemy.upCharacters.nearest(this.esCm);
  } else {
    target = enemy.upCharacters.nearestThawed(this.esCm);
  }
  
  if (include_portals) {
    target = enemy.allPortals.nearest(this.esCm, target);
  }
  
  return target;
}

// decide whether character should persist or not (checking every tick)
function Persist() {
  //DebugCall("Persist");

  var persist_ticks = Persistence*TicksPerSecond;
  var result = (Uniform(0, persist_ticks) > 1);
  
  return result;
}

Character.prototype.quote =
function() {
  return "Character{name:" + Quote(this.name) + "}";
}

Character.prototype.unobstructedCm =
function(es_cm, es, step_cm) {
  //DebugCall("Character.unobstructedCm", [es_cm, es, step_cm], Quote(this));
    
  if (es_cm == undefined) {
    es_cm = this.esCm;
  }
  if (es == undefined) {
    es = this.es;
  }
  if (step_cm == undefined) {
    step_cm = this.stepCm;
  }
  
  var min_cm = 0;
  var max_cm = step_cm;

  // binary search in direction of motion
  var tolerance_cm = 0.5/PixelsPerEWCm;
  while (max_cm - min_cm > tolerance_cm) {
    //DebugLog.write("  " + this.name + " min=" + Quote(min_cm) + ", max=" + Quote(max_cm));
  
    var try_cm = (min_cm + max_cm)/2;
    
    var try_east_cm = es_cm[0] + try_cm*es[0];
    var try_south_cm = es_cm[1] + try_cm*es[1]; 
    var try_x = CmToMapX(try_east_cm);
    var try_y = CmToMapY(try_south_cm);
    //DebugLog.write("  " + this.name + " try " + Quote(try_cm) + " cm to " + Location(try_x, try_y));
      
    if (this.isObstructed([try_x, try_y])) {
      //DebugLog.write("  obstructed");
      max_cm = try_cm;
    } else {
      //DebugLog.write("  clear");
      min_cm = try_cm;
    }
  }
  
  //DebugLog.write(" " + this.name + " can step " + Quote(min_cm) + " cm but not " + Quote(max_cm) + " cm.");
  return min_cm;
}

function ClickBoost(screen_x, screen_y) {
  //DebugCall("ClickBoost", [screen_x, screen_y]);

  var map_x = ScreenToMapX(Layer, screen_x);
  var map_y = ScreenToMapY(Layer, screen_y);

  var character = FindUp([map_x, map_y], CommandRedFlag, CommandBlueFlag);
  if (character != undefined) {
    TheBolt = Bolt(character, screen_x, BoostBoltColor);
    character.boost();
  }
}

function ClickDefect(screen_x, screen_y) {
  //DebugCall("ClickDefect", [screen_x, screen_y]);

  var map_x = ScreenToMapX(Layer, screen_x);
  var map_y = ScreenToMapY(Layer, screen_y);

  var character = FindUp([map_x, map_y], CommandBlueFlag, CommandRedFlag);
  if (character != undefined) {
    TheBolt = Bolt(character, screen_x);
    character.defect();
  }
}

function ClickDisable(screen_x, screen_y) {
  //DebugCall("ClickDisable", [screen_x, screen_y]);

  var map_x = ScreenToMapX(Layer, screen_x);
  var map_y = ScreenToMapY(Layer, screen_y);

  var character = FindUp([map_x, map_y], CommandBlueFlag, CommandRedFlag);
  if (character != undefined) {
    TheBolt = Bolt(character, screen_x);
    character.disable("click");
  }
}

function ClickFreeze(screen_x, screen_y) {
  //DebugCall("ClickFreeze", [screen_x, screen_y]);
  var map_x = ScreenToMapX(Layer, screen_x);
  var map_y = ScreenToMapY(Layer, screen_y);

  var character = FindUp([map_x, map_y], CommandBlueFlag, CommandRedFlag);
  if (character != undefined) {
    TheBolt = Bolt(character, screen_x, FreezeBoltColor);
    character.freeze();
  }
}