// draw.js
// drawing routines for battle simulation (Sphere)

/*  Copyright (C) 2008-2009  Stephen R. Gold

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
    
// configuration

SingleStepFlag = DebugOnlyFlag("SingleStep", false);

PauseMessage = PlayerEnum("PauseMessage", "Stopped", [ "", "Paused", "Stopped" ]);

RequireScript("briefing.js");
RequireScript("entities/banner.js");
RequireScript("entities/character.js");
RequireScript("overlays/bases.js");
RequireScript("overlays/bolt.js");
RequireScript("overlays/clock.js");
RequireScript("overlays/description.js");
RequireScript("overlays/dtags.js");
RequireScript("overlays/eye.js");
RequireScript("overlays/scoreboard.js");
RequireScript("overlays/tzones.js");
RequireScript("overlays/zap.js");
RequireScript("team.js");
RequireScript("ui/mouse.js");
RequireScript("ui/text_box.js");
RequireScript("utilities/graphics.js");

// styles

MessageStyle = LoadWindowStyle("rounded12.rws");
MessageStyle.thickness = 12;

// colors

MapWindowFillColor = CreateColor(1, 17, 71); // dark blue (to match water on map)
MessageColor = CreateColor(255, 192, 64);    // orange
VectorColor = CreateColor(0, 75, 0);         // dark green
SelectAreaColor = CreateColor(0, 75, 0);     // dark green

// initializations done before the map engine is running
function InitDraw() {
  //DebugCall("InitDraw()");
  
  // minimum size of pixel, in cm
  switch (MapView) {
    case "extreme closeup":  
                     ResolutionCm = 1;   break;
    case "closeup":  ResolutionCm = 1.5; break;
    case "normal":   ResolutionCm = 2;   break;
    case "wide":     ResolutionCm = 4;   break;
    default:
      Abort("Unknown MapView: " + Quote(MapView));
  }
  
  // create IconGrid interfaces
    
  var rect = Screen.clone();
  CreateGrids(rect); 

  // create MapWindow in what's left
   
  MapWindow = Window(rect, MapWindowFillColor);
  MapWindow.click = MapClick;
  ActiveWindows.push(MapWindow);
  
  // lay out MapWindow overlays
  
  //   ... messages in the center
  var msg = rect.overlay(0.50, 0.70, 0, 0.1);
  Message = new TextLine(msg, ShadowColor, MessageColor, undefined, "center");
  Message.setStyle(MessageStyle);
  
  //   ... clocks in the upper right
  var clock_rects = [];
  clock_rects[0] = rect.overlay(0.80, 0.02, 0, 0.07);
  clock_rects[1] = rect.overlay(0.80, 0.11, 0, 0.07);
  clock_rects[2] = rect.overlay(0.80, 0.20, 0, 0.07);
  CreateClocks(clock_rects);

  //   ... scoreboard across the top
  var red_board_rect = rect.overlay(0.28, 0.03, 0, 0.12);
  var blue_board_rect = rect.overlay(0.72, 0.03, 0, 0.12);
  CreateScoreboard(red_board_rect, blue_board_rect);
  
  DebugDescribeEntity = undefined;
  DescribeEntity = undefined;
  SelectedEntities = new Roster("SelectedEntities");
  PersonVectorFlag = false;
  SelectAreaFlag = false;
  
  //DebugLog.write(" leaving InitDraw()");
}

// SimUpdate() is called on every tick
//   (after all the person updates) via SetUpdateScript()
function SimUpdate() {
  //DebugCall("SimUpdate", [], Quote(Ticks));

  // handle recent keypresses
  HandleKeyboard();

  // invoke JavaScript garbage collector
  GarbageCollect();  

  // check for victory conditions
  CheckForVictory();

  if (FillMapWindowFlag) {
    // fill background
    MapWindow.draw();
  }
  
  if (SingleStepFlag) {
    PauseFlag = true;
  }
}

function DrawGroundOverlays() {
  //DebugCall("DrawGroundOverlays");

  for (var name in AllEntities.members) {
    var entity = AllEntities.members[name];
    if (entity.layer == GroundLayer) {
      entity.draw();
    }
  }

  // graphic overlays at ground level
  DrawBases();
  DrawDirectionalTags();
  DrawPersonVectors();
  DrawSelectArea();
  DrawTargetZones();
}

function DrawObstacleOverlays() {
  //DebugCall("DrawObstacleOverlays");

  var list = [];
  
  for (var name in AllEntities.members) {
    var entity = AllEntities.members[name];
    if (entity.layer == ObstacleLayer) {
      list.push(entity);
    }
  }
  
  list.sort(function(a,b){return a.esCm[1]-b.esCm[1]});
  for (var i in list) {
    var entity = list[i];
    entity.draw();
  }
}

function DrawMissileOverlays() {
  //DebugCall("DrawMissileOverlays");

  for (var name in AllEntities.members) {
    var entity = AllEntities.members[name];
    if (entity.layer == MissileLayer) {
      entity.draw();
    }
  }
}

// DrawAll() is called after every frame 
//   (after all layers are rendered) via SetRenderScript()
function DrawAll() {
  //DebugCall("DrawAll");

  if (StartMs == undefined) {
    StartMs = GetTime();
  }
  
  // draw map overlays
  DrawMapWindowOverlays();

  // draw the GUI grids
  DrawGUI();
  
  // draw mouse pointer and handle recent clicks
  var pointer_image;
  if (DescribeEntity == undefined && DebugDescribeEntity == undefined) {
    pointer_image = DefaultPointerImage;
  }
  UpdateMouse(pointer_image);

  // re-center the map for the next render
  CenterMap();

  // ready to update again
  //DebugCall(" leaving DrawAll");
}

// graphics that are overlaid onto the map window
function DrawMapWindowOverlays() {
  //DebugCall("DrawMapWindowOverlays");

  // magical effects
  DrawZaps();
  if (TheBolt != undefined) {
    TheBolt.draw();
  }

  // informational overlays
  DrawEye();
  if (DescribeEntity != undefined) {
    DescribeEntity.drawSilhouette();
    DescribeEntity.drawDescription();
  } else if (DebugDescribeEntity != undefined) {
    DebugDescribeEntity.drawSilhouette();
    DebugDescribeEntity.drawDebugDescription();
  } else {
    for (var name in SelectedEntities.members) {
      var entity = SelectedEntities.members[name];
      entity.drawSilhouette();
    }
    DrawSelectAreaPeople();
  }
  
  // text overlays
  DrawClocks();
  DrawScoreboard();
  if (IsPaused() && PauseMessage.length > 0) {
    Message.draw(PauseMessage);
  }
  //DebugCall("leaving DrawMapWindowOverlays");
}

function ClickSelectAll() {
  //DebugCall("ClickSelectAll");

  SelectedEntities = new Roster("SelectedEntities");
  if (CommandBlueFlag) {
    for (var name in Blue.upCharacters.members) {
      var entity = Blue.upCharacters.members[name];
      if (entity != undefined) {
        SelectedEntities.add(entity);
      }
    }
  }
  if (CommandRedFlag) {
    for (var name in Red.upCharacters.members) {
      var entity = Red.upCharacters.members[name];
      if (entity != undefined) {
        SelectedEntities.add(entity);
      }
    }
  }
}

function ClickSelectArea(screen_x, screen_y) {
  //DebugCall("ClickSelectArea", [screen_x, screen_y]);
  
  SelectedEntities = new Roster("SelectedEntities");

  SelectAreaFlag = true;
  SelectAreaX = ScreenToMapX(Layer, screen_x);
  SelectAreaY = ScreenToMapY(Layer, screen_y);

  EndClickAction = "EndClickSelectArea()";
}

function DrawSelectArea() {
  if (!SelectAreaFlag) {
    return;
  }
  //DebugCall("DrawSelectArea");
  
  var x1 = ScreenToMapX(Layer, GetMouseX());
  var x2 = SelectAreaX;
  if (x1 > x2) {
    var t = x1; x1 = x2; x2 = t;
  }
  var width = x2 - x1;

  var y1 = ScreenToMapY(Layer, GetMouseY());
  var y2 = SelectAreaY;
  if (y1 > y2) {
    var t = y1; y1 = y2; y2 = t;
  }
  var height = y2 - y1;

  x1 = MapToScreenX(Layer, x1);
  y1 = MapToScreenY(Layer, y1);
  OutlinedRectangle(x1, y1, width, height, SelectAreaColor);
}

function DrawSelectAreaPeople() {
  //DebugCall("DrawSelectAreaPeople");
  if (!SelectAreaFlag) {
    return;
  }
  
  var map_x = ScreenToMapX(Layer, GetMouseX());
  var map_y = ScreenToMapY(Layer, GetMouseY());
  var roster = UpInArea(SelectAreaX, SelectAreaY, map_x, map_y);
  
  for (var name in roster.members) {
    var entity = roster.members[name];
    entity.drawSilhouette();   
  }
}

function EndClickSelectArea() {
  //DebugCall("EndClickSelectArea");
  
  var map_x = ScreenToMapX(Layer, GetMouseX());
  var map_y = ScreenToMapY(Layer, GetMouseY());

  SelectedEntities = UpInArea(SelectAreaX, SelectAreaY, map_x, map_y);
  SelectAreaFlag = false;
}

function FindUp(map_xy, blue_flag, red_flag) {
  //DebugCall("FindUp", [map_xy, blue_flag, red_flag]);
  
  var result = undefined;
  
  if (blue_flag) {
    result = Blue.upCharacters.best(map_xy);
  }
  if (red_flag) {
    result = Red.upCharacters.best(map_xy, result);
  }
  
  return result;
}

function UpInArea(x1, y1, x2, y2) {
  //DebugCall("UpInArea", [x1, y1, x2, y2]);
  
  var result = new Roster("result");
  
  if (CommandBlueFlag) {
    for (var name in Blue.upCharacters.members) {
      var entity = Blue.upCharacters.members[name];
      if (entity != undefined && entity.isInArea(x1, y1, x2, y2)) {
        result.add(entity);
      }
    }
  }
  
  if (CommandRedFlag) {
    for (var name in Red.upCharacters.members) {
      var entity = Red.upCharacters.members[name];
      if (entity != undefined && entity.isInArea(x1, y1, x2, y2)) {
        result.add(entity);
      }
    }
  }
  
  return result;
}

function ClickGoto(screen_x, screen_y) {
  //DebugCall("ClickGoto", [screen_x, screen_y]);

  if (!SelectedEntities.isEmpty()) {
    EndClickGoto();
    return;
  }
  
  var map_x = ScreenToMapX(Layer, screen_x);
  var map_y = ScreenToMapY(Layer, screen_y);
  var character = FindUp([map_x, map_y], CommandBlueFlag, CommandRedFlag);
  if (character != undefined) {
    SelectedEntities = new Roster("SelectedEntities");
    SelectedEntities.add(character);
    PersonVectorFlag = true;
    EndClickAction = "EndClickGoto()";
  }
}

function DrawPersonVectors() {
  if (!PersonVectorFlag) {
    return;
  }
  //DebugCall("DrawPersonVectors");

  var x2 = GetMouseX();
  var y2 = GetMouseY();

  for (var name in SelectedEntities.members) {
    var entity = SelectedEntities.members[name];
    if (entity != undefined) {
      var x1 = entity.getScreenX();
      var y1 = entity.getScreenY();
      DrawVector(x1, y1, x2, y2, VectorColor);
    }
  }
}

function EndClickGoto() {
  //DebugCall("EndClickGoto");

  var map_x = ScreenToMapX(Layer, GetMouseX());
  var map_y = ScreenToMapY(Layer, GetMouseY());

  var teams = ["red", "blue"];
  for (var j in teams) {
    var team = teams[j];
   
    var ables = new Roster("ables");
    for (var name in SelectedEntities.members) {
      var entity = SelectedEntities.members[name];
      if (entity != undefined 
       && entity.team.name == team    
       && entity.isUp()) { 
        ables.add(entity);
      }
    }
  
    if (!ables.isEmpty()) {
      var locus;
      if (BannerModel != "none") {
        var banner = AddBanner(ables, map_x + 2*j - 1, map_y);
        locus = AttractingLocus(banner.name, LocusDefaultThicknessCm);
      } else {
        var east_cm = MapXToCm(map_x);
        var south_cm = MapYToCm(map_y);
        locus = FixedLocus([east_cm, south_cm]);
      }
      if (locus != undefined) {
        var order = GoTo(locus);
        for (var name in ables.members) {
          var entity = ables.members[name];
          entity.pushOrder(order);
        }
      }
    }
  }
    
  PersonVectorFlag = false;
  SelectedEntities = new Roster("SelectedEntities");
}

function HandleKeyboard() {
  //DebugCall("HandleKeyboard");

  if (IsKeyPressed(KEY_S)) {   // Stop
    PauseFlag = true;
  }
    
  if (PauseFlag) {
    if (IsKeyPressed(KEY_B)) { // Briefing
      GiveBriefing();
    }
    if (IsKeyPressed(KEY_G)) { // Go
      PauseFlag = false;
    }
    if (IsKeyPressed(KEY_M)) { // Menu
      QuitToMenu();
    }
    if (IsKeyPressed(KEY_R)) { // Restart
      QuitToRestart();
    }
  }
}
