// describe.js
// describe game entities (in a way that makes sense to the player)

/*  Copyright (C) 2008-2009  Stephen R. Gold

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/

RequireScript("entities/banner.js");
RequireScript("entities/character.js");
RequireScript("utilities/direction.js");
RequireScript("utilities/random.js");


function DescribeCm(cm) {
  //DebugCall("DescribeCm", [cm]);
  
  // TODO: option for feet, cubits, whatever
  
  cm = Math.round(Number(cm));
    
  if (cm > 1e5) {
    return String(cm/1e5) + " kilometers";
  } else if (cm > 100) {
    return String(cm/100) + " meters";
  }
  return String(cm) + "centimeters";
}

TeamEntity.prototype.describeDamage =
function() {
  //DebugCall("Team.describeDamage", [], Quote(this));

  var hp = this.hitPoints;
  var max_hp = this.maxHitPoints;
  
  if (this.isCharacter()) {
    if (hp == max_hp) {
      return "at full health";
    } else {
      return "injured";
    }
  } else {
    if (hp == max_hp) {
      return "undamaged";
      
    } else if (max_hp > 0) {
      var percentage = 100*(max_hp - hp)/max_hp;
      return String(Math.round(percentage)) + "% destroyed";
      
    } else {
      Abort("Invalid max_hp: " + Quote(max_hp));
    }
  }
}

function DescribeLocus(locus) {
  //DebugCall("DescribeLocus", [locus]);

  var words = locus.split(LocusDivider);
  var base = words[0];
  for (var i = 1; i <= 4; i++) {
     if (words[i] == undefined || words[i] == "") {
      words[i] = 0;
    } else {
      words[i] = Number(words[i]);
    }
  }
  var east_cm  = words[1];
  var south_cm = words[2];
  var inner_cm = words[3];
  var outer_cm = words[4];

  var center;
  if (base == "origin") {
    center = "the center of the battlefield";
  } else {
    var entity = AllEntities.find(base);
    if (entity.isCharacter()) {
      center = base.describe();
    } else if (entity.isBanner()) {
      center = "his " + DescribeModel(entity.model);
    } else if (entity.isPortal(base)) {
      var dist_cm = NormV(entity.esCm);
      var direction = Direction16(entity.esCm[0], entity.esCm[1]);  
      center = "the portal located " + DescribeCm(dist_cm) + " " + direction 
           + " of the center of the battlefield";
    } else {
      center = base; // could be a deleted portal?
    }
  }

  var location = DescribeOffsetCm(center, east_cm, south_cm);
    
  if (inner_cm == 0) {
    if (outer_cm == 0) {
      return location;
    }
    return "within " + DescribeCm(outer_cm) + " of " + location;

  } else if (inner_cm > 1e5) {
    return "as far as possible from " + location;
         
  }
  
  return DescribeCm(inner_cm) 
         + " to " + DescribeCm(outer_cm) 
         + " away from " + location;
}


function DescribeModel(model) {
  //DebugCall("DescribeModel", [model]);

  switch (model) {
    case "footprint":  return "footprint";
    case "pillar":     return "stone pillar";
    case "tree-bare":  return "leafless tree";
    case "tree-leafy": return "leafy tree";
    case "gonfalon":   return "gonfalon (banner)";
    case "archer":     return "archer";
    case "elf":        return "elf-mage";
    case "knight":     return "armored knight";
    case "sands":      return "sword-and-shield warrior";
    case "arrow":      return "arrow";
    case "portal":     return "portal";
  }
  Abort("Unknown model in DescribeModel(): " + Quote(model));
}

function DescribeOffsetCm(center, east_cm, south_cm) {
  //DebugCall("DescribeOffsetCm", [center, east_cm, south_cm]);
  
  east_cm = Number(east_cm);
  south_cm = Number(east_cm);
  
  if (east_cm == 0 && south_cm == 0) {
    return center;
  }
  
  var dist_cm = Norm(east_cm, south_cm);
  var direction = Direction16(east_cm, south_cm);
  
  return "a point " + DescribeCm(dist_cm) + " " + direction 
       + " of " + center;
}

function DescribeOrder(order, verb_form) {
  //DebugCall("DescribeOrder", [order, verb_form]);

  var desc = "to ";
  var arg1 = OrderArg(order);
  var enemy = "enemy";
  var mode = OrderMode(order);
  switch (mode) {
  
    case "approach.portal":
      desc += "get within striking distance of the nearest "
           + enemy + " " + DescribeModel("portal");
      break;
      
    case "avoid.any":
      desc += Vor("avoid", "upright warrior", arg1);
      break;
      
    case "avoid.foe":
      desc += Vor("avoid", "upright " + enemy + " warrior", arg1);
      break;
 
    case "avoid.hand":
      desc += Vor("avoid", 
              "upright " + enemy + " " + DescribeModel("sands")
                         + " or " + DescribeModel("knight"), arg1);
      break;     
     
    case "defend.source":
      desc += Vor("engage", "target", arg1, "his assigned " + DescribeModel("portal"));
      break;
      
    case "directional":
      desc += "move according to the arrow keys until Ctrl is pressed";
      break;
      
    case "engage.any":
      desc += Vor("engage", "target", arg1);
      break;
      
    case "engage.archer":
      desc += Vor("engage", DescribeModel("archer") + " target", arg1);
      break;
      
    case "engage.elf":
      desc += Vor("engage", DescribeModel("elf") + " target", arg1);
      break;

    case "engage.knight":
      desc += Vor("engage", DescribeModel("knight") + " target", arg1);
      break;

    case "engage.sands":
      desc += Vor("engage", DescribeModel("sands") + " target", arg1);
      break;

    case "go":
      desc += "get to " + DescribeLocus(arg1);
      break;
      
    case "spin":
      desc += "spin clockwise in place";
      break;
      
    case "stand":
      desc += "stay wherever he happens to be";
      break;
      
    case "stay":
      desc += "stay " + DescribeLocus(arg1);
      break;
      
    case "wander":
      desc += "move randomly, changing course every " + arg1 + " seconds";
      break;
    
    default:
      Abort("Unknown mode in DescribeOrder(): " + Quote(mode));
  }
  
  return desc;
}


function Vor(verb, object, range, center) {
  var desc = "";

  if (verb == "avoid") {
    desc += "move away from ";
  } else {
    desc += verb + " ";
  }
  
  if (range != "") {
    desc += "any " + object + " within " + range + " cm";
    if (center != undefined) {
      desc += " of " + center;
    }
  } else if (center != undefined) {
    desc += "the " + object + " nearest to " + center;
  } else {
    desc += "the nearest " + object;
  }
  
  return desc;   
}

function DescribeOrderConjunction(order) {
  //DebugCall("DescribeOrderConjunction", [order]);
  
  var mode = OrderMode(order);
  switch (mode) {
    case "approach.portal":
    case "stay":
      return "and while he's there";
      
    case "avoid.any":
    case "avoid.foe":
    case "avoid.hand":
    case "engage.archer":
    case "engage.elf":
    case "engage.knight":
    case "engage.sands":
      return "or if there's none";
      
    case "defend.source":
    case "engage.any":
      return "or if there's no target";

    case "directional":
      return "and thereafter (or when no key is pressed)";

    case "go":
      return "and once he's been there";

    case "spin":
    case "stand":
    case "wander":
      return "and while doing so";
      
    default:
      Abort("Unknown mode in DescribeOrderConjunction(): " + Quote(mode));
  }
}

function DescribeProductPercentage(pair) {
  //DebugCall("DescribeProductPercentage", [pair]);
  
  var words = pair.split("%");
  desc = words[0] + "% " + DescribeModel(words[1]) + "s";
  
  return desc;
}

function DescribeProducts(product) {
  //DebugCall("DescribeProducts", [product]);
  
  if (IsMix(product)) {
    var result = "set to deliver a random mix consisting of ";        
    var pairs = product.split(MixSeparator);
    pairs.shift();
    var len = pairs.length;
    if (len == 1) {
      result += DescribeProductPercentage(pairs[0]);
    } else {
      var desc_list = [];
      for (var i = 0; i < len-1; i++) {
        desc_list.push(DescribeProductPercentage(pairs[i]));
      }
      result += desc_list.join(", ");
      result += " and " + DescribeProductPercentage(pairs[len-1]);
    }
    
    return result;
  }
  
  switch (product) {
    case "none":
      return "switched off";
  
    case "archer":
    case "elf":
    case "knight":
    case "sands":
      return "set to deliver only " + DescribeModel(product) + "s";

    case "alternate":
      return "set to alternate between " + DescribeModel("sands") 
           + "s and " + DescribeModel("archer") + "s";    
  }
  
  Abort("Unknown product in DescribeProducts(): " + Quote(product));
}


function DescribeStatus(status) {
  //DebugCall("DescribeStatus", [status]);

  switch (status) {
    case "":     return "not doing anything";
    case "adv":  return "up but not fighting";
    case "att":  return "up and fighting";
    case "dis":  return "down and unable to fight";
    case "fro":  return "up but magically frozen";
    case "fly":  return "flying toward its target";
    case "pas":  return "still airborne after missing its target";
    case "spa":  return "in the process of materializing onto the battlefield";
    case "spe":  return "lying spent on the battlefield";
    case "wav":  return "waving in the breeze";
  }
  Abort("Unknown status in DescribeStatus(): " + Quote(status));
}
