// coords.js
// coordinate conversion routines for battle simulation (Sphere)

/*  Copyright (C) 2008  Stephen R. Gold

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
    
RequireScript("utilities/math.js");
// also uses ResolutionCm

// data

var Layer;                  // default map layer where people live
var MapHeight, MapWidth;    // dimensions of the map in pixels
var MapEWCm, MapNSCm;       // dimensions of the map in centimeters

var PixelsPerNSCm, PixelsPerEWCm, PixelsPerVCm; // pixels per centimeter, various directions

// initialization routine, called every time a map is loaded

function InitCoords(default_layer, viewing_angle_degrees) {
  //DebugLog.write("InitCoords()");

  Layer = default_layer;

  // viewing_angle: 0 would be looking straight down from the zenith
  if (!(viewing_angle_degrees > 0 || viewing_angle_degrees <= 90)) {
    Abort("Invalid viewing_angle_degrees in InitCoords: " 
      + Quote(viewing_angle_degrees));
  }
  
  // map is viewed obliquely from the south 
  PixelsPerNSCm = CosineDegrees(viewing_angle_degrees)/ResolutionCm;
  PixelsPerEWCm = 1/ResolutionCm;
  PixelsPerVCm = SineDegrees(viewing_angle_degrees)/ResolutionCm;

  // dimensions of the map in pixels
  MapWidth = GetLayerWidth(Layer) * GetTileWidth();
  MapHeight = GetLayerHeight(Layer) * GetTileHeight();
  
  // dimensions of the map in centimeters
  FieldEWCm = MapWidth/PixelsPerEWCm;
  FieldNSCm = MapHeight/PixelsPerNSCm;
}

// note: "cm" coordinates are measured from the center of the map

function CmToMapX(east_cm) {
  //DebugLog.write("CmToMapX(" + Quote(east_cm) + ")");

  var right = east_cm/FieldEWCm;
  var map_x = Math.round(MapWidth * (0.5 + right));
  
  return map_x;
}

function CmToMapY(south_cm) {
  //DebugLog.write("CmToMapY(" + Quote(south_cm) + ")");

  var down = south_cm/FieldNSCm;
  var map_y = Math.round(MapHeight * (0.5 + down));
  
  return map_y;
}

function MapXToCm(map_x) {
  //DebugLog.write("MapXToCm(" + Quote(map_x) + ")");

  var right = map_x/MapWidth - 0.5;
  var east_cm = right * FieldEWCm;
  
  return east_cm;
}

function MapYToCm(map_y) {
  //DebugLog.write("MapYToCm(" + Quote(map_y) + ")");

  var down = map_y/MapHeight - 0.5;
  var south_cm = down * FieldNSCm;

  return south_cm;
}