//////////////////////////////
// 		Rad_Sound.js v1.50  	//
//      By Radnen 2008      //
/////////////////////////////////////////////
//	Recently added into the RadLib Library //
//////////////////////////////////////////////////////
// You can now use playlists by supplying an array of
// filenames for it to load data into. You can even
// supply an array of descriptions for each song.
//////////////////////////////////////////////////////
// To get full use out of it you may need to overload
// the flipscreen function.
//////////////////////////////////////////////////////
// EX:
// var _FlipScreen = FlipScreen;
// var Audio = new SoundEngine();
// FlipScreen = function()
// {
//		Audio.update();
//		_FlipScreen();
// }
//////////////////////////////////////////////////////

// Declare this only once, and append songs onto it //
function SoundEngine()
{
	if (!this instanceof SoundEngine) return new SoundEngine();
	
	this.fade = 0;
	this.music = null;
	this.sounds = [];
	this.playlist = []; // Set this to an array of sound objects to use the playlist feature.
	this.pauselist = true;
	this.filename = "";
	this.filenames = []; // Playlist filenames //
	this.descriptions = []; // Textual description; name of song //
	this.time = GetTime();
	this.msecs = 0;
	this.current = 0;
	this.curSound = 0;
	// You'll need descriptions for this to do anything! //
	this.displayInformation = false;
	this.font = GetSystemFont();
	this.ticks = 0;
	this.fadeType = LINEAR;
	this.maxTicks = 200;
}

// Sound can be a filename or it can be a sound object.
// However, sound objects lose filename data!
SoundEngine.prototype.play = function(sound, repeat)
{
	if (repeat == undefined) repeat = true;
	// Filename //
	if (typeof sound == "string") {
		if (sound != "" && sound != this.filename) {
			if (this.music != null) this.music.stop();
			this.music = LoadSound(sound);
			this.filename = sound;
		}
	}
	// Object //
	else if (sound != null) {
		if (this.music != null) this.music.stop();
		this.music = sound;
		this.filename = "";
	}
	else Abort("Error: Invalid sound object", "Rad_Sound.js", 74);
	this.music.play(repeat);
	this.fade = 0;
}

SoundEngine.prototype.setVolume = function(num)
{
	if (this.music != null) {
		this.music.setVolume(num);
	}
}

SoundEngine.prototype.incrementVolume = function()
{
	this.music.setVolume(this.music.getVolume()+1);
}

SoundEngine.prototype.decrementVolume = function()
{
	this.music.setVolume(this.music.getVolume()-1);
}

SoundEngine.prototype.setPitch = function(num)
{
	this.music.setPitch(num);
}

SoundEngine.prototype.playSound = function(sound)
{
	if (this.sounds[this.curSound] != undefined) this.sounds[this.curSound].stop();
	this.sounds[this.curSound] = sound;
	this.sounds[this.curSound].play(false);
	this.curSound++;
	if (this.curSound > 16) this.curSound = 0;
}

SoundEngine.prototype.stop = function()
{
	if (this.music != null && this.music.isPlaying()) {
		this.music.stop();
		this.music = null;
		this.filename = "";
	}
}

SoundEngine.prototype.waitForFade = function()
{
	SetFrameRate(GetMapEngineFrameRate());
	while (this.fade != 0) {
		UpdateMapEngine();
		RenderMap();
		FlipScreen();
	}
	SetFrameRate(0);
}

// Use this to add a list of songs by supplying an array of filenames.
SoundEngine.prototype.addPlaylist = function(playlist, descriptions)
{
	if (descriptions != undefined) this.descriptions = descriptions;
	for(var i = 0; i < playlist.length; ++i) {
		this.playlist[i] = LoadSound(playlist[i]);
	}
	this.filenames = playlist;
}

// Call this whenever you want to fade the current music in //
SoundEngine.prototype.fadeIn = function(msecs)
{
	this.music.setVolume(0);
	this.msecs = msecs;
	this.fade = FADE_IN;
	this.time = GetTime();
}

// Call this whenever you want to fade the current music out //
SoundEngine.prototype.fadeOut = function(msecs)
{
	this.music.setVolume(255);
	this.msecs = msecs;
	this.fade = FADE_OUT;
	this.time = GetTime();
}

// Use this to enable the playllist feature.
SoundEngine.prototype.enablePlaylisting = function()
{
	if (this.playlist.length > 0) {
		if (this.filename != "") this.music.stop();
		this.pauselist = false;
		this.playlist[this.current].play(false);
	}
}

// Use this to disable it.
SoundEngine.prototype.disablePlaylisting = function(reset)
{
	this.playlist[this.current].stop();
	this.pauselist = true;
	if(reset != undefined) this.current = 0;
}

// Call this in an update loop //
SoundEngine.prototype.update = function()
{
	if (!this.pauselist) this.updatePlaylist();
	else {
		// Fade In:
		if (this.fade == 1) {
			if (this.time + this.msecs > GetTime()) {
				var X = GetTime() - this.time;
				if (this.fadeType == LINEAR)
					this.music.setVolume(X * 255 / this.msecs);
				else
					this.music.setVolume((Math.log(X)*Math.sin(X)) * 255 / this.msecs);
			}
			else this.fade = 0;
		}
		
		// Fade Out:
		if (this.fade == 2) {
			if (this.time + this.msecs > GetTime()) {
				this.music.setVolume(255 - (GetTime() - this.time) * 255 / this.msecs);
			}
			else this.fade = 0;
		}
	}
}

// Internally used within the update loop //
SoundEngine.prototype.updatePlaylist = function()
{
	if (!this.playlist[this.current].isPlaying()) {
		this.current++;
		if (this.current > this.playlist.length) this.current = 0;
		this.play(this.playlist[this.current], false);
		this.ticks = 0;
	}
	if (this.displayInfo) {
		if (this.ticks < this.maxTicks) {
			this.font.drawText(0, GetScreenHeight()-this.font.getHeight(), "Now Playing: " + this.descriptions[this.current]);
			this.ticks++;
		}
	}
}