//          MoveSys.js v1.0          //
// An Advanced Pixel Movement Engine //
//         Radnen July 2009          //

// Create a MovementQueue //
// Attach an Input //
// Accept Movement Calls //

const COMMAND_MOVE_NORTHEAST = 100;
const COMMAND_MOVE_NORTHWEST = 101;
const COMMAND_MOVE_SOUTHEAST = 102;
const COMMAND_MOVE_SOUTHWEST = 103;

const SCRIPT_ON_TILE_ENTER = 200;
const SCRIPT_ON_TILE_LEAVE = 201;

function MoveSys()
{
	this.moveQueue = [];
	this.input = null;
	this.faced = false;
	this.tileSize = 0;
	this.afterMove = false;
	this.isDiag = false;
	this.keyPaused = [];
	this.onHit = function(){};
	
	BindKey(KEY_UP, '', '');
	this.keyPaused[KEY_UP] = false;
	BindKey(KEY_DOWN, '', '');
	this.keyPaused[KEY_DOWN] = false;
	BindKey(KEY_LEFT, '', '');
	this.keyPaused[KEY_LEFT] = false;
	BindKey(KEY_RIGHT, '', '');
	this.keyPaused[KEY_RIGHT] = false;
}

MoveSys.prototype.processQueue = function()
{
	if (this.moveQueue.length > 0)
	{
		var MoveObject = this.moveQueue.pop();
		var pixels = this.tileSize/GetPersonSpeedX(MoveObject.name);
		switch (MoveObject.command)
		{
			case COMMAND_MOVE_NORTHEAST:
				QueuePersonCommand(MoveObject.name, COMMAND_FACE_NORTHEAST, false);
				if (this.isObstructed(MoveObject.name, MoveObject.command)) return;
				QueuePersonScript(MoveObject.name, GetPersonValue(MoveObject.name, "TileLeaveCode"), true);
				for (var i = 0; i < pixels; ++i)
				{
					QueuePersonCommand(MoveObject.name, COMMAND_MOVE_NORTH, MoveObject.immediate);
					QueuePersonCommand(MoveObject.name, COMMAND_MOVE_EAST, true);
				}
			break;
			case COMMAND_MOVE_NORTHWEST:
				QueuePersonCommand(MoveObject.name, COMMAND_FACE_NORTHWEST, false);
				if (this.isObstructed(MoveObject.name, MoveObject.command)) return;
				QueuePersonScript(MoveObject.name, GetPersonValue(MoveObject.name, "TileLeaveCode"), true);
				for (var i = 0; i < pixels; ++i)
				{
					QueuePersonCommand(MoveObject.name, COMMAND_MOVE_NORTH, MoveObject.immediate);
					QueuePersonCommand(MoveObject.name, COMMAND_MOVE_WEST, true);
				}
			break;
			case COMMAND_MOVE_SOUTHEAST:
				QueuePersonCommand(MoveObject.name, COMMAND_FACE_SOUTHEAST, false);
				if (this.isObstructed(MoveObject.name, MoveObject.command)) return;
				QueuePersonScript(MoveObject.name, GetPersonValue(MoveObject.name, "TileLeaveCode"), true);
				for (var i = 0; i < pixels; ++i)
				{
					QueuePersonCommand(MoveObject.name, COMMAND_MOVE_SOUTH, MoveObject.immediate);
					QueuePersonCommand(MoveObject.name, COMMAND_MOVE_EAST, true);
				}
			break;
			case COMMAND_MOVE_SOUTHWEST:
				QueuePersonCommand(MoveObject.name, COMMAND_FACE_SOUTHWEST, false);
				if (this.isObstructed(MoveObject.name, MoveObject.command)) return;
				QueuePersonScript(MoveObject.name, GetPersonValue(MoveObject.name, "TileLeaveCode"), true);
				for (var i = 0; i < pixels; ++i)
				{
					QueuePersonCommand(MoveObject.name, COMMAND_MOVE_SOUTH, MoveObject.immediate);
					QueuePersonCommand(MoveObject.name, COMMAND_MOVE_WEST, true);
				}
			break;
			case COMMAND_MOVE_NORTH:
			case COMMAND_MOVE_SOUTH:
			case COMMAND_MOVE_EAST:
			case COMMAND_MOVE_WEST:
				if (MoveObject.command == COMMAND_MOVE_NORTH) QueuePersonCommand(MoveObject.name, COMMAND_FACE_NORTH, false);
				if (MoveObject.command == COMMAND_MOVE_SOUTH) QueuePersonCommand(MoveObject.name, COMMAND_FACE_SOUTH, false);
				if (MoveObject.command == COMMAND_MOVE_EAST) QueuePersonCommand(MoveObject.name, COMMAND_FACE_EAST, false);
				if (MoveObject.command == COMMAND_MOVE_WEST) QueuePersonCommand(MoveObject.name, COMMAND_FACE_WEST, false);
				if (this.isObstructed(MoveObject.name, MoveObject.command)) { this.onHit(); return; }
				QueuePersonScript(MoveObject.name, GetPersonValue(MoveObject.name, "TileLeaveCode"), true);
				this.move(MoveObject.name, MoveObject.command, MoveObject.immediate);
			break;
			default:
				this.move(MoveObject.name, MoveObject.command, MoveObject.immediate);
			break;
		}
		QueuePersonScript(MoveObject.name, GetPersonValue(MoveObject.name, "TileEnterCode"), true);
	}
}

MoveSys.prototype.attachInput = function(name) { this.input = name; }

MoveSys.detachInput = function() { this.input = null; };

MoveSys.prototype.getInputPerson = function()
{
	if (this.input != null) return this.input;
	else Abort("MoveSys error @ MoveSys.getInputPerson(): no input attached.");
}

MoveSys.prototype.queuePersonCommand = function(name, command, immediate)
{
	this.moveQueue.push({name: name, command: command, immediate: immediate});
}

MoveSys.prototype.setTileSize = function(num) { this.tileSize = num; }

MoveSys.prototype.setDiagonal = function(value) { this.isDiag = value; }

MoveSys.prototype.move = function(name, command, immediate, pixels)
{
	if (!pixels) pixels = this.tileSize / GetPersonSpeedX(name);
	for (var i = 0; i < pixels; ++i)
		QueuePersonCommand(name, command, immediate);
}

MoveSys.prototype.update = function()
{
	this.processQueue();
	
	if (this.input == null) return;
	if (IsCommandQueueEmpty(this.input))
	{
		if (this.isAnyMovementKeyPressed() && !this.faced)
		{
			this.faced = true;
			var d = GetPersonDirection(this.input);
			SetPersonFrame(this.input, 0);
			if (this.isKeyPressed(KEY_UP) && d != "north")
			{
				QueuePersonCommand(this.input, COMMAND_FACE_NORTH, true);
				this.move(this.input, COMMAND_WAIT, false, 8);
			}
			if (this.isKeyPressed(KEY_DOWN) && d != "south")
			{
				QueuePersonCommand(this.input, COMMAND_FACE_SOUTH, true);
				this.move(this.input, COMMAND_WAIT, false, 8);
			}
			if (this.isKeyPressed(KEY_LEFT) && d != "west")
			{
				QueuePersonCommand(this.input, COMMAND_FACE_WEST, true);
				this.move(this.input, COMMAND_WAIT, false, 8);
			}
			if (this.isKeyPressed(KEY_RIGHT) && d != "east")
			{
				QueuePersonCommand(this.input, COMMAND_FACE_EAST, true);
				this.move(this.input, COMMAND_WAIT, false, 8);
			}
		}
		else if (this.isAnyMovementKeyPressed() && this.faced)
		{
			this.processMovement();
			this.afterMove = true;
		}
		else this.faced = false;
	}
	else if(this.afterMove)
	{
		if ( IsTriggerAt(GetPersonX(this.input), GetPersonY(this.input), 0)) ExecuteTrigger(GetPersonX(this.input), GetPersonY(this.input),layer);
		this.afterMove = false;
	}
}

MoveSys.prototype.processMovement = function()
{
	if (IsKeyPressed(KEY_UP) && IsKeyPressed(KEY_RIGHT) && this.isDiag)
		this.queuePersonCommand(this.input, COMMAND_MOVE_NORTHEAST, false);
	else if (IsKeyPressed(KEY_UP) && IsKeyPressed(KEY_LEFT) && this.isDiag)
		this.queuePersonCommand(this.input, COMMAND_MOVE_NORTHWEST, false);
	else if (IsKeyPressed(KEY_DOWN) && IsKeyPressed(KEY_RIGHT) && this.isDiag)
		this.queuePersonCommand(this.input, COMMAND_MOVE_SOUTHEAST, false);
	else if (IsKeyPressed(KEY_DOWN) && IsKeyPressed(KEY_LEFT) && this.isDiag)
		this.queuePersonCommand(this.input, COMMAND_MOVE_SOUTHWEST, false);
	else if (this.isKeyPressed(KEY_UP))
		this.queuePersonCommand(this.input, COMMAND_MOVE_NORTH, false);
	else if (this.isKeyPressed(KEY_DOWN))
		this.queuePersonCommand(this.input, COMMAND_MOVE_SOUTH, false);
	else if (this.isKeyPressed(KEY_LEFT))
		this.queuePersonCommand(this.input, COMMAND_MOVE_WEST, false);
	else if (this.isKeyPressed(KEY_RIGHT))
		this.queuePersonCommand(this.input, COMMAND_MOVE_EAST, false);
}

MoveSys.prototype.isKeyPressed = function(key)
{
	return (IsKeyPressed(key) && !this.keyPaused[key]);
}

MoveSys.prototype.isObstructed = function(name, direction)
{
	var x = GetPersonX(name);
	var y = GetPersonY(name);
	switch(direction)
	{
		case COMMAND_MOVE_NORTH:
			return IsPersonObstructed(name, x, y-this.tileSize);
		break;
		case COMMAND_MOVE_SOUTH:
			return IsPersonObstructed(name, x, y+this.tileSize);
		break;
		case COMMAND_MOVE_EAST:
			return IsPersonObstructed(name, x+this.tileSize, y);
		break;
		case COMMAND_MOVE_WEST:
			return IsPersonObstructed(name, x-this.tileSize, y);
		break;
	}
}

MoveSys.prototype.isAnyMovementKeyPressed = function()
{
	if (IsKeyPressed(KEY_UP)) return true;
	else if (IsKeyPressed(KEY_DOWN)) return true;
	else if (IsKeyPressed(KEY_LEFT)) return true;
	else if (IsKeyPressed(KEY_RIGHT)) return true;
	else return false;
}

MoveSys.prototype.toggleKey = function(Key, value)
{
	this.keyPaused[Key] = !value;
}

// Magic happenz here //

var _SetPersonScript = SetPersonScript;
SetPersonScript = function(name, which, script)
{
	switch (which)
	{
		case SCRIPT_ON_TILE_ENTER:
			SetPersonValue(name, "TileEnterCode", script);
		break;
		case SCRIPT_ON_TILE_LEAVE:
			SetPersonValue(name, "TileLeaveCode", script);
		break;
		default:
			_SetPersonScript(name, which, script);
		break;
	}
}

function SetPlayerDirection()
{
	var direction = GetPersonDirection("Player");
	if (!IsKeyPressed(KEY_UP) && !IsKeyPressed(KEY_DOWN) && !IsKeyPressed(KEY_LEFT)
			&& !IsKeyPressed(KEY_RIGHT) && IsCommandQueueEmpty("Player") && direction.indexOf("look", 0) != 0)
	{
		if (IsStandardDirection(direction)) SetPersonDirection("Player", "look" + direction);
	}
}

function IsStandardDirection(direction)
{
	return (direction == "south" || direction == "north" || direction == "west" || direction == "east"); 
}