
const IDENTITY    = 0;
const TRANSLATION = 1;
const SCALING     = 2;
const ROTATION_X  = 3;
const ROTATION_Y  = 4;
const ROTATION_Z  = 5;

const FOCAL_DISTANCE = 500;

const ORIGIN_X = GetScreenWidth()  / 2;
const ORIGIN_Y = GetScreenHeight() / 2;

var SIN = new Array();
var COS = new Array();

for (var i = -360; i < 360; ++i)
{
  SIN.push(Math.sin(i * Math.PI / 180));
  COS.push(Math.cos(i * Math.PI / 180));
}


function _2D(x, y)
{
  this.x = x || 0;
  this.y = y || 0;
}

function _3D(x, y, z)
{
  this.x = x || 0;
  this.y = y || 0;
  this.z = z || 0;
}

function Vertex(lx, ly, lz)
{
  this.local   = new _3D(lx, ly, lz);
  this.world   = new _3D();
  this.aligned = new _3D();
  this.screen  = new _2D();
}

function Matrix(id, x, y, z)
{
  this.data = new Array(4);
  
  for (var i = 0; i < 4; ++i)
    this.data[i] = new Array(4);
  
  var m = this.data;
  
  switch (id)
  {
    case IDENTITY:
      m[0][0] = 1; m[0][1] = 0; m[0][2] = 0; m[0][3] = 0;
      m[1][0] = 0; m[1][1] = 1; m[1][2] = 0; m[1][3] = 0;
      m[2][0] = 0; m[2][1] = 0; m[2][2] = 1; m[2][3] = 0;
      m[3][0] = 0; m[3][1] = 0; m[3][2] = 0; m[3][3] = 1;
      break;
    
    case TRANSLATION:
      m[0][0] = 1; m[0][1] = 0; m[0][2] = 0; m[0][3] = x;
      m[1][0] = 0; m[1][1] = 1; m[1][2] = 0; m[1][3] = y;
      m[2][0] = 0; m[2][1] = 0; m[2][2] = 1; m[2][3] = z;
      m[3][0] = 0; m[3][1] = 0; m[3][2] = 0; m[3][3] = 1;
      break;
    
    case SCALING:
      m[0][0] = x; m[0][1] = 0; m[0][2] = 0; m[0][3] = 0;
      m[1][0] = 0; m[1][1] = y; m[1][2] = 0; m[1][3] = 0;
      m[2][0] = 0; m[2][1] = 0; m[2][2] = z; m[2][3] = 0;
      m[3][0] = 0; m[3][1] = 0; m[3][2] = 0; m[3][3] = 1;
      break;
    
    case ROTATION_X:
      m[0][0] = 1; m[0][1] = 0;          m[0][2] = 0;           m[0][3] = 0;
      m[1][0] = 0; m[1][1] = COS[x+360]; m[1][2] = -SIN[x+360]; m[1][3] = 0;
      m[2][0] = 0; m[2][1] = SIN[x+360]; m[2][2] =  COS[x+360]; m[2][3] = 0;
      m[3][0] = 0; m[3][1] = 0;          m[3][2] = 0;           m[3][3] = 1;
      break;
    
    case ROTATION_Y:
      m[0][0] =  COS[x+360]; m[0][1] = 0; m[0][2] = SIN[x+360]; m[0][3] = 0;
      m[1][0] = 0;           m[1][1] = 1; m[1][2] = 0;          m[1][3] = 0;
      m[2][0] = -SIN[x+360]; m[2][1] = 0; m[2][2] = COS[x+360]; m[2][3] = 0;
      m[3][0] = 0;           m[3][1] = 0; m[3][2] = 0;          m[3][3] = 1;
      break;
    
    case ROTATION_Z:
      m[0][0] = COS[x+360]; m[0][1] = -SIN[x+360]; m[0][2] = 0; m[0][3] = 0;
      m[1][0] = SIN[x+360]; m[1][1] =  COS[x+360]; m[1][2] = 0; m[1][3] = 0;
      m[2][0] = 0;          m[2][1] = 0;           m[2][2] = 1; m[2][3] = 0;
      m[3][0] = 0;          m[3][1] = 0;           m[3][2] = 0; m[3][3] = 1;
      break;
      
    default:
      Abort("Error: invalid matrix type: " + id);
  }
}

function VectorMultMatrix(r, v, matrix)
{
  var m = matrix.data;
  
  r.x = v.x * m[0][0] + v.y * m[0][1] + v.z * m[0][2] + m[0][3];
  r.y = v.x * m[1][0] + v.y * m[1][1] + v.z * m[1][2] + m[1][3];
  r.z = v.x * m[2][0] + v.y * m[2][1] + v.z * m[2][2] + m[2][3];
}

function MatrixMultMatrix(matrix1, matrix2)
{
  var result = new Matrix(IDENTITY);
  var r  = result.data;
  var m1 = matrix1.data;
  var m2 = matrix2.data;
  
  for (var i = 0; i < 4; ++i)
    for (var k = 0; k < 4; ++k)
      r[k][i] = m1[0][i] * m2[k][0] + 
                m1[1][i] * m2[k][1] + 
                m1[2][i] * m2[k][2] + 
                m1[3][i] * m2[k][3];
  
  return result;
}

function Camera(position, angle)
{
  this.position = position;
  this.angle    = angle;
  this.matrix;
}

Camera.prototype.update = function()
{
  this.matrix = new Matrix(TRANSLATION, -this.position.x, -this.position.y, -this.position.z);
  this.matrix = MatrixMultMatrix(this.matrix, new Matrix(ROTATION_X, -this.angle.x));
  this.matrix = MatrixMultMatrix(this.matrix, new Matrix(ROTATION_Y, -this.angle.y));
  this.matrix = MatrixMultMatrix(this.matrix, new Matrix(ROTATION_Z, -this.angle.z));

}

function _3DEngine()
{
  this.objects = new Array();
  
  this.camera = new Camera(new _3D(-100, 0, -600), new _3D(0, 0, 0));
}

_3DEngine.prototype.add = function()
{
  for (var i = 0; i < arguments.length; ++i)
    this.objects.push(arguments[i]);
}

_3DEngine.prototype.update = function()
{
  this.camera.update();
  
  for (var k = 0; k < this.objects.length; ++k)
  {
    var o = this.objects[k];
    
    o.update();
    
    var m = new Matrix(SCALING, o.scale.x, o.scale.y, o.scale.z);
    m = MatrixMultMatrix(m, new Matrix(ROTATION_X, o.angle.x));
    m = MatrixMultMatrix(m, new Matrix(ROTATION_Y, o.angle.y));
    m = MatrixMultMatrix(m, new Matrix(ROTATION_Z, o.angle.z));
    m = MatrixMultMatrix(m, new Matrix(TRANSLATION, o.position.x, o.position.y, o.position.z));

    var v = o.vertices;
   
    for (var i = 0; i < v.length; ++i)
    {
      VectorMultMatrix(v[i].world,   v[i].local, m);
      VectorMultMatrix(v[i].aligned, v[i].world, this.camera.matrix);
      
      if (v[i].aligned.z == 0)
        v[i].aligned.z = 1;
     
      v[i].screen.x = FOCAL_DISTANCE * v[i].aligned.x / v[i].aligned.z + ORIGIN_X;
      v[i].screen.y = FOCAL_DISTANCE * v[i].aligned.y / v[i].aligned.z + ORIGIN_Y;
    }
  }
}

_3DEngine.prototype.draw = function()
{
  for (var k = 0; k < this.objects.length; ++k)
    this.objects[k].draw();
}
















