#define DIRECTDRAW_VERSION 0x0300
#include <windows.h>
#include <ddraw.h>
#include <stdio.h>
#include "../common/video.h"
#include "colorspace.h"
#include "resource.h"
#include "x++.hpp"
#include "win32x.h"


typedef struct _IMAGE
{
  int width;
  int height;

  union
  {
    BGRA* bgra;
    BGR*  bgr;
  };
  byte* alpha;

  bool (*blit_routine)(_IMAGE* image, int x, int y);

  RGBA* locked_pixels;
}* IMAGE;


enum BIT_DEPTH
{
  BD_AUTODETECT,
  BD_32,
  BD_24,
};

struct CONFIGURATION
{
  BIT_DEPTH bit_depth;

  bool fullscreen;
  bool vsync;
};


static void LoadConfiguration();
static void SaveConfiguration();
static BOOL CALLBACK ConfigureDialogProc(HWND window, UINT message, WPARAM wparam, LPARAM lparam);

static bool InitFullscreen();
static bool SetDisplayMode();
static bool CreateSurfaces();
static bool InitWindowed();

static bool CloseFullscreen();
static bool CloseWindowed();

static void FillImagePixels(IMAGE image, RGBA* data);
static void OptimizeBlitRoutine(IMAGE image);

static bool NullBlit(IMAGE image, int x, int y);
static bool TileBlit(IMAGE image, int x, int y);
static bool SpriteBlit(IMAGE image, int x, int y);
static bool NormalBlit(IMAGE image, int x, int y);


static CONFIGURATION Configuration;
static int           BitsPerPixel;

static HWND  SphereWindow;
static void* ScreenBuffer;

static LONG OldWindowStyle;
static LONG OldWindowStyleEx;

// windowed output
static HDC     RenderDC;
static HBITMAP RenderBitmap;

// fullscreen output
static LPDIRECTDRAW dd;
static LPDIRECTDRAWSURFACE ddPrimary;
static LPDIRECTDRAWSURFACE ddSecondary;


////////////////////////////////////////////////////////////////////////////////

void EXPORT GetDriverInfo(DRIVERINFO* driverinfo)
{
  driverinfo->name        = "Standard 32-bit Color";
  driverinfo->author      = "Chad Austin";
  driverinfo->date        = __DATE__;
  driverinfo->version     = "1.00";
  driverinfo->description = "24/32-bit color output in both windowed and fullscreen modes";
}

////////////////////////////////////////////////////////////////////////////////

void EXPORT ConfigureDriver(HWND parent)
{
  LoadConfiguration();
  DialogBox(DriverInstance, MAKEINTRESOURCE(IDD_CONFIGURE), parent, ConfigureDialogProc);
  SaveConfiguration();
}

////////////////////////////////////////////////////////////////////////////////

void LoadConfiguration()
{
  char config_file_name[MAX_PATH];
  GetDriverConfigFile(config_file_name);

  // load the fields from the file
  int bit_depth = GetPrivateProfileInt("standard32", "BitDepth", 0, config_file_name);
  Configuration.bit_depth = (bit_depth == 32 ? BD_32 : (bit_depth == 24 ? BD_24 : BD_AUTODETECT));

  Configuration.fullscreen = GetPrivateProfileInt("standard32", "Fullscreen", 1, config_file_name) != 0;
  Configuration.vsync      = GetPrivateProfileInt("standard32", "VSync",      1, config_file_name) != 0;
}

////////////////////////////////////////////////////////////////////////////////

void SaveConfiguration()
{
  char config_file_name[MAX_PATH];
  GetDriverConfigFile(config_file_name);

  // save the fields to the file
  int bit_depth = (Configuration.bit_depth == BD_32 ? 32 : (Configuration.bit_depth == BD_24 ? 24 : 0));
  WritePrivateProfileInt("standard32", "BitDepth",   bit_depth,                config_file_name);
  WritePrivateProfileInt("standard32", "Fullscreen", Configuration.fullscreen, config_file_name);
  WritePrivateProfileInt("standard32", "VSync",      Configuration.vsync,      config_file_name);
}

////////////////////////////////////////////////////////////////////////////////

BOOL CALLBACK ConfigureDialogProc(HWND window, UINT message, WPARAM wparam, LPARAM lparam)
{
  switch (message)
  {
    case WM_INITDIALOG:
      // set the bit depth radio buttons
      if (Configuration.bit_depth == BD_AUTODETECT)
        SendDlgItemMessage(window, IDC_BITDEPTH_AUTODETECT, BM_SETCHECK, BST_CHECKED, 0);
      else if (Configuration.bit_depth == BD_32)
        SendDlgItemMessage(window, IDC_BITDEPTH_32, BM_SETCHECK, BST_CHECKED, 0);
      else if (Configuration.bit_depth == BD_24)
        SendDlgItemMessage(window, IDC_BITDEPTH_24, BM_SETCHECK, BST_CHECKED, 0);

      // set the check boxes
      CheckDlgButton(window, IDC_FULLSCREEN, Configuration.fullscreen ? BST_CHECKED : BST_UNCHECKED);
      CheckDlgButton(window, IDC_VSYNC,      Configuration.vsync      ? BST_CHECKED : BST_UNCHECKED);

      // update the check states
      SendMessage(window, WM_COMMAND, MAKEWPARAM(IDC_FULLSCREEN, BN_PUSHED), 0);

      return TRUE;

    ////////////////////////////////////////////////////////////////////////////

    case WM_COMMAND:
      switch (LOWORD(wparam))
      {
        case IDOK:
          if (IsDlgButtonChecked(window, IDC_BITDEPTH_32))
            Configuration.bit_depth = BD_32;
          else if (IsDlgButtonChecked(window, IDC_BITDEPTH_24))
            Configuration.bit_depth = BD_24;
          else
            Configuration.bit_depth = BD_AUTODETECT;

          Configuration.fullscreen = (IsDlgButtonChecked(window, IDC_FULLSCREEN) != FALSE);
          Configuration.vsync      = (IsDlgButtonChecked(window, IDC_VSYNC)      != FALSE);

          EndDialog(window, 1);
          return TRUE;

        case IDCANCEL:
          EndDialog(window, 0);
          return TRUE;

        case IDC_FULLSCREEN:
          EnableWindow(GetDlgItem(window, IDC_VSYNC), IsDlgButtonChecked(window, IDC_FULLSCREEN));
          EnableWindow(GetDlgItem(window, IDC_BITDEPTH_AUTODETECT), IsDlgButtonChecked(window, IDC_FULLSCREEN));
          if (IsDlgButtonChecked(window, IDC_BITDEPTH_AUTODETECT) && !IsDlgButtonChecked(window, IDC_FULLSCREEN))
          {
            CheckDlgButton(window, IDC_BITDEPTH_AUTODETECT, BST_UNCHECKED);
            CheckDlgButton(window, IDC_BITDEPTH_32,         BST_CHECKED);
          }
          return TRUE;
      }
      return FALSE;

    ////////////////////////////////////////////////////////////////////////////

    default:
      return FALSE;
  }
}

////////////////////////////////////////////////////////////////////////////////

bool EXPORT InitVideoDriver(HWND window, int screen_width, int screen_height)
{
  SphereWindow = window;
  ScreenWidth  = screen_width;
  ScreenHeight = screen_height;

  // set default clipping rectangle
  SetClippingRectangle(0, 0, screen_width, screen_height);

  LoadConfiguration();
  if (Configuration.fullscreen)
    return InitFullscreen();
  else
    return InitWindowed();
}

////////////////////////////////////////////////////////////////////////////////

bool InitFullscreen()
{
  HRESULT ddrval;
  bool    retval;

  // store old window styles
  OldWindowStyle = GetWindowLong(SphereWindow, GWL_STYLE);
  OldWindowStyleEx = GetWindowLong(SphereWindow, GWL_EXSTYLE);

  SetWindowLong(SphereWindow, GWL_STYLE, WS_POPUP);
  SetWindowLong(SphereWindow, GWL_EXSTYLE, 0);

  // create DirectDraw object
  ddrval = DirectDrawCreate(NULL, &dd, NULL);
  if (ddrval != DD_OK)
  {
    MessageBox(SphereWindow, "DirectDrawCreate() failed", "standard32", MB_OK);
    return false;
  }

  // set application behavior
  ddrval = dd->SetCooperativeLevel(SphereWindow, DDSCL_EXCLUSIVE | DDSCL_FULLSCREEN);
  if (ddrval != DD_OK)
  {
    dd->Release();
    MessageBox(SphereWindow, "SetCooperativeLevel() failed", "standard32", MB_OK);
    return false;
  }

  // set display mode
  retval = SetDisplayMode();
  if (retval == false)
  {
    dd->Release();
    MessageBox(SphereWindow, "SetDisplayMode() failed", "standard32", MB_OK);
    return false;
  }

  // create surfaces
  retval = CreateSurfaces();
  if (retval == false)
  {
    dd->Release();
    MessageBox(SphereWindow, "CreateSurfaces() failed", "standard32", MB_OK);
    return false;
  }

  ShowCursor(FALSE);

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool SetDisplayMode()
{
  HRESULT ddrval;

  switch (Configuration.bit_depth)
  {
    case BD_AUTODETECT:
      ddrval = dd->SetDisplayMode(ScreenWidth, ScreenHeight, 32);
      BitsPerPixel = 32;
      if (ddrval != DD_OK)
      {
        ddrval = dd->SetDisplayMode(ScreenWidth, ScreenHeight, 24);
        BitsPerPixel = 24;
      }
      return ddrval == DD_OK;

    case BD_32:
      ddrval = dd->SetDisplayMode(ScreenWidth, ScreenHeight, 32);
      BitsPerPixel = 32;
      return ddrval == DD_OK;

    case BD_24:
      ddrval = dd->SetDisplayMode(ScreenWidth, ScreenHeight, 24);
      BitsPerPixel = 24;
      return ddrval == DD_OK;

    default:
      return false;
  }
}

////////////////////////////////////////////////////////////////////////////////

bool CreateSurfaces()
{
  // define the surface
  DDSURFACEDESC ddsd;
  ddsd.dwSize = sizeof(ddsd);
  
  if (Configuration.vsync)
  {
    ddsd.dwFlags           = DDSD_CAPS | DDSD_BACKBUFFERCOUNT;
    ddsd.ddsCaps.dwCaps    = DDSCAPS_PRIMARYSURFACE | DDSCAPS_FLIP | DDSCAPS_COMPLEX;
    ddsd.dwBackBufferCount = 1;
  }
  else
  {
    ddsd.dwFlags        = DDSD_CAPS;
    ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE;
  }

  // create the primary surface
  HRESULT ddrval = dd->CreateSurface(&ddsd, &ddPrimary, NULL);
  if (ddrval != DD_OK)
    return false;

  if (Configuration.vsync)
  {
    ddsd.ddsCaps.dwCaps = DDSCAPS_BACKBUFFER;
    ddrval = ddPrimary->GetAttachedSurface(&ddsd.ddsCaps, &ddSecondary);
    if (ddrval != DD_OK)
    {
      ddPrimary->Release();
      return false;
    }
  }

  // allocate a blitting buffer
  ScreenBuffer = new byte[ScreenWidth * ScreenHeight * (BitsPerPixel / 8)];

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool InitWindowed()
{
  // calculate bits per pixel
  BitsPerPixel = (Configuration.bit_depth == BD_32 ? 32 : 24);

  // create the render DC
  RenderDC = CreateCompatibleDC(NULL);
  if (RenderDC == NULL)
    return false;

  // define/create the render DIB section
  BITMAPINFO bmi;
  memset(&bmi, 0, sizeof(bmi));
  BITMAPINFOHEADER& bmih = bmi.bmiHeader;
  bmih.biSize        = sizeof(bmih);
  bmih.biWidth       = ScreenWidth;
  bmih.biHeight      = -ScreenHeight;
  bmih.biPlanes      = 1;
  bmih.biBitCount    = BitsPerPixel;
  bmih.biCompression = BI_RGB;
  RenderBitmap = CreateDIBSection(RenderDC, &bmi, DIB_RGB_COLORS, (void**)&ScreenBuffer, NULL, 0);
  if (RenderBitmap == NULL)
  {
    DeleteDC(RenderDC);
    return false;
  }

  SelectObject(RenderDC, RenderBitmap);

  // center the window
  RECT WindowRect = { 0, 0, ScreenWidth, ScreenHeight };
  AdjustWindowRectEx(
    &WindowRect,
    GetWindowLong(SphereWindow, GWL_STYLE),
    (GetMenu(SphereWindow) ? TRUE : FALSE),
    GetWindowLong(SphereWindow, GWL_EXSTYLE));

  int window_width  = WindowRect.right - WindowRect.left;
  int window_height = WindowRect.bottom - WindowRect.top;

  MoveWindow(
    SphereWindow,
    (GetSystemMetrics(SM_CXSCREEN) - window_width) / 2,
    (GetSystemMetrics(SM_CYSCREEN) - window_height) / 2,
    window_width,
    window_height,
    TRUE);

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool EXPORT CloseVideoDriver()
{
  if (Configuration.fullscreen)
    return CloseFullscreen();
  else
    return CloseWindowed();
}

////////////////////////////////////////////////////////////////////////////////

bool CloseFullscreen()
{
  SetWindowLong(SphereWindow, GWL_STYLE, OldWindowStyle);
  SetWindowLong(SphereWindow, GWL_EXSTYLE, OldWindowStyleEx);

  ShowCursor(TRUE);
  delete[] ScreenBuffer;
  dd->Release();
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool CloseWindowed()
{
  DeleteDC(RenderDC);
  DeleteObject(RenderBitmap);
  return true;
}

////////////////////////////////////////////////////////////////////////////////

void FillImagePixels(IMAGE image, RGBA* pixels)
{
  // fill the image pixels
  if (BitsPerPixel == 32)
  {
    image->bgra = new BGRA[image->width * image->height];
    for (int i = 0; i < image->width * image->height; i++)
    {
      image->bgra[i].red   = pixels[i].red;
      image->bgra[i].green = pixels[i].green;
      image->bgra[i].blue  = pixels[i].blue;
    }
  }
  else
  {
    image->bgr  = new BGR[image->width * image->height];
    for (int i = 0; i < image->width * image->height; i++)
    {
      image->bgr[i].red   = pixels[i].red;
      image->bgr[i].green = pixels[i].green;
      image->bgr[i].blue  = pixels[i].blue;
    }
  }

  // fill the alpha array
  image->alpha = new byte[image->width * image->height];
  for (int i = 0; i < image->width * image->height; i++)
    image->alpha[i] = pixels[i].alpha;
}

////////////////////////////////////////////////////////////////////////////////

void OptimizeBlitRoutine(IMAGE image)
{
  // null blit
  bool is_empty = true;
  for (int i = 0; i < image->width * image->height; i++)
    if (image->alpha[i] != 0)
    {
      is_empty = false;
      break;
    }
  if (is_empty)
  {
    image->blit_routine = NullBlit;
    return;
  }

  // tile blit
  bool is_tile = true;
  for (int i = 0; i < image->width * image->height; i++)
    if (image->alpha[i] != 255)
    {
      is_tile = false;
      break;
    }
  if (is_tile)
  {
    image->blit_routine = TileBlit;
    return;
  }

  // sprite blit
  bool is_sprite = true;
  for (int i = 0; i < image->width * image->height; i++)
    if (image->alpha[i] != 0 &&
        image->alpha[i] != 255)
    {
      is_sprite = false;
      break;
    }
  if (is_sprite)
  {
    image->blit_routine = SpriteBlit;
    return;
  }

  // normal blit
  image->blit_routine = NormalBlit;
}

////////////////////////////////////////////////////////////////////////////////

bool EXPORT FlipScreen()
{
  if (Configuration.fullscreen)
  {
    LPDIRECTDRAWSURFACE surface;
    if (Configuration.vsync)
      surface = ddSecondary;
    else
      surface = ddPrimary;

    // lock the surface
    DDSURFACEDESC ddsd;
    ddsd.dwSize = sizeof(ddsd);
    HRESULT ddrval = surface->Lock(NULL, &ddsd, DDLOCK_WAIT, NULL);

    // if the surface was lost, restore it
    if (ddrval == DDERR_SURFACELOST)
    {
      surface->Restore();
      if (surface == ddSecondary)
        ddPrimary->Restore();

      // attempt to lock again
      ddrval = surface->Lock(NULL, &ddsd, DDLOCK_WAIT, NULL);
      if (ddrval != DD_OK)
      {
        Sleep(100);
        return false;
      }
    }

    // render backbuffer to the screen
    if (BitsPerPixel == 32)
    {
      BGRA* dst = (BGRA*)ddsd.lpSurface;
      BGRA* src = (BGRA*)ScreenBuffer;
      for (int i = 0; i < ScreenHeight; i++)
      {
        memcpy(dst, src, ScreenWidth * 4);
        dst += ddsd.lPitch / 4;
        src += ScreenWidth;
      }
    }
    else
    {
      BGR* dst = (BGR*)ddsd.lpSurface;
      BGR* src = (BGR*)ScreenBuffer;
      for (int i = 0; i < ScreenHeight; i++)
      {
        memcpy(dst, src, ScreenWidth * 3);
        dst += ddsd.lPitch / 3;
        src += ScreenWidth;
      }
    }

    // unlock the surface and do the flip!
    surface->Unlock(NULL);
    if (Configuration.vsync)
      ddPrimary->Flip(NULL, DDFLIP_WAIT);
  }
  else
  {
    // make sure the lines are on dword boundaries
    if (BitsPerPixel == 24)
    {
      int pitch = (ScreenWidth * 3 + 3) / 4 * 4;
      byte* dst = (byte*)ScreenBuffer;
      BGR*  src = (BGR*)ScreenBuffer;
      for (int i = ScreenHeight - 1; i >= 0; i--)
      {
        memmove(dst + i * pitch,
                src + i * ScreenWidth,
                ScreenWidth * 3);
      }
    }

    // blit the render buffer to the window
    HDC dc = GetDC(SphereWindow);
    BitBlt(dc, 0, 0, ScreenWidth, ScreenHeight, RenderDC, 0, 0, SRCCOPY);
    ReleaseDC(SphereWindow, dc);
  }

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool EXPORT ApplyColorMask(RGBA mask)
{
  // no mask
  if (mask.alpha == 0)
    return true;

  if (BitsPerPixel == 32)
  {
    BGRA* Screen = (BGRA*)ScreenBuffer;

    // full mask
    if (mask.alpha == 255)
    {
      BGRA Pixel = { mask.blue, mask.green, mask.red };  // alpha on the destination surface does not matter
      for (int iy = ClippingRectangle.top; iy < ClippingRectangle.bottom; iy++)
        for (int ix = ClippingRectangle.left; ix < ClippingRectangle.right; ix++)
          Screen[iy * ScreenWidth + ix] = Pixel;
      return true;
    }

    // partial mask
    for (int iy = ClippingRectangle.top; iy < ClippingRectangle.bottom; iy++)
      for (int ix = ClippingRectangle.left; ix < ClippingRectangle.right; ix++)
      {
        int i = iy * ScreenWidth + ix;
        Screen[i].red   = (mask.red   * mask.alpha + Screen[i].red    * (255 - mask.alpha)) / 256;
        Screen[i].green = (mask.green * mask.alpha + Screen[i].green  * (255 - mask.alpha)) / 256;
        Screen[i].blue  = (mask.blue  * mask.alpha + Screen[i].blue   * (255 - mask.alpha)) / 256;
      }
    return true;
  }
  else
  {
    BGR* Screen = (BGR*)ScreenBuffer;

    // full mask
    if (mask.alpha == 255)
    {
      BGR Pixel = { mask.blue, mask.green, mask.red };
      for (int iy = ClippingRectangle.top; iy < ClippingRectangle.bottom; iy++)
        for (int ix = ClippingRectangle.left; ix < ClippingRectangle.right; ix++)
          Screen[iy * ScreenWidth + ix] = Pixel;
      return true;
    }

    // partial mask
    for (int iy = ClippingRectangle.top; iy < ClippingRectangle.bottom; iy++)
      for (int ix = ClippingRectangle.left; ix < ClippingRectangle.right; ix++)
      {
        int i = iy * ScreenWidth + ix;
        Screen[i].red   = (mask.red   * mask.alpha + Screen[i].red    * (255 - mask.alpha)) / 256;
        Screen[i].green = (mask.green * mask.alpha + Screen[i].green  * (255 - mask.alpha)) / 256;
        Screen[i].blue  = (mask.blue  * mask.alpha + Screen[i].blue   * (255 - mask.alpha)) / 256;
      }
    return true;
  }
}

////////////////////////////////////////////////////////////////////////////////

IMAGE EXPORT CreateImage(int width, int height, RGBA* pixels)
{
  // allocate the image
  IMAGE image = new _IMAGE;
  image->width  = width;
  image->height = height;

  FillImagePixels(image, pixels);
  OptimizeBlitRoutine(image);
  return image;
}

////////////////////////////////////////////////////////////////////////////////

IMAGE EXPORT GrabImage(int x, int y, int width, int height)
{
  if (x < 0 ||
      y < 0 ||
      x + width > ScreenWidth ||
      y + height > ScreenHeight)
    return NULL;

  IMAGE image = new _IMAGE;
  image->width        = width;
  image->height       = height;
  image->blit_routine = TileBlit;

  if (BitsPerPixel == 32)
  {
    BGRA* Screen = (BGRA*)ScreenBuffer;
    image->bgra = new BGRA[width * height];
    for (int iy = 0; iy < height; iy++)
      memcpy(image->bgra + iy * width,
             Screen + (y + iy) * ScreenWidth + x,
             width * 4);
  }
  else
  {
    BGR* Screen = (BGR*)ScreenBuffer;
    image->bgr = new BGR[width * height];
    for (int iy = 0; iy < height; iy++)
      memcpy(image->bgr + iy * width,
             Screen + (y + iy) * ScreenWidth + x,
             width * 3);
  }

  image->alpha = new byte[width * height];
  memset(image->alpha, 255, width * height);

  return image;
}

////////////////////////////////////////////////////////////////////////////////

bool EXPORT DestroyImage(IMAGE image)
{
  if (BitsPerPixel == 32)
    delete[] image->bgra;
  else
    delete[] image->bgr;
  delete[] image->alpha;
  delete image;
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool EXPORT BlitImage(IMAGE image, int x, int y)
{
  // don't draw it if it's off the screen
  if (x + (int)image->width < ClippingRectangle.left ||
      y + (int)image->height < ClippingRectangle.top ||
      x >= ClippingRectangle.right ||
      y >= ClippingRectangle.bottom)
    return true;

  return image->blit_routine(image, x, y);
}

////////////////////////////////////////////////////////////////////////////////

bool NullBlit(IMAGE image, int x, int y)
{
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool TileBlit(IMAGE image, int x, int y)
{
  calculate_clipping_metrics(image->width, image->height);

  if (BitsPerPixel == 32)
  {
    BGRA* Screen = (BGRA*)ScreenBuffer;
    for (int iy = image_offset_y; iy < image_offset_y + image_blit_height; iy++)
      memcpy(Screen + ((y + iy) * ScreenWidth + x + image_offset_x),
             image->bgra + iy * image->width + image_offset_x,
             image_blit_width * sizeof(BGRA));
  }
  else
  {
    BGR* Screen = (BGR*)ScreenBuffer;
    for (int iy = image_offset_y; iy < image_offset_y + image_blit_height; iy++)
      memcpy(Screen + ((y + iy) * ScreenWidth + x + image_offset_x),
             image->bgr + iy * image->width + image_offset_x,
             image_blit_width * sizeof(BGR));
  }

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool SpriteBlit(IMAGE image, int x, int y)
{
  calculate_clipping_metrics(image->width, image->height);

  if (BitsPerPixel == 32)
  {
    BGRA* Screen = (BGRA*)ScreenBuffer;
    for (int iy = image_offset_y; iy < image_offset_y + image_blit_height; iy++)
      for (int ix = image_offset_x; ix < image_offset_x + image_blit_width; ix++)
        if (image->alpha[iy * image->width + ix])
          Screen[(y + iy) * ScreenWidth + (x + ix)] =
            image->bgra[iy * image->width + ix];
  }
  else
  {
    BGR* Screen = (BGR*)ScreenBuffer;
    for (int iy = image_offset_y; iy < image_offset_y + image_blit_height; iy++)
      for (int ix = image_offset_x; ix < image_offset_x + image_blit_width; ix++)
        if (image->alpha[iy * image->width + ix])
          Screen[(y + iy) * ScreenWidth + (x + ix)] =
            image->bgr[iy * image->width + ix];
  }

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool NormalBlit(IMAGE image, int x, int y)
{
  calculate_clipping_metrics(image->width, image->height);

  if (BitsPerPixel == 32)
  {
    for (int iy = image_offset_y; iy < image_offset_y + image_blit_height; iy++)
      for (int ix = image_offset_x; ix < image_offset_x + image_blit_width; ix++)
      {
        BGRA* dest  = (BGRA*)ScreenBuffer + (y + iy) * ScreenWidth + x + ix;
        BGRA  src   = image->bgra[iy * image->width + ix];
        byte  alpha = image->alpha[iy * image->width + ix];

        if (alpha == 255)
          *dest = src;
        else if (alpha >= 0)
        {
          dest->red   = (dest->red   * (256 - alpha) + src.red   * alpha) / 256;
          dest->green = (dest->green * (256 - alpha) + src.green * alpha) / 256;
          dest->blue  = (dest->blue  * (256 - alpha) + src.blue  * alpha) / 256;
        }
      }
  }
  else
  {
    for (int iy = image_offset_y; iy < image_offset_y + image_blit_height; iy++)
      for (int ix = image_offset_x; ix < image_offset_x + image_blit_width; ix++)
      {
        BGR* dest  = (BGR*)ScreenBuffer + (y + iy) * ScreenWidth + x + ix;
        BGR  src   = image->bgr[iy * image->width + ix];
        byte alpha = image->alpha[iy * image->width + ix];

        if (alpha == 255)
          *dest = src;
        else if (alpha >= 0)
        {
          dest->red   = (dest->red   * (256 - alpha) + src.red   * alpha) / 256;
          dest->green = (dest->green * (256 - alpha) + src.green * alpha) / 256;
          dest->blue  = (dest->blue  * (256 - alpha) + src.blue  * alpha) / 256;
        }
      }
  }

  return true;
}

////////////////////////////////////////////////////////////////////////////////

int EXPORT GetImageWidth(IMAGE image)
{
  return image->width;
}

////////////////////////////////////////////////////////////////////////////////

int EXPORT GetImageHeight(IMAGE image)
{
  return image->height;
}

////////////////////////////////////////////////////////////////////////////////

RGBA* EXPORT LockImage(IMAGE image)
{
  image->locked_pixels = new RGBA[image->width * image->height];

  // rgb
  if (BitsPerPixel == 32)
  {
    for (int i = 0; i < image->width * image->height; i++)
    {
      image->locked_pixels[i].red   = image->bgra[i].red;
      image->locked_pixels[i].green = image->bgra[i].green;
      image->locked_pixels[i].blue  = image->bgra[i].blue;
    }
  }
  else
  {
    for (int i = 0; i < image->width * image->height; i++)
    {
      image->locked_pixels[i].red   = image->bgr[i].red;
      image->locked_pixels[i].green = image->bgr[i].green;
      image->locked_pixels[i].blue  = image->bgr[i].blue;
    }
  }

  // alpha
  for (int i = 0; i < image->width * image->height; i++)
    image->locked_pixels[i].alpha = image->alpha[i];

  return image->locked_pixels;
}

////////////////////////////////////////////////////////////////////////////////

void EXPORT UnlockImage(IMAGE image)
{
  if (BitsPerPixel == 32)
    delete[] image->bgra;
  else
    delete[] image->bgr;
  delete[] image->alpha;
  
  FillImagePixels(image, image->locked_pixels);
  OptimizeBlitRoutine(image);
  delete[] image->locked_pixels;
}

////////////////////////////////////////////////////////////////////////////////

void EXPORT DirectBlit(int x, int y, int w, int h, RGBA* pixels, int method)
{
  if (method == 0)
    return;

  calculate_clipping_metrics(w, h);

  if (method == 1)
  {

    if (BitsPerPixel == 32)
    {
      BGRA* Screen = (BGRA*)ScreenBuffer;
      for (int iy = image_offset_y; iy < image_offset_y + image_blit_height; iy++)
        for (int ix = image_offset_x; ix < image_offset_x + image_blit_width; ix++)
        {
          Screen[(y + iy) * ScreenWidth + x + ix].red   = pixels[iy * w + ix].red;
          Screen[(y + iy) * ScreenWidth + x + ix].green = pixels[iy * w + ix].green;
          Screen[(y + iy) * ScreenWidth + x + ix].blue  = pixels[iy * w + ix].blue;
        }
    }
    else
    {
      BGR* Screen = (BGR*)ScreenBuffer;
      for (int iy = image_offset_y; iy < image_offset_y + image_blit_height; iy++)
        for (int ix = image_offset_x; ix < image_offset_x + image_blit_width; ix++)
        {
          Screen[(y + iy) * ScreenWidth + x + ix].red   = pixels[iy * w + ix].red;
          Screen[(y + iy) * ScreenWidth + x + ix].green = pixels[iy * w + ix].green;
          Screen[(y + iy) * ScreenWidth + x + ix].blue  = pixels[iy * w + ix].blue;
        }
    }

  }
  else if (method == 2)
  {

    if (BitsPerPixel == 32)
    {
      for (int iy = image_offset_y; iy < image_offset_y + image_blit_height; iy++)
        for (int ix = image_offset_x; ix < image_offset_x + image_blit_width; ix++)
        {
          BGRA* dest = (BGRA*)ScreenBuffer + (y + iy) * ScreenWidth + x + ix;
          RGBA  src  = pixels[iy * w + ix];

          if (src.alpha == 255)
          {
            dest->red   = src.red;
            dest->green = src.green;
            dest->blue  = src.blue;
          }
          else if (src.alpha >= 0)
          {
            dest->red   = (dest->red   * (256 - src.alpha) + src.red   * src.alpha) / 256;
            dest->green = (dest->green * (256 - src.alpha) + src.green * src.alpha) / 256;
            dest->blue  = (dest->blue  * (256 - src.alpha) + src.blue  * src.alpha) / 256;
          }
        }
    }
    else
    {
      for (int iy = image_offset_y; iy < image_offset_y + image_blit_height; iy++)
        for (int ix = image_offset_x; ix < image_offset_x + image_blit_width; ix++)
        {
          BGR* dest  = (BGR*)ScreenBuffer + (y + iy) * ScreenWidth + x + ix;
          RGBA src   = pixels[iy * w + ix];

          if (src.alpha == 255)
          {
            dest->red   = src.red;
            dest->green = src.green;
            dest->blue  = src.blue;
          }
          else if (src.alpha >= 0)
          {
            dest->red   = (dest->red   * (256 - src.alpha) + src.red   * src.alpha) / 256;
            dest->green = (dest->green * (256 - src.alpha) + src.green * src.alpha) / 256;
            dest->blue  = (dest->blue  * (256 - src.alpha) + src.blue  * src.alpha) / 256;
          } // end for
        } // end for
    } // end if (BitsPerPixel)

  } // end if
}

////////////////////////////////////////////////////////////////////////////////

void EXPORT DirectGrab(int x, int y, int w, int h, RGBA* pixels)
{
  if (x < 0 ||
      y < 0 ||
      x + w > ScreenWidth ||
      y + h > ScreenHeight)
    return;

  if (BitsPerPixel == 32)
  {
    BGRA* Screen = (BGRA*)ScreenBuffer;
    for (int iy = 0; iy < h; iy++)
      for (int ix = 0; ix < w; ix++)
      {
        pixels[iy * w + ix].red   = Screen[(y + iy) * ScreenWidth + x + ix].red;
        pixels[iy * w + ix].green = Screen[(y + iy) * ScreenWidth + x + ix].green;
        pixels[iy * w + ix].blue  = Screen[(y + iy) * ScreenWidth + x + ix].blue;
        pixels[iy * w + ix].alpha = 255;
      }
  }
  else
  {
    BGR* Screen = (BGR*)ScreenBuffer;
    for (int iy = 0; iy < h; iy++)
      for (int ix = 0; ix < w; ix++)
      {
        pixels[iy * w + ix].red   = Screen[(y + iy) * ScreenWidth + x + ix].red;
        pixels[iy * w + ix].green = Screen[(y + iy) * ScreenWidth + x + ix].green;
        pixels[iy * w + ix].blue  = Screen[(y + iy) * ScreenWidth + x + ix].blue;
        pixels[iy * w + ix].alpha = 255;
      }
  }
}

////////////////////////////////////////////////////////////////////////////////
