#include <windows.h>
#include <stdio.h>
#include "video.h"
#include "internal.h"


static bool InitVDriver(int x, int y);
static bool CloseVDriver(void);

// this function should not be exposed
static bool (__stdcall * _FlipScreen)(void);

bool (__stdcall * SetClippingRectangle)(int x, int y, int w, int h);
bool (__stdcall * GetClippingRectangle)(int* x, int* y, int* w, int* h);
bool (__stdcall * ApplyColorMask)(RGBA mask);

IMAGE (__stdcall * CreateImage)(int width, int height, RGBA* data);
IMAGE (__stdcall * GrabImage)(int x, int y, int width, int height);
bool  (__stdcall * DestroyImage)(IMAGE image);
bool  (__stdcall * BlitImage)(IMAGE image, int x, int y);
int   (__stdcall * GetImageWidth)(IMAGE image);
int   (__stdcall * GetImageHeight)(IMAGE image);
RGBA* (__stdcall * LockImage)(IMAGE image);
void  (__stdcall * UnlockImage)(IMAGE image);
void  (__stdcall * DirectBlit)(int x, int y, int w, int h, RGBA* pixels, int method);
void  (__stdcall * DirectGrab)(int x, int y, int w, int h, RGBA* pixels);


static HWND      SphereWindow;
static HINSTANCE GraphicsDriver;
static int ScreenWidth;
static int ScreenHeight;


////////////////////////////////////////////////////////////////////////////////

bool InitVideo(HWND window, SPHERECONFIG* config)
{
  char graphicsdriver[MAX_PATH];

  sprintf(graphicsdriver, "system\\video\\%s", config->videodriver);

  // Loads driver
  GraphicsDriver = LoadLibrary(graphicsdriver);
  if (GraphicsDriver == NULL)
    return false;

  // Gets addresses of all of the graphics functions
  _FlipScreen          = (bool (__stdcall *)(void))                      GetProcAddress(GraphicsDriver, "FlipScreen");
  SetClippingRectangle = (bool (__stdcall *)(int, int, int, int))        GetProcAddress(GraphicsDriver, "SetClippingRectangle");
  GetClippingRectangle = (bool (__stdcall *)(int*, int*, int*, int*))    GetProcAddress(GraphicsDriver, "GetClippingRectangle");
  ApplyColorMask       = (bool (__stdcall *)(RGBA))                      GetProcAddress(GraphicsDriver, "ApplyColorMask");

  CreateImage          = (IMAGE (__stdcall *)(int, int, RGBA*))   GetProcAddress(GraphicsDriver, "CreateImage");
  GrabImage            = (IMAGE (__stdcall *)(int, int, int, int))GetProcAddress(GraphicsDriver, "GrabImage");
  DestroyImage         = (bool  (__stdcall *)(IMAGE))                    GetProcAddress(GraphicsDriver, "DestroyImage");
  BlitImage            = (bool  (__stdcall *)(IMAGE, int, int))          GetProcAddress(GraphicsDriver, "BlitImage");
  GetImageWidth        = (int   (__stdcall *)(IMAGE))                    GetProcAddress(GraphicsDriver, "GetImageWidth");
  GetImageHeight       = (int   (__stdcall *)(IMAGE))                    GetProcAddress(GraphicsDriver, "GetImageHeight");
  LockImage            = (RGBA* (__stdcall *)(IMAGE))                    GetProcAddress(GraphicsDriver, "LockImage");
  UnlockImage          = (bool  (__stdcall *)(IMAGE))                    GetProcAddress(GraphicsDriver, "UnlockImage");
  DirectBlit           = (void  (__stdcall *)(int, int, int, int, RGBA*, int))GetProcAddress(GraphicsDriver, "DirectBlit");
  DirectGrab           = (void  (__stdcall *)(int, int, int, int, RGBA*))GetProcAddress(GraphicsDriver, "DirectGrab");

  if (!_FlipScreen ||
      !SetClippingRectangle ||
      !GetClippingRectangle ||
      !ApplyColorMask ||
      !CreateImage ||
      !GrabImage ||
      !DestroyImage ||
      !BlitImage ||
      !GetImageWidth ||
      !GetImageHeight ||
      !LockImage ||
      !UnlockImage ||
      !DirectBlit ||
      !DirectGrab)
  {
    FreeLibrary(GraphicsDriver);
    return false;
  }

  SphereWindow = window;

  // Initializes driver
  if (InitVDriver(320, 240) == false)
  {
    FreeLibrary(GraphicsDriver);
    return false;
  }

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool CloseVideo(void)
{
  if (GraphicsDriver)
  {
    CloseVDriver();
    FreeLibrary(GraphicsDriver);
  }
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool InitVDriver(int x, int y)
{
  bool (__stdcall * InitVideoDriver)(HWND window, int screen_width, int screen_height);

  ScreenWidth = x;
  ScreenHeight = y;

  // Gets address of driver init func
  InitVideoDriver = (bool (__stdcall *)(HWND, int, int))GetProcAddress(GraphicsDriver, "InitVideoDriver");
  if (InitVideoDriver == NULL)
    return false;

  return InitVideoDriver(SphereWindow, ScreenWidth, ScreenHeight);
}

////////////////////////////////////////////////////////////////////////////////

bool CloseVDriver(void)
{
  bool (__stdcall * CloseVideoDriver)(void);

  // Gets address of
  CloseVideoDriver = (bool (__stdcall *)(void))GetProcAddress(GraphicsDriver, "CloseVideoDriver");
  if (CloseVideoDriver == NULL)
  {
    FreeLibrary(GraphicsDriver);
    return false;
  }

  return CloseVideoDriver();
}

////////////////////////////////////////////////////////////////////////////////

bool SwitchResolution(int x, int y)
{
  if (x == ScreenWidth && y == ScreenHeight)
    return true;

  CloseVDriver();
  if (!InitVDriver(x, y))
  {
    FreeLibrary(GraphicsDriver);
    QuitMessage("Could not switch resolutions");
  }
  return true;
}

////////////////////////////////////////////////////////////////////////////////

int GetScreenWidth(void)
{
  return ScreenWidth;
}

////////////////////////////////////////////////////////////////////////////////

int GetScreenHeight(void)
{
  return ScreenHeight;
}

////////////////////////////////////////////////////////////////////////////////

bool FlipScreen(void)
{
  static int NumFlips;
  if (NumFlips++ % 8 == 0)
    UpdateSystem();
  return _FlipScreen();
}

////////////////////////////////////////////////////////////////////////////////
