#include <windows.h>
#include <mmsystem.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include "internal.h"
#include "input.h"
#include "../sphere.hpp"


LRESULT CALLBACK SphereWindowProc(HWND window, UINT msg, WPARAM wparam, LPARAM lparam);


static HWND SphereWindow;


////////////////////////////////////////////////////////////////////////////////

static void LoadSphereConfiguration(HINSTANCE instance, SPHERECONFIG* config)
{
  char ConfigPath[MAX_PATH];
#ifdef NDEBUG
  GetModuleFileName(instance, ConfigPath, MAX_PATH);
  *strrchr(ConfigPath, '\\') = 0;
#else
  GetCurrentDirectory(MAX_PATH, ConfigPath);
#endif
  strcat(ConfigPath, "\\engine.ini");

  // Loads configuration settings
  LoadSphereConfig(config, ConfigPath);
}

////////////////////////////////////////////////////////////////////////////////

int __cdecl main(int argc, const char**argv)
{
  // seed the random number generator
  srand(time(NULL));

  // load it from a file
  SPHERECONFIG Config;
  LoadSphereConfiguration(GetModuleHandle(NULL), &Config);
  if (strlen(Config.videodriver) == 0)
  {
    // tell user
    MessageBox(NULL, "Sphere configuration not found.\n"
                     "Sphere will now run config.exe", "Sphere", MB_OK);

    // execute setup.exe
    STARTUPINFO si;
    ZeroMemory(&si, sizeof(si));
    si.cb = sizeof(si);

    PROCESS_INFORMATION pi;
    if (CreateProcess("config.exe", "", NULL, NULL, TRUE,
          NORMAL_PRIORITY_CLASS, NULL, NULL, &si, &pi) == FALSE)
    {
      MessageBox(NULL, "Could not run config.exe", "Sphere", MB_OK);
      return 0;
    }

    while (WaitForSingleObject(pi.hProcess, 100) != WAIT_OBJECT_0)
      ;

    LoadSphereConfiguration(GetModuleHandle(NULL), &Config);
    if (strlen(Config.videodriver) == 0)
    {
      MessageBox(NULL, "Could not load configuration", "Sphere", MB_OK);
      return 0;
    }
  }

  // register the window class
  WNDCLASS wc;
  memset(&wc, 0, sizeof(wc));
  wc.lpfnWndProc   = SphereWindowProc;
  wc.hInstance     = GetModuleHandle(NULL);
  wc.hIcon         = LoadIcon(NULL, IDI_APPLICATION);
  wc.hCursor       = NULL;
  wc.hbrBackground = NULL;
  wc.lpszClassName = "SphereWindowClass";
  if (RegisterClass(&wc) == 0)
  {
    MessageBox(NULL, "Error: Could not register the window class", "Sphere", MB_OK);
    return 0;
  }

  // create the sphere window
  SphereWindow = CreateWindow(,
    "SphereWindowClass", "Sphere",
    WS_CAPTION | WS_MINIMIZEBOX | WS_POPUP | WS_VISIBLE,
    CW_USEDEFAULT, 0, CW_USEDEFAULT, 0,
    NULL, NULL, GetModuleHandle(NULL), NULL);
  if (SphereWindow == NULL)
  {
    MessageBox(NULL, "Error: Could not create the window", "Sphere", MB_OK);
    return 0;
  }

  // initialize video subsystem
  if (InitVideo(SphereWindow, &Config) == false)
  {
    DestroyWindow(SphereWindow);
    MessageBox(NULL, "Error: Could not initialize video subsystem", "Sphere", MB_OK);
    return 0;
  }

  // initialize audio subsystem
  if (InitAudio(SphereWindow, &Config) == false)
  {
    CloseVideo();
    DestroyWindow(SphereWindow);
    MessageBox(NULL, "Error: Could not initialize audio subsystem", "Sphere", MB_OK);
    return 0;
  }

  // Initializes input subsystem
  if (InitInput(SphereWindow, &Config) == false)
  {
    CloseAudio();
    CloseVideo();
    DestroyWindow(SphereWindow);
    MessageBox(NULL, "Error: Could not initialize input subsystem", "Sphere", MB_OK);
    return 0;
  }

  UpdateSystem();
  CSphereEngine().Run(argc, argv);
  UpdateSystem();

  // Clean up
  CloseInput();
  CloseAudio();
  CloseVideo();
  DestroyWindow(SphereWindow);
  return 0;
}

////////////////////////////////////////////////////////////////////////////////

LRESULT CALLBACK SphereWindowProc(HWND window, UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch (msg)
  {
    case WM_SYSKEYDOWN:
      if (wparam == 115) // if user hit ALT+F4
      {
        CloseInput();
        CloseAudio();
        CloseVideo();
        DestroyWindow(window);
        exit(0);
      }
      return 0;

    case WM_KEYDOWN:
      OnKeyDown(wparam);
      return 0;

    ////////////////////////////////////////////////////////////////////////////

    case WM_SYSKEYUP:
    case WM_KEYUP:
      OnKeyUp(wparam);
      return 0;

    ////////////////////////////////////////////////////////////////////////////

    case WM_MOUSEMOVE:
      SetCursor(NULL);
      OnMouseMove(LOWORD(lparam), HIWORD(lparam));
      return 0;

    case WM_LBUTTONDOWN:
      OnMouseDown(MOUSE_LEFT);
      return 0;

    case WM_LBUTTONUP:
      OnMouseUp(MOUSE_LEFT);
      return 0;

    case WM_MBUTTONDOWN:
      OnMouseDown(MOUSE_MIDDLE);
      return 0;

    case WM_MBUTTONUP:
      OnMouseUp(MOUSE_MIDDLE);
      return 0;

    case WM_RBUTTONDOWN:
      OnMouseDown(MOUSE_RIGHT);
      return 0;

    case WM_RBUTTONUP:
      OnMouseUp(MOUSE_RIGHT);
      return 0;

    ////////////////////////////////////////////////////////////////////////////

    case WM_PAINT:  // handle the paint message, just don't do anything
    {
      PAINTSTRUCT ps;
      BeginPaint(window, &ps);
      EndPaint(window, &ps);
      return 0;
    }

    ////////////////////////////////////////////////////////////////////////////

    case MM_MCINOTIFY:  // Make MIDI files repeat
    {
      char ret[512];
      mciSendString("stop midifile", ret, 500, window);
      mciSendString("seek midifile to start", ret, 500, window);
      mciSendString("play midifile notify", ret, 500, window);
      return 0;
    }

    ////////////////////////////////////////////////////////////////////////////

    default:
      return DefWindowProc(window, msg, wparam, lparam);
  }
}

////////////////////////////////////////////////////////////////////////////////

void QuitMessage(const char* message)
{
  CloseInput();
  CloseAudio();
  CloseVideo();
  DestroyWindow(SphereWindow);
  UpdateSystem();
  MessageBox(NULL, message, "Sphere", MB_OK);
  exit(1);
}

////////////////////////////////////////////////////////////////////////////////

void UpdateSystem(void)
{
  int count = 0;
  MSG msg;
  while (++count <= 5 && PeekMessage(&msg, NULL, 0, 0, PM_NOREMOVE))
  {
    if (GetMessage(&msg, NULL, 0, 0))
    {
      TranslateMessage(&msg);
      DispatchMessage(&msg);
    }
    else
      DestroyWindow(SphereWindow);
  }
}

////////////////////////////////////////////////////////////////////////////////
