#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "version.h"
#include "sphere.hpp"
#include "engine.hpp"
#include "render.h"
#include "inputx.h"

#include "system.h"
#include "input.h"
#include "timer.h"
#include "filesystem.h"

#include "simage.hpp"

#include "configfile.h"
#include "x++.hpp"


////////////////////////////////////////////////////////////////////////////////

CSphereEngine::CSphereEngine()
: m_SystemFont(NULL)
, m_SystemWindowStyle(NULL)
, m_SystemArrow(NULL)
, m_SystemUpArrow(NULL)
, m_SystemDownArrow(NULL)
{
  EnterDirectory("system");

  char system_font[520];
  char system_window_style[520];
  char system_arrow[520];
  char system_up_arrow[520];
  char system_down_arrow[520];

  // read all system filenames from system.ini
  ReadConfigFileString("system.ini", "", "IntroImage",  m_IntroImage,        512, "unknown");
  ReadConfigFileString("system.ini", "", "IntroSound",  m_IntroSound,        512, "unknown");
  ReadConfigFileString("system.ini", "", "Font",        system_font,         512, "unknown");
  ReadConfigFileString("system.ini", "", "WindowStyle", system_window_style, 512, "unknown");
  ReadConfigFileString("system.ini", "", "Arrow",       system_arrow,        512, "unknown");
  ReadConfigFileString("system.ini", "", "UpArrow",     system_up_arrow,     512, "unknown");
  ReadConfigFileString("system.ini", "", "DownArrow",   system_down_arrow,   512, "unknown");    

  // get full paths so they can be passed on to the game engine
  m_SystemFiles.font         = new char[520];
  m_SystemFiles.window_style = new char[520];
  m_SystemFiles.arrow        = new char[520];
  m_SystemFiles.up_arrow     = new char[520];
  m_SystemFiles.down_arrow   = new char[520];

  GetAbsolutePath(m_SystemFiles.font,         system_font);
  GetAbsolutePath(m_SystemFiles.window_style, system_window_style);
  GetAbsolutePath(m_SystemFiles.arrow,        system_arrow);
  GetAbsolutePath(m_SystemFiles.up_arrow,     system_up_arrow);
  GetAbsolutePath(m_SystemFiles.down_arrow,   system_down_arrow);

  LeaveDirectory();
}

////////////////////////////////////////////////////////////////////////////////

CSphereEngine::~CSphereEngine()
{
  delete[] m_SystemFiles.font;
  delete[] m_SystemFiles.window_style;
  delete[] m_SystemFiles.arrow;
  delete[] m_SystemFiles.up_arrow;
  delete[] m_SystemFiles.down_arrow;
}

////////////////////////////////////////////////////////////////////////////////

void
CSphereEngine::Run(int num_arguments, const char** arguments)
{
  bool manualselection = false;
  char game[520];

  // check for manual game selection
  for (int i = 0; i < num_arguments - 1; i++)  // if last parameter is -game, it doesn't mean anything
  {
    if (strcmp(arguments[i], "-game") == 0)
    {
      strcpy(game, arguments[i + 1]);
      manualselection = true;
      break;
    }
  }

  // start the game specified on the command line
  if (manualselection)
  {
    EnterDirectory("games");
    RunGame(game);
    LeaveDirectory();
  }
  else
  {  
    Intro();

    EnterDirectory("games");

    while (MainMenu(game))
      RunGame(game);

    LeaveDirectory();
  }
}

////////////////////////////////////////////////////////////////////////////////

bool
CSphereEngine::Intro()
{
  EnterDirectory("system");

  // load the intro files
  IMAGE IntroImage = SLoadImage(m_IntroImage);
  if (IntroImage == NULL)
    QuitMessage("Error: Could not load intro image");

  SOUNDEFFECT IntroSound = LoadSoundEffect(m_IntroSound);
  if (IntroSound == NULL)
  {
    DestroyImage(IntroImage);
    QuitMessage("Error: Could not load intro sound");
  }

  PlaySoundEffect(IntroSound);

  // fade in
  BlitImage(IntroImage, 0, 0);
  FadeIn(500);

  // display the image until key is pressed
  while (KeysLeft() == false)
  {
    RefreshInput();
    BlitImage(IntroImage, 0, 0);
    FlipScreen();
  }
  GetKey();  // eat the key that was pressed

  // fade out
  BlitImage(IntroImage, 0, 0);
  FadeOut(500);
  
  // clean up
  DestroySoundEffect(IntroSound);
  DestroyImage(IntroImage);

  LeaveDirectory();
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
CSphereEngine::MainMenu(char* game)
{
  LoadSystemObjects();

  // get all game subdirectories (must have a game.inf file)
  int numgames;
  char **gamelist;
  GetGameList(&numgames, &gamelist);

  // if there are no games, exit with an error message
  if (numgames == 0)
  {
    while (!KeysLeft())
    {
      ClearScreen();
      m_SystemFont->DrawText(8, 8, "Error: No games detected");
      FlipScreen();
    }
    GetKey();  // eat the key that was pressed

    return false;
  }

  // fill the menu
  sMenu menu;
  for (int i = 0; i < numgames; i++)
    menu.AddItem(gamelist[i] + strlen(gamelist[i]) + 1);

  // draw the background
  ClearScreen();
  m_SystemFont->SetMask(rgbaRed);
  m_SystemFont->DrawText(8, 8, "Sphere");
  m_SystemFont->DrawText(8 + m_SystemFont->GetStringWidth("Sphere "), 8, SPHERE_VERSION);
  m_SystemFont->SetMask(rgbaWhite);
  m_SystemFont->DrawText(8, 24, "Chad Austin (c) 1997-2000");

  // execute the menu
  MENUDISPLAYDATA mdd = {
    m_SystemFont,
    m_SystemWindowStyle,
    m_SystemArrow,
    m_SystemUpArrow,
    m_SystemDownArrow,
  };

  const int MENU_Y = 64;
  int selection = menu.ExecuteV(mdd, 16, MENU_Y, 320 - 16 * 2, 240 - 16 - MENU_Y, 16);
  
  // get the name of the selected game
  if (selection == -1)
    strcpy(game, "");
  else
    strcpy(game, gamelist[selection]);

  // clean up
  FreeGameList(&numgames, &gamelist);
  DestroySystemObjects();

  // go home  ^_^
  if (strlen(game))
    return true;
  return false;
}

////////////////////////////////////////////////////////////////////////////////

bool
CSphereEngine::RunGame(const char* game)
{
  CGameEngine(&m_SystemFiles).Run(game);
  ClearKeyQueue();
  return true;
}

////////////////////////////////////////////////////////////////////////////////

// needed for qsort()
#ifndef CDECL
#ifdef _WIN32
#define CDECL __cdecl
#else
#define CDECL
#endif
#endif

int CDECL stringcompare(const void* a, const void* b)
{
  const char* _a = *(const char**)a;
  const char* _b = *(const char**)b;
  _a += strlen(_a) + 1;
  _b += strlen(_b) + 1;
  return strcmp_ci(_a, _b);
}

////////////////////////////////////////////////////////////////////////////////

void
CSphereEngine::GetGameList(int* numgames, char*** gamelist)
{
  *numgames = 0;
  *gamelist = NULL;

  DIRECTORYLIST dl = BeginDirectoryList("*");

  while (!DirectoryListDone(dl))
  {
    char directory[520];
    GetNextDirectory(dl, directory);

    EnterDirectory(directory);

    // read the game name
    char gamename[520];
    ReadConfigFileString("game.inf", "", "name", gamename, 520, "");

    // if the game name is empty, the game doesn't exist
    if (strlen(gamename) != 0)
    {
      *gamelist = (char**)realloc(*gamelist, sizeof(*gamelist) * ((*numgames) + 1));
      (*gamelist)[*numgames] = (char*)malloc(520);
      strcpy((*gamelist)[*numgames], directory);
      strcpy((*gamelist)[*numgames] + strlen((*gamelist)[*numgames]) + 1, gamename);
      (*numgames)++;
    }

    LeaveDirectory();
  }

  EndDirectoryList(dl);

  // alphabetize the menu
  qsort(*gamelist, *numgames, sizeof(**gamelist), stringcompare);
}

////////////////////////////////////////////////////////////////////////////////

void
CSphereEngine::FreeGameList(int* numgames, char*** gamelist)
{
  for (int i = 0; i < *numgames; i++)
    free((*gamelist)[i]);
  free(*gamelist);

  *numgames = 0;
  *gamelist = NULL;
}

////////////////////////////////////////////////////////////////////////////////

void
CSphereEngine::LoadSystemObjects()
{
  m_SystemFont = new SFONT;
  if (m_SystemFont->Load(m_SystemFiles.font) == false)
    QuitMessage("Error: Could not load system font");

  m_SystemWindowStyle = new SWINDOWSTYLE;
  if (m_SystemWindowStyle->Load(m_SystemFiles.window_style) == false)
  {
    delete m_SystemFont;
    QuitMessage("Error: Could not load system window style");
  }

  m_SystemArrow = SLoadImage(m_SystemFiles.arrow);
  if (m_SystemArrow == NULL)
  {
    delete m_SystemFont;
    delete m_SystemWindowStyle;
    QuitMessage("Error: Could not load system arrow");
  }

  m_SystemUpArrow = SLoadImage(m_SystemFiles.up_arrow);
  if (m_SystemUpArrow == NULL)
  {
    delete m_SystemFont;
    delete m_SystemWindowStyle;
    DestroyImage(m_SystemArrow);
    QuitMessage("Error: Could not load system up arrow");
  }

  m_SystemDownArrow = SLoadImage(m_SystemFiles.down_arrow);
  if (m_SystemDownArrow == NULL)
  {
    delete m_SystemFont;
    delete m_SystemWindowStyle;
    DestroyImage(m_SystemArrow);
    DestroyImage(m_SystemUpArrow);
    QuitMessage("Error: Could not load system down arrow");
  }
}

////////////////////////////////////////////////////////////////////////////////

void
CSphereEngine::DestroySystemObjects()
{
  delete m_SystemFont;
  m_SystemFont = NULL;

  delete m_SystemWindowStyle;
  m_SystemWindowStyle = NULL;

  DestroyImage(m_SystemArrow);
  DestroyImage(m_SystemUpArrow);
  DestroyImage(m_SystemDownArrow);
}

////////////////////////////////////////////////////////////////////////////////
