#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "sfont.hpp"
#include "x++.hpp"
#include "video.h"


////////////////////////////////////////////////////////////////////////////////

SFONT::SFONT()
: m_CurrentColor(-1)
{
}

////////////////////////////////////////////////////////////////////////////////

SFONT::~SFONT()
{
  Destroy();
}

////////////////////////////////////////////////////////////////////////////////

bool
SFONT::Load(const char* filename)
{
  // load file
  if (m_Font.Load(filename) == false)
    return false;

  // Sphere fonts must have 256 characters
  if (m_Font.GetNumCharacters() != 256)
    return false;

  Destroy();

  Color clr;
  clr.color = rgbaWhite;

  // allocate and initialize the new character buffer
  for (int i = 0; i < 256; i++)
  {
    sFontCharacter& c = m_Font.GetCharacter(i);
    clr.characters[i] = new RGBA[c.GetWidth() * c.GetHeight()];
    memcpy(clr.characters[i], c.GetPixels(), c.GetWidth() * c.GetHeight() * sizeof(RGBA));
  }

  m_CurrentColor = 0;
  m_Colors.push_back(clr);

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
SFONT::SetMask(RGBA color)
{
  if (m_CurrentColor == -1)
    return false;

  // see if color is cached
  for (int i = 0; i < m_Colors.size(); i++)
  {
    if (memcmp(&m_Colors[i].color, &color, sizeof(RGBA)) == 0)
    {
      m_CurrentColor = i;
      return true;
    }
  }

  Color clr;
  clr.color = color;

  // now apply this color to the new font
  for (int i = 0; i < 256; i++)
  {
    sFontCharacter& c = m_Font.GetCharacter(i);
    clr.characters[i] = new RGBA[c.GetWidth() * c.GetHeight()];
    memcpy(clr.characters[i], c.GetPixels(), c.GetWidth() * c.GetHeight() * sizeof(RGBA));

    RGBA* dst = clr.characters[i] + c.GetWidth() * c.GetHeight();
    while (dst > clr.characters[i])
    {
      dst--;

      dst->red   = (color.alpha * color.red   + (255 - color.alpha) * dst->red)   / 256;
      dst->green = (color.alpha * color.green + (255 - color.alpha) * dst->green) / 256;
      dst->blue  = (color.alpha * color.blue  + (255 - color.alpha) * dst->blue)  / 256;
      dst->alpha = color.alpha * dst->alpha / 256;
    }
  }

  
  m_CurrentColor = m_Colors.size();
  m_Colors.push_back(clr);

  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
SFONT::DrawText(int x, int y, const char* text) const
{
  if (m_CurrentColor == -1)
    return false;

  while (*text)
  {
    const sFontCharacter& c = m_Font.GetCharacter(*text);
    DirectBlit(x, y, c.GetWidth(), c.GetHeight(), m_Colors[m_CurrentColor].characters[*text], 2);
    x += c.GetWidth();
    text++;
  }
  return true; 
}

////////////////////////////////////////////////////////////////////////////////

bool
SFONT::DrawTextBox(int x, int y, int w, int h, int offset, const char* text) const
{
  if (m_CurrentColor == -1)
    return false;

  const char* p = text;

  if (*p == 0)
    return true;

  int numwords = 0;
  char** words = NULL;

  // parse the text into words
  bool done = false;
  while (!done)
  {
    words = (char**)realloc(words, sizeof(char*) * ++numwords);
    words[numwords - 1] = (char*)malloc(520 * sizeof(char));

    // if it's a newline, add it by itself
    if (*p == '\n')
    {
      words[numwords - 1][0] = *p++;
      words[numwords - 1][1] = 0;
    }
    else
    // add non-whitespace characters
    {
      int w = 0;
      while (*p && !isspace(*p))
        words[numwords - 1][w++] = *p++;
      words[numwords - 1][w] = 0;
    }

    if (*p != 0)
    {
      // skip whitespace characters
      while (*p != 0 && isspace(*p) && *p != '\n')
        p++;
    }

    if (*p == 0)
      done = true;
  }

  // render the words
  int cx = 0;
  int cy = 0;
  for (int i = 0; i < numwords; i++)
  {
    char* wd = words[i];

    if (strcmp(wd, "\n") == 0 || cx + GetStringWidth(wd) > w)
    {
      cy += 12;
      cx = 0;
    }

    DrawText(cx + x, cy + y + offset, wd);

    cx += GetStringWidth(wd) + GetStringWidth(" ");
  }

  // free words
  for (int i = 0; i < numwords; i++)
    free(words[i]);
  free(words);
 
  return true;
}

////////////////////////////////////////////////////////////////////////////////

int
SFONT::GetMaxHeight() const
{
  int max_height = 0;
  for (int i = 0; i < m_Font.GetNumCharacters(); i++)
    if (m_Font.GetCharacter(i).GetHeight() > max_height)
      max_height = m_Font.GetCharacter(i).GetHeight();
  return max_height;
}

////////////////////////////////////////////////////////////////////////////////

int
SFONT::GetStringWidth(const char* string) const
{
  int width = 0;
  while (*string)
  {
    width += m_Font.GetCharacter(*string).GetWidth();
    string++;
  }
  return width;
}

////////////////////////////////////////////////////////////////////////////////

void
SFONT::Destroy()
{
  for (int i = 0; i < m_Colors.size(); i++)
    for (int j = 0; j < 256; j++)
      delete[] m_Colors[i].characters[j];
  m_Colors.clear();
  m_CurrentColor = -1;
}

////////////////////////////////////////////////////////////////////////////////
