#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "script.hpp"

#include "engineinterface.hpp"
#include "menu.hpp"
#include "render.h"
#include "inputx.h"

#include "smap.hpp"
#include "sfont.hpp"
#include "swindowstyle.hpp"
#include "simage.hpp"

#include "system.h"
#include "input.h"
#include "timer.h"


static IEngineInterface* EngineInterface;


struct SS_SURFACE
{
  CImage32* surface;
  RGBA      pen_color;
  RGBA      brush_color;
};


const int MAX_FRAME_SKIP = 10;


////////////////////////////////////////////////////////////////////////////////

bool InitializeScriptSystem(IEngineInterface* interface)
{
  EngineInterface = interface;
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool ShutdownScriptSystem()
{
  EngineInterface = NULL;
  return true;
}

////////////////////////////////////////////////////////////////////////////////

CScript::CScript()
: m_Runtime(NULL)
, m_Context(NULL)
, m_Script(NULL)
, m_Global(NULL)
, m_HasError(false)
, m_FrameRate(0)
, m_ShouldRender(true)
, m_FramesSkipped(0)
, m_IdealTime(0)
{
}

////////////////////////////////////////////////////////////////////////////////

CScript::~CScript()
{
  // destroy JS objects
  if (m_Runtime) {
    JS_DestroyRuntime(m_Runtime);
  }
}

////////////////////////////////////////////////////////////////////////////////

bool
CScript::Load(const char* filename)
{
  // initialize JS engine
  m_Runtime = JS_NewRuntime(4 * 1024 * 1024);
  if (m_Runtime == NULL) {
    return false;
  }

  // create context
  m_Context = JS_NewContext(m_Runtime, 8 * 1024);
  if (m_Context == NULL) {
    JS_DestroyRuntime(m_Runtime);
    m_Runtime = NULL;
    return false;
  }

  // create global class
  static JSClass global_class = {
    "global", 0,
    JS_PropertyStub, JS_PropertyStub, JS_PropertyStub, JS_PropertyStub,
    JS_EnumerateStub, JS_ResolveStub, JS_ConvertStub, JS_FinalizeStub,
    NULL, NULL, NULL, NULL, NULL, NULL, NULL,
  };
  
  // create global object
  m_Global = JS_NewObject(m_Context, &global_class, NULL, NULL);
  if (m_Global == NULL) {
    JS_DestroyContext(m_Context);
    m_Context = NULL;
    JS_DestroyRuntime(m_Runtime);
    m_Runtime = NULL;
    return false;
  }

  // assign this object to the context
  JS_SetContextPrivate(m_Context, this);

  JS_InitStandardClasses(m_Context, m_Global);
  JS_SetErrorReporter(m_Context, ErrorReporter);

  InitializeSphereFunctions();
  InitializeSphereObjects();

  // compile the script
  m_Script = JS_CompileFile(m_Context, m_Global, filename);
  if (m_Script == NULL) {
    m_Global = NULL;
    JS_DestroyContext(m_Context);
    m_Context = NULL;
    JS_DestroyRuntime(m_Runtime);
    m_Runtime = NULL;
    
    if (m_HasError)
      EngineInterface->ExitWithMessage(m_Error.BuildMessage().c_str());
    return m_HasError;
  }

  // evaluate the script
  jsval val;
  JSBool result = JS_ExecuteScript(m_Context, m_Global, m_Script, &val);
  if (!result) {
    JS_DestroyScript(m_Context, m_Script);
    m_Script = NULL;
    m_Global = NULL;
    JS_DestroyContext(m_Context);
    m_Context = NULL;
    JS_DestroyRuntime(m_Runtime);
    m_Runtime = NULL;
    
    if (m_HasError)
      EngineInterface->ExitWithMessage(m_Error.BuildMessage().c_str());
    return m_HasError;
  }
    
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
CScript::ExecuteFunction(const char* function)
{
  // check if function actually exists
  jsval val;
  if (JS_LookupProperty(m_Context, m_Global, function, &val) == JS_FALSE)
    return false;
  if (val == JSVAL_VOID)
    return false;

  // call the function
  JS_CallFunctionName(m_Context, m_Global, function, 0, NULL, &val);
  if (m_HasError)
    EngineInterface->ExitWithMessage(m_Error.BuildMessage().c_str());
  return true;
}

////////////////////////////////////////////////////////////////////////////////

void
CScript::InitializeSphereFunctions()
{
  static JSFunctionSpec functions[] = {
    #define SS_FUNCTION(name, numargs) { #name, ss##name, numargs, 0, 0 },
    #include "ss_functions.table"
    #undef SS_FUNCTION
    { 0, 0, 0, 0, 0 },
  };

  JS_DefineFunctions(m_Context, m_Global, functions);

  // system functions that don't need to be native
  char* script =
    "function Color(r, g, b, a) {"
    "  if (a == undefined) a = 255;"
    "  return { red: r, green: g, blue: b, alpha: a };"
    "}"
  ;

  jsval v;
  JS_EvaluateScript(m_Context, m_Global, script, strlen(script), "", 0, &v);
}

////////////////////////////////////////////////////////////////////////////////

void
CScript::InitializeSphereObjects()
{
  static struct {
    const char* name;
    const char* value;
  } constants[] = {
    
    // colors

    { "Black",     "Color(000, 000, 000)" },
    { "DarkGrey",  "Color(063, 063, 063)" },
    { "Grey",      "Color(127, 127, 127)" },
    { "LightGrey", "Color(191, 191, 191)" },
    { "White",     "Color(255, 255, 255)" },
    { "Red",       "Color(255, 000, 000)" },
    { "Green",     "Color(000, 255, 000)" },
    { "Blue",      "Color(000, 000, 255)" },
    { "Cyan",      "Color(000, 255, 255)" },
    { "Magenta",   "Color(255, 000, 255)" },
    { "Yellow",    "Color(255, 255, 000)" },

    // keyboard constants

    { "KEY_ESCAPE",     "1" },
    { "KEY_F1",         "2" },
    { "KEY_F2",         "3" },
    { "KEY_F3",         "4" },
    { "KEY_F4",         "5" },
    { "KEY_F5",         "6" },
    { "KEY_F6",         "7" },
    { "KEY_F7",         "8" },
    { "KEY_F8",         "9" },
    { "KEY_F9",         "10" },
    { "KEY_F10",        "11" },
    { "KEY_F11",        "12" },
    { "KEY_F12",        "13" },
    { "KEY_TILDE",      "14" },
    { "KEY_0",          "15" },
    { "KEY_1",          "16" },
    { "KEY_2",          "17" },
    { "KEY_3",          "18" },
    { "KEY_4",          "19" },
    { "KEY_5",          "20" },
    { "KEY_6",          "21" },
    { "KEY_7",          "22" },
    { "KEY_8",          "23" },
    { "KEY_9",          "24" },
    { "KEY_MINUS",      "25" },
    { "KEY_EQUALS",     "26" },
    { "KEY_BACKSPACE",  "27" },
    { "KEY_TAB",        "28" },
    { "KEY_A",          "29" },
    { "KEY_B",          "30" },
    { "KEY_C",          "31" },
    { "KEY_D",          "32" },
    { "KEY_E",          "33" },
    { "KEY_F",          "34" },
    { "KEY_G",          "35" },
    { "KEY_H",          "36" },
    { "KEY_I",          "37" },
    { "KEY_J",          "38" },
    { "KEY_K",          "39" },
    { "KEY_L",          "40" },
    { "KEY_M",          "41" },
    { "KEY_N",          "42" },
    { "KEY_O",          "43" },
    { "KEY_P",          "44" },
    { "KEY_Q",          "45" },
    { "KEY_R",          "46" },
    { "KEY_S",          "47" },
    { "KEY_T",          "48" },
    { "KEY_U",          "49" },
    { "KEY_V",          "50" },
    { "KEY_W",          "51" },
    { "KEY_X",          "52" },
    { "KEY_Y",          "53" },
    { "KEY_Z",          "54" },
    { "KEY_SHIFT",      "55" },
    { "KEY_CTRL",       "56" },
    { "KEY_ALT",        "57" },
    { "KEY_SPACE",      "58" },
    { "KEY_OPENBRACE",  "59" },
    { "KEY_CLOSEBRACE", "60" },
    { "KEY_SEMICOLON",  "61" },
    { "KEY_APOSTROPHE", "62" },
    { "KEY_COMMA",      "63" },
    { "KEY_PERIOD",     "64" },
    { "KEY_SLASH",      "65" },
    { "KEY_BACKSLASH",  "66" },
    { "KEY_ENTER",      "67" },
    { "KEY_INSERT",     "68" },
    { "KEY_DELETE",     "69" },
    { "KEY_HOME",       "70" },
    { "KEY_END",        "71" },
    { "KEY_PAGEUP",     "72" },
    { "KEY_PAGEDOWN",   "73" },
    { "KEY_UP",         "74" },
    { "KEY_RIGHT",      "75" },
    { "KEY_DOWN",       "76" },
    { "KEY_LEFT",       "77" },
  };

  // initialize the constants
  for (int i = 0; i < sizeof(constants) / sizeof(*constants); i++) {
    string s;
    s += "this.";
    s += constants[i].name;
    s += " = ";
    s += constants[i].value;

    jsval rval;
    JS_EvaluateScript(m_Context, m_Global, s.c_str(), s.length(), "", 0, &rval);
  }
}

////////////////////////////////////////////////////////////////////////////////

void
CScript::ErrorReporter(JSContext* cx, const char* message, JSErrorReport* report)
{
  // somehow flag an error and stop the script
  CScript* This = (CScript*)JS_GetContextPrivate(cx);

  // only display first error
  if (This->m_HasError)
    return;

  This->m_HasError = true;
  This->m_Error.message = message;
  This->m_Error.filename = (report->filename ? report->filename : "");
  This->m_Error.line_number = report->lineno;
  if (report->linebuf) {
    This->m_Error.line = report->linebuf;
    This->m_Error.token_offset = report->tokenptr - report->linebuf;
  } else {
    This->m_Error.line = "";
    This->m_Error.token_offset = 0;
  }
}

////////////////////////////////////////////////////////////////////////////////

void
CScript::SetError(const char* message)
{
  m_HasError = true;
  m_Error.message = message;
  m_Error.filename = "";
  m_Error.line_number = 0;
  m_Error.line = "";
  m_Error.token_offset = 0;
}

////////////////////////////////////////////////////////////////////////////////

inline string itos(int i) {
  char s[80];
  sprintf(s, "%d", i);
  return s;
}

string
CScript::error__::BuildMessage()
{
  string buf;
  buf += "Script error in '" + filename + "', line: " + itos(line_number) + "\n";
  buf += "\n";
  buf += message + "\n";
  buf += line + "\n";

  return buf;
}


////////////////////////////////////////////////////////////////////////////////

// system function definition macros and inline functions

#define begin_func(name, minargs)                                                                \
  JSBool CScript::ss##name(JSContext* cx, JSObject* obj, uintN argc, jsval* argv, jsval* rval) { \
    CScript* This = (CScript*)JS_GetContextPrivate(cx);                                          \
    if (argc < minargs) { This->SetError(#name " called with less than " #minargs " parameters"); *rval = JSVAL_NULL; return JS_FALSE; } \
    int arg = 0;

#define end_func() return (This->m_HasError ? JS_FALSE : JS_TRUE); }

#define declare_method(name) \
  JSBool name(JSContext* cx, JSObject* obj, uintN argc, jsval* argv, jsval* rval)

#define begin_method(object, name, minargs)                                           \
  JSBool name(JSContext* cx, JSObject* obj, uintN argc, jsval* argv, jsval* rval) {   \
    CScript* This = (CScript*)JS_GetContextPrivate(cx);                               \
    if (argc < minargs) { This->SetError(#name " called with less than " #minargs " parameters"); *rval = JSVAL_NULL; return JS_FALSE; } \
    int arg = 0;                                                                      \
    object Object = (object)JS_GetPrivate(cx, obj);

#define end_method() \
  return JS_TRUE; }

#define stop_execution() \
  return JS_FALSE


// parameter grabbing

inline int argInt(JSContext* cx, jsval arg)
{
  int32 i = 0;
  JS_ValueToInt32(cx, arg, &i);
  return i;
}

inline char* argStr(JSContext* cx, jsval arg)
{
  char* s = "";
  s = JS_GetStringBytes(JS_ValueToString(cx, arg));
  return s;
}

inline bool argBool(JSContext* cx, jsval arg)
{
  JSBool b = JS_FALSE;
  JS_ValueToBoolean(cx, arg, &b);
  return (b == JS_TRUE);
}

inline RGBA argColor(JSContext* cx, jsval arg)
{
  RGBA ret = { 0, 0, 0, 0 };
  if (!JSVAL_IS_OBJECT(arg)) {
    return ret;
  }

  jsval vred;   bool bred   = (JS_TRUE == JS_GetProperty(cx, JSVAL_TO_OBJECT(arg), "red",   &vred));
  jsval vgreen; bool bgreen = (JS_TRUE == JS_GetProperty(cx, JSVAL_TO_OBJECT(arg), "green", &vgreen));
  jsval vblue;  bool bblue  = (JS_TRUE == JS_GetProperty(cx, JSVAL_TO_OBJECT(arg), "blue",  &vblue));
  jsval valpha; bool balpha = (JS_TRUE == JS_GetProperty(cx, JSVAL_TO_OBJECT(arg), "alpha", &valpha));
  if (!bred || !bgreen || !bblue || !balpha) {
    return ret;
  }
  ret.red   = argInt(cx, vred);
  ret.green = argInt(cx, vgreen);
  ret.blue  = argInt(cx, vblue);
  ret.alpha = argInt(cx, valpha);
  return ret;
}

#define int_arg(name)   int name   = argInt(cx, argv[arg++])
#define str_arg(name)   char* name = argStr(cx, argv[arg++])
#define bool_arg(name)  bool name  = argBool(cx, argv[arg++])
#define color_arg(name) RGBA name  = argColor(cx, argv[arg++])

// return values

#define return_int(expr)    *rval = INT_TO_JSVAL(expr)
#define return_bool(expr)   *rval = BOOLEAN_TO_JSVAL(expr)
#define return_object(expr) *rval = OBJECT_TO_JSVAL(expr)
#define return_str(expr)    *rval = STRING_TO_JSVAL(JS_NewString(cx, expr, strlen(expr)))
#define return_double(expr) *rval = DOUBLE_TO_JSVAL(JS_NewDouble(cx, expr))


/******************************************************************************/
/* ENGINE *********************************************************************/
/******************************************************************************/

begin_func(Exit, 0)
  EngineInterface->Exit();
  stop_execution();
end_func()

////////////////////////////////////////

begin_func(ExitMessage, 1)
  str_arg(message);
  EngineInterface->ExitWithMessage(message);
  stop_execution();
end_func()

////////////////////////////////////////

begin_func(ChangeMusic, 1)
  str_arg(song);
  EngineInterface->ChangeSong(song);
end_func()

////////////////////////////////////////

begin_func(PlaySound, 1)
  str_arg(sound);
  EngineInterface->PlayEffect(sound);
end_func()



/******************************************************************************/
/* TIMING *********************************************************************/
/******************************************************************************/

begin_func(GetTime, 0)
  return_int(GetTime());
end_func()

////////////////////////////////////////

begin_func(Delay, 1)
  int_arg(milliseconds);

  dword start = GetTime();
  while (GetTime() - start < (dword)milliseconds)
    UpdateSystem();
end_func()



/******************************************************************************/
/* INPUT **********************************************************************/
/******************************************************************************/

begin_func(AnyKeyPressed, 0)
  RefreshInput();
  return_bool(AnyKeyPressed());
end_func()

////////////////////////////////////////

begin_func(KeyPressed, 1)
  int_arg(key);

  RefreshInput();
  return_bool(KeyPressed(key));
end_func()

////////////////////////////////////////

begin_func(GetMouseX, 0)
  RefreshInput();
  return_int(GetMouseX());
end_func()

////////////////////////////////////////

begin_func(GetMouseY, 0)
  RefreshInput();
  return_int(GetMouseY());
end_func()

////////////////////////////////////////

begin_func(MouseButtonPressed, 1)
  int_arg(button);

  RefreshInput();
  return_bool(MouseButtonPressed(button));
end_func()

////////////////////////////////////////

begin_func(ClearKeyQueue, 0)
  ClearKeyQueue();
end_func()

////////////////////////////////////////

begin_func(KeysLeft, 0)
  return_int(KeysLeft());
end_func()

////////////////////////////////////////

begin_func(GetKey, 0)
  return_int(GetKey());
end_func()



/******************************************************************************/
/* GRAPHICS *******************************************************************/
/******************************************************************************/

begin_func(FlipScreen, 0)
  if (This->m_FrameRate == 0)
    FlipScreen();
  else
  {
    // never skip more than MAX_FRAME_SKIP frames
    if (This->m_ShouldRender || This->m_FramesSkipped >= MAX_FRAME_SKIP)
    {
      FlipScreen();
      This->m_FramesSkipped = 0;
    }
    else
      This->m_FramesSkipped++;

    if (GetTime() * This->m_FrameRate < (dword)This->m_IdealTime)
    {
      This->m_ShouldRender = true;
      while (GetTime() * This->m_FrameRate < (dword)This->m_IdealTime)
        ;
    }
    else
      This->m_ShouldRender = false;

    // update timing variables
    This->m_IdealTime += 1000;
  }
end_func()

////////////////////////////////////////

begin_func(SetClippingRectangle, 4)
  int_arg(x);
  int_arg(y);
  int_arg(w);
  int_arg(h);

  SetClippingRectangle(x, y, w, h);
end_func()

////////////////////////////////////////

begin_func(ApplyColorMask, 1)
  color_arg(c);
  if (This->m_ShouldRender)
    ApplyColorMask(c);
end_func()

////////////////////////////////////////

begin_func(SetFrameRate, 1)
  int_arg(frame_rate);

  This->m_FrameRate = frame_rate;
  This->m_IdealTime = GetTime() * frame_rate + 1000;
  This->m_ShouldRender = true;
end_func()

////////////////////////////////////////

begin_func(GetFrameRate, 0)
  return_int(This->m_FrameRate);
end_func()

////////////////////////////////////////

begin_func(GetScreenWidth, 0)
  return_int(GetScreenWidth());
end_func()

////////////////////////////////////////

begin_func(GetScreenHeight, 0)
  return_int(GetScreenHeight());
end_func()

////////////////////////////////////////

begin_func(ClearScreen, 0)
  ClearScreen();
end_func()



/******************************************************************************/
/* GRAPHICS - IMAGES **********************************************************/
/******************************************************************************/

begin_func(LoadImage, 1)
  str_arg(filename);

  IMAGE image = EngineInterface->LoadImage(filename);
  if (image == NULL) {
    // show the error message
    char message[FILENAME_MAX + 80];
    sprintf(message, "Error: Could not load image '%s'", filename);
    EngineInterface->ExitWithMessage(message);
    stop_execution();
  }

  // FUNCTION DECLARATION
  void ssFinalizeImage(JSContext* cx, JSObject* obj);

  // define image class
  static JSClass clasp = {
    "image", JSCLASS_HAS_PRIVATE,
    JS_PropertyStub, JS_PropertyStub, JS_PropertyStub, JS_PropertyStub,
    JS_EnumerateStub, JS_ResolveStub, JS_ConvertStub, ssFinalizeImage,
    NULL, NULL, NULL, NULL, NULL, NULL, NULL,
  };
  
  // create the object
  JSObject* image_object = JS_NewObject(cx, &clasp, NULL, NULL);
  if (image_object == NULL) {
    DestroyImage(image);
    return JS_FALSE;
  }

  // assign the image to the object
  JS_SetPrivate(cx, image_object, image);

  declare_method(ssImageBlit);

  // assign the "blit" method to the object
  static JSFunctionSpec fs[] = {
    { "blit", ssImageBlit, 2, 0, 0 },
    { 0, 0, 0, 0, 0 },
  };
  JS_DefineFunctions(cx, image_object, fs);

  return_object(image_object);
end_func()

////////////////////////////////////////

begin_func(GrabImage, 4)
  int_arg(x);
  int_arg(y);
  int_arg(w);
  int_arg(h);

  // verify that the frame rate is 0
  if (This->m_FrameRate != 0)
  {
    EngineInterface->ExitWithMessage("GrabImage() should never be called if SetFrameRate() is non-zero");
    stop_execution();
  }

  IMAGE i = GrabImage(x, y, w, h);
  
  // FUNCTION DECLARATION
  void ssFinalizeImage(JSContext* cx, JSObject* obj);

  // define image class
  static JSClass clasp = {
    "image", JSCLASS_HAS_PRIVATE,
    JS_PropertyStub, JS_PropertyStub, JS_PropertyStub, JS_PropertyStub,
    JS_EnumerateStub, JS_ResolveStub, JS_ConvertStub, ssFinalizeImage,
    NULL, NULL, NULL, NULL, NULL, NULL, NULL,
  };
  
  // create the object
  JSObject* image_object = JS_NewObject(cx, &clasp, NULL, NULL);
  if (image_object == NULL) {
    DestroyImage(i);
    return JS_FALSE;
  }

  // assign the image to the object
  JS_SetPrivate(cx, image_object, i);

  // FUNCTION DECLARATION
  declare_method(ssImageBlit);

  // assign the "blit" method to the object
  static JSFunctionSpec fs[] = {
    { "blit", ssImageBlit, 2, 0, 0 },
    { 0, 0, 0, 0, 0 },
  };
  JS_DefineFunctions(cx, image_object, fs);

  return_object(image_object);
end_func()

////////////////////////////////////////

begin_method(IMAGE, ssImageBlit, 2)
  int_arg(x);
  int_arg(y);

  if (This->m_ShouldRender)
    BlitImage(Object, x, y);
end_method()

////////////////////////////////////////

void ssFinalizeImage(JSContext* cx, JSObject* obj)
{
  IMAGE i = (IMAGE)JS_GetPrivate(cx, obj);
  if (i) DestroyImage(i);
}

////////////////////////////////////////



/******************************************************************************/
/* GRAPHICS - TEXT ************************************************************/
/******************************************************************************/

begin_func(SetFont, 1)
  str_arg(filename);
  EngineInterface->SetFont(filename);
end_func()

////////////////////////////////////////

begin_func(SetTextColor, 1)
  color_arg(c);
  EngineInterface->GetFont()->SetMask(c);
end_func()

////////////////////////////////////////

begin_func(DrawText, 3)
  int_arg(x);
  int_arg(y);
  str_arg(text);

  if (This->m_ShouldRender)
    EngineInterface->GetFont()->DrawText(x, y, text);
end_func()

////////////////////////////////////////

begin_func(DrawTextBox, 6)
  int_arg(x);
  int_arg(y);
  int_arg(w);
  int_arg(h);
  int_arg(offset);
  str_arg(text);

  if (This->m_ShouldRender)
    EngineInterface->GetFont()->DrawTextBox(x, y, w, h, offset, text);
end_func()

////////////////////////////////////////

begin_func(StringWidth, 1)
  str_arg(s);
  return_int(EngineInterface->GetFont()->GetStringWidth(s));
end_func()



/******************************************************************************/
/* GRAPHICS - WINDOWS *********************************************************/
/******************************************************************************/

begin_func(SetWindowStyle, 1)
  str_arg(filename);
  EngineInterface->SetWindowStyle(filename);
end_func()

////////////////////////////////////////

begin_func(DrawWindow, 4)
  int_arg(x);
  int_arg(y);
  int_arg(w);
  int_arg(h);

  if (This->m_ShouldRender)
    EngineInterface->GetWindowStyle()->DrawWindow(x, y, w, h);
end_func()



/******************************************************************************/
/* GRAPHICS - EFFECTS *********************************************************/
/******************************************************************************/

begin_func(FadeIn, 1)
  int_arg(milliseconds);
  FadeIn(milliseconds);
end_func()

////////////////////////////////////////

begin_func(FadeOut, 1)
  int_arg(milliseconds);
  FadeOut(milliseconds);
end_func()



/******************************************************************************/
/* MAP ENGINE *****************************************************************/
/******************************************************************************/

begin_func(MapEngine, 0)
  EngineInterface->MapEngine();
end_func()

////////////////////////////////////////

begin_func(ChangeMap, 1)
  str_arg(filename);
  EngineInterface->SetMap(filename);
end_func()

////////////////////////////////////////

begin_func(GetTile, 3)
  int_arg(x);
  int_arg(y);
  int_arg(layer);
  return_int(EngineInterface->GetMap()->GetMap().GetLayer(layer).GetTile(x, y));
end_func()

////////////////////////////////////////

begin_func(SetTile, 4)
  int_arg(x);
  int_arg(y);
  int_arg(layer);
  int_arg(tile);
  EngineInterface->GetMap()->GetMap().GetLayer(layer).SetTile(x, y, tile);
end_func()

////////////////////////////////////////

begin_func(SetColorMask, 2)
  color_arg(mask);
  int_arg(num_frames);
  EngineInterface->SetColorMask(mask, num_frames);
end_func()

////////////////////////////////////////

begin_func(AddFrameHook, 2)
  int_arg(update_rate);
  str_arg(function);
  EngineInterface->AddFrameHook(update_rate, function);
end_func()

////////////////////////////////////////

begin_func(RemoveFrameHook, 1)
  str_arg(function);
  EngineInterface->RemoveFrameHook(function);
end_func()



/******************************************************************************/
/* MAP ENGINE - PARTY *********************************************************/
/******************************************************************************/

begin_func(ClearParty, 0)
  EngineInterface->ClearParty();
end_func()

////////////////////////////////////////

begin_func(AddPartyCharacter, 1)
  str_arg(filename);
  EngineInterface->AddPartyCharacter(filename);
end_func()



/******************************************************************************/
/* MAP ENGINE - DOODADS *******************************************************/
/******************************************************************************/

begin_func(SetDoodadFrame, 1)
  int_arg(frame);
  EngineInterface->SetDoodadFrame(frame);
end_func()

////////////////////////////////////////

begin_func(SetDoodadObstructive, 1)
  bool_arg(obstructive);
  EngineInterface->SetDoodadObstructive(obstructive);
end_func()



/******************************************************************************/
/* SURFACES *******************************************************************/
/******************************************************************************/

begin_func(CreateSurface, 2)
  int_arg(width);
  int_arg(height);

  // FUNCTION DECLARATION
  void ssFinalizeSurface(JSContext* cx, JSObject* obj);

  // define image class
  static JSClass clasp = {
    "surface", JSCLASS_HAS_PRIVATE,
    JS_PropertyStub, JS_PropertyStub, JS_PropertyStub, JS_PropertyStub,
    JS_EnumerateStub, JS_ResolveStub, JS_ConvertStub, ssFinalizeSurface,
    NULL, NULL, NULL, NULL, NULL, NULL, NULL,
  };
  
  // create the object
  JSObject* surface_object = JS_NewObject(cx, &clasp, NULL, NULL);
  if (surface_object == NULL) {
    return JS_FALSE;
  }

  // create the surface
  SS_SURFACE* surface = new SS_SURFACE;
  surface->surface     = new CImage32(width, height);
  surface->pen_color   = rgbaWhite;
  surface->brush_color = rgbaBlack;
  surface->brush_color.alpha = 0;

  // assign the surface to the object
  JS_SetPrivate(cx, surface_object, surface);

  declare_method(ssSurfaceBlit);
  declare_method(ssSurfaceCreateImage);
  declare_method(ssSurfaceSetPenColor);
  declare_method(ssSurfaceSetBrushColor);
  declare_method(ssSurfaceSetPixel);
  declare_method(ssSurfaceLine);
  declare_method(ssSurfaceCircle);
  declare_method(ssSurfaceRectangle);

  // assign the "blit" method to the object
  static JSFunctionSpec fs[] = {
    { "blit",          ssSurfaceBlit,          2, 0, 0 },
    { "createImage",   ssSurfaceCreateImage,   0, 0, 0 },
    { "setPenColor",   ssSurfaceSetPenColor,   1, 0, 0 },
    { "setBrushColor", ssSurfaceSetBrushColor, 1, 0, 0 },
    { "setPixel",      ssSurfaceSetPixel,      2, 0, 0 },
    { "line",          ssSurfaceLine,          4, 0, 0 },
    { "circle",        ssSurfaceCircle,        3, 0, 0 },
    { "rectangle",     ssSurfaceRectangle,     4, 0, 0 },
    { 0, 0, 0, 0, 0 },
  };
  JS_DefineFunctions(cx, surface_object, fs);

  return_object(surface_object);
end_func()

////////////////////////////////////////

void ssFinalizeSurface(JSContext* cx, JSObject* obj)
{
  SS_SURFACE* Object = (SS_SURFACE*)JS_GetPrivate(cx, obj);
  delete Object->surface;
  delete Object;
}

////////////////////////////////////////

begin_method(SS_SURFACE*, ssSurfaceBlit, 2)
  int_arg(x);
  int_arg(y);
  
  DirectBlit(
    x,
    y,
    Object->surface->GetWidth(),
    Object->surface->GetHeight(),
    Object->surface->GetPixels(),
    2
  );

end_method()

////////////////////////////////////////

begin_method(SS_SURFACE*, ssSurfaceCreateImage, 0)
  str_arg(filename);

  
  CImage32* surface = Object->surface;
  IMAGE image = CreateImage(
    surface->GetWidth(),
    surface->GetHeight(),
    surface->GetPixels());
  

  // define image class
  static JSClass clasp = {
    "image", JSCLASS_HAS_PRIVATE,
    JS_PropertyStub, JS_PropertyStub, JS_PropertyStub, JS_PropertyStub,
    JS_EnumerateStub, JS_ResolveStub, JS_ConvertStub, ssFinalizeImage,
    NULL, NULL, NULL, NULL, NULL, NULL, NULL,
  };
  
  // create the object
  JSObject* image_object = JS_NewObject(cx, &clasp, NULL, NULL);
  if (image_object == NULL) {
    DestroyImage(image);
    return JS_FALSE;
  }

  // assign the image to the object
  JS_SetPrivate(cx, image_object, image);

  declare_method(ssImageBlit);

  // assign the "blit" method to the object
  static JSFunctionSpec fs[] = {
    { "blit", ssImageBlit, 2, 0, 0 },
    { 0, 0, 0, 0, 0 },
  };
  JS_DefineFunctions(cx, image_object, fs);

  return_object(image_object);
end_method()

////////////////////////////////////////

begin_method(SS_SURFACE*, ssSurfaceSetPenColor, 1)
  color_arg(color);

  // attach the color to this object
  Object->pen_color = color;
end_method()

////////////////////////////////////////

begin_method(SS_SURFACE*, ssSurfaceSetBrushColor, 1)
  color_arg(color);
  
  // attach the color to this object
  Object->brush_color = color;
end_method()

////////////////////////////////////////

begin_method(SS_SURFACE*, ssSurfaceSetPixel, 2)
  int_arg(x);
  int_arg(y);

  // get surface
  Object->surface->SetPixel(x, y, Object->pen_color);
end_method()

////////////////////////////////////////

begin_method(SS_SURFACE*, ssSurfaceLine, 4)
  int_arg(x1);
  int_arg(y1);
  int_arg(x2);
  int_arg(y2);

  // get surface
  Object->surface->Line(x1, y1, x2, y2, Object->pen_color);
end_method()

////////////////////////////////////////

begin_method(SS_SURFACE*, ssSurfaceCircle, 3)
  int_arg(x);
  int_arg(y);
  int_arg(r);

  // get surface
  Object->surface->Circle(x, y, r, Object->pen_color, Object->brush_color);

end_method()

////////////////////////////////////////

begin_method(SS_SURFACE*, ssSurfaceRectangle, 4)
  int_arg(x1);
  int_arg(y1);
  int_arg(x2);
  int_arg(y2);

  // get surface
  Object->surface->Rectangle(x1, y1, x2, y2, Object->pen_color, Object->brush_color);

end_method()

////////////////////////////////////////



/******************************************************************************/
/* ANIMATION ******************************************************************/
/******************************************************************************/

begin_func(PlayAnimation, 1)
  str_arg(filename);
  EngineInterface->PlayAnimation(filename);
end_method()



/******************************************************************************/
/* FILES **********************************************************************/
/******************************************************************************/

begin_func(OpenFile, 1)
  str_arg(filename);
  
  // FUNCTION DECLARATION
  void ssFinalizeFile(JSContext* cx, JSObject* obj);

  // define image class
  static JSClass clasp = {
    "file", JSCLASS_HAS_PRIVATE,
    JS_PropertyStub, JS_PropertyStub, JS_PropertyStub, JS_PropertyStub,
    JS_EnumerateStub, JS_ResolveStub, JS_ConvertStub, ssFinalizeFile,
    NULL, NULL, NULL, NULL, NULL, NULL, NULL,
  };
  
  // create the object
  JSObject* file_object = JS_NewObject(cx, &clasp, NULL, NULL);
  if (file_object == NULL) {
    return JS_FALSE;
  }

  // open the file
  DATAFILE* file = EngineInterface->OpenFile(filename);
  if (file == NULL) {
    char str[1000];
    sprintf(str, "Error: Could not open file '%s'", filename);
    EngineInterface->ExitWithMessage(str);
    return JS_FALSE;
  }

  // assign the file to the object
  JS_SetPrivate(cx, file_object, file);

  declare_method(ssFileWrite);
  declare_method(ssFileRead);

  // assign the "blit" method to the object
  static JSFunctionSpec fs[] = {
    { "write", ssFileWrite, 2, 0, 0 },
    { "read",  ssFileRead,  2, 0, 0 },
    { 0, 0, 0, 0, 0 },
  };
  JS_DefineFunctions(cx, file_object, fs);

  return_object(file_object);
end_func()

////////////////////////////////////////

void ssFinalizeFile(JSContext* cx, JSObject* obj)
{
  DATAFILE* file = (DATAFILE*)JS_GetPrivate(cx, obj);
  EngineInterface->CloseFile(file);
}

////////////////////////////////////////

begin_method(DATAFILE*, ssFileWrite, 2)
  str_arg(key);

  if (JSVAL_IS_INT(argv[1]))
    WriteConfigInt(Object->config, "", key, JSVAL_TO_INT(argv[1]));
  else if (JSVAL_IS_BOOLEAN(argv[1]))
    WriteConfigBool(Object->config, "", key, (JSVAL_TO_BOOLEAN(argv[1]) == JS_TRUE));
  else if (JSVAL_IS_DOUBLE(argv[1]))
  {
    double* d = JSVAL_TO_DOUBLE(argv[1]);
    WriteConfigDouble(Object->config, "", key, *d);
  }
//  else if (JSVAL_IS_STRING(argv[1]))
  else  // anything else is a string
  {
    WriteConfigString(Object->config, "", key, argStr(cx, argv[1]));
  }
  
end_method()

////////////////////////////////////////

begin_method(DATAFILE*, ssFileRead, 2)
  str_arg(key);
  
  if (JSVAL_IS_INT(argv[1]))
  {
    int i;
    ReadConfigInt(Object->config, "", key, &i, JSVAL_TO_INT(argv[1]));
    return_int(i);
  }
  else if (JSVAL_IS_BOOLEAN(argv[1]))
  {
    bool b;
    ReadConfigBool(Object->config, "", key, &b, (JSVAL_TO_BOOLEAN(argv[1]) == JS_TRUE));
    return_bool(b);
  }
  else if (JSVAL_IS_DOUBLE(argv[1]))
  {
    double d;
    double* def = JSVAL_TO_DOUBLE(argv[1]);
    ReadConfigDouble(Object->config, "", key, &d, *def);
    return_double(d);
  }
//  else if (JSVAL_IS_STRING(argv[1]))
  else // anything else is a string
  {
    char str[8001];
    ReadConfigString(Object->config, "", key, str, 8000, argStr(cx, argv[1]));
    return_str(str);
  }

end_method()  



/******************************************************************************/
/* MENUS **********************************************************************/
/******************************************************************************/

begin_func(Menu, 0)

  sMenu* menu = new sMenu();

  // FUNCTION DECLARATION
  void ssFinalizeMenu(JSContext* cx, JSObject* obj);

  // define image class
  static JSClass clasp = {
    "menu", JSCLASS_HAS_PRIVATE,
    JS_PropertyStub, JS_PropertyStub, JS_PropertyStub, JS_PropertyStub,
    JS_EnumerateStub, JS_ResolveStub, JS_ConvertStub, ssFinalizeMenu,
    NULL, NULL, NULL, NULL, NULL, NULL, NULL,
  };
  
  // create the object
  JSObject* menu_object = JS_NewObject(cx, &clasp, NULL, NULL);
  if (menu_object == NULL) {
    delete menu;
    return JS_FALSE;
  }

  // assign the menu to the object
  JS_SetPrivate(cx, menu_object, menu);

  declare_method(ssMenuAddItem);
  declare_method(ssMenuExecuteVertical);
  declare_method(ssMenuExecuteHorizontal);
  declare_method(ssMenuSetPointer);
  declare_method(ssMenuSetUpArrow);
  declare_method(ssMenuSetDownArrow);

  // assign the "blit" method to the object
  static JSFunctionSpec fs[] = {
    { "addItem",           ssMenuAddItem,           1, 0, 0 },
    { "executeVertical",   ssMenuExecuteVertical,   5, 0, 0 },
    { "executeHorizontal", ssMenuExecuteHorizontal, 5, 0, 0 },
    { 0, 0, 0, 0, 0 },
  };
  JS_DefineFunctions(cx, menu_object, fs);

  return_object(menu_object);
end_func()

////////////////////////////////////////

void ssFinalizeMenu(JSContext* cx, JSObject* obj)
{
  sMenu* menu = (sMenu*)JS_GetPrivate(cx, obj);
  delete menu;
}

////////////////////////////////////////

begin_method(sMenu*, ssMenuAddItem, 1)
  str_arg(item);
  Object->AddItem(item);
end_method()

////////////////////////////////////////

begin_method(sMenu*, ssMenuExecuteVertical, 5)
  int_arg(x);
  int_arg(y);
  int_arg(w);
  int_arg(h);
  int_arg(offset);

  return_int(Object->ExecuteV(EngineInterface->GetMenuDisplayData(), x, y, w, h, offset));
end_method()

////////////////////////////////////////

begin_method(sMenu*, ssMenuExecuteHorizontal, 5)
  int_arg(x);
  int_arg(y);
  int_arg(w);
  int_arg(h);
  int_arg(offset);

  return_int(Object->ExecuteH(EngineInterface->GetMenuDisplayData(), x, y, w, h, offset));
end_method()

////////////////////////////////////////
