#include "WindowStyleWindow.hpp"
#include "ResizeDialog.hpp"
#include "FileDialogs.hpp"
#include "x++.hpp"
#include "resource.h"


const int ID_ALPHASLIDER = 7001;
const int ID_ALPHASTATIC = 7002;


BEGIN_MESSAGE_MAP(CWindowStyleWindow, CSaveableDocumentWindow)

  ON_WM_LBUTTONDOWN()
  ON_WM_SIZE()
  ON_WM_VSCROLL()
  ON_WM_PAINT()

  ON_COMMAND(ID_WINDOWSTYLE_EDIT_UPPERLEFT,  OnEditUpperLeft)
  ON_COMMAND(ID_WINDOWSTYLE_EDIT_TOP,        OnEditTop)
  ON_COMMAND(ID_WINDOWSTYLE_EDIT_UPPERRIGHT, OnEditUpperRight)
  ON_COMMAND(ID_WINDOWSTYLE_EDIT_RIGHT,      OnEditRight)
  ON_COMMAND(ID_WINDOWSTYLE_EDIT_LOWERRIGHT, OnEditLowerRight)
  ON_COMMAND(ID_WINDOWSTYLE_EDIT_BOTTOM,     OnEditBottom)
  ON_COMMAND(ID_WINDOWSTYLE_EDIT_LOWERLEFT,  OnEditLowerLeft)
  ON_COMMAND(ID_WINDOWSTYLE_EDIT_LEFT,       OnEditLeft)
  ON_COMMAND(ID_WINDOWSTYLE_EDIT_BACKGROUND, OnEditBackground)

  ON_COMMAND(ID_WINDOWSTYLE_RESIZESECTION,    OnResizeSection)
  ON_COMMAND(ID_WINDOWSTYLE_FILLSECTIONRGB,   OnFillSectionRGB)
  ON_COMMAND(ID_WINDOWSTYLE_FILLSECTIONALPHA, OnFillSectionAlpha)

  ON_COMMAND(ID_WINDOWSTYLE_ZOOM_1X, OnZoom1x)
  ON_COMMAND(ID_WINDOWSTYLE_ZOOM_2X, OnZoom2x)
  ON_COMMAND(ID_WINDOWSTYLE_ZOOM_4X, OnZoom4x)
  ON_COMMAND(ID_WINDOWSTYLE_ZOOM_8X, OnZoom8x)

  ON_COMMAND(ID_WINDOWSTYLE_COPY,  OnCopy)
  ON_COMMAND(ID_WINDOWSTYLE_PASTE, OnPaste)

  ON_UPDATE_COMMAND_UI(ID_WINDOWSTYLE_EDIT_UPPERLEFT,  OnUpdateEditUpperLeft)
  ON_UPDATE_COMMAND_UI(ID_WINDOWSTYLE_EDIT_TOP,        OnUpdateEditTop)
  ON_UPDATE_COMMAND_UI(ID_WINDOWSTYLE_EDIT_UPPERRIGHT, OnUpdateEditUpperRight)
  ON_UPDATE_COMMAND_UI(ID_WINDOWSTYLE_EDIT_RIGHT,      OnUpdateEditRight)
  ON_UPDATE_COMMAND_UI(ID_WINDOWSTYLE_EDIT_LOWERRIGHT, OnUpdateEditLowerRight)
  ON_UPDATE_COMMAND_UI(ID_WINDOWSTYLE_EDIT_BOTTOM,     OnUpdateEditBottom)
  ON_UPDATE_COMMAND_UI(ID_WINDOWSTYLE_EDIT_LOWERLEFT,  OnUpdateEditLowerLeft)
  ON_UPDATE_COMMAND_UI(ID_WINDOWSTYLE_EDIT_LEFT,       OnUpdateEditLeft)
  ON_UPDATE_COMMAND_UI(ID_WINDOWSTYLE_EDIT_BACKGROUND, OnUpdateEditBackground)

  ON_UPDATE_COMMAND_UI(ID_WINDOWSTYLE_ZOOM_1X, OnUpdateZoom1x)
  ON_UPDATE_COMMAND_UI(ID_WINDOWSTYLE_ZOOM_2X, OnUpdateZoom2x)
  ON_UPDATE_COMMAND_UI(ID_WINDOWSTYLE_ZOOM_4X, OnUpdateZoom4x)
  ON_UPDATE_COMMAND_UI(ID_WINDOWSTYLE_ZOOM_8X, OnUpdateZoom8x)

END_MESSAGE_MAP()


////////////////////////////////////////////////////////////////////////////////

CWindowStyleWindow::CWindowStyleWindow()
: CSaveableDocumentWindow("", IDR_WINDOWSTYLE, CSize(200, 150))
, m_bCreated(false)
, m_iSelectedBitmap(sWindowStyle::UPPER_LEFT)
, m_iZoomFactor(4)
, m_HighlightPen(new CPen(PS_SOLID, 1, 0xFF00FF))
{
  m_WindowStyle.Create(16, 16);

  // allocate DIB sections and empty them
  for (int i = 0; i < 9; i++)
    m_DIBs[i] = NULL;
  UpdateDIBSections();

  SetSaved(false);
  SetModified(false);

  // create the window and child widgets
  Create(AfxRegisterWndClass(0, ::LoadCursor(NULL, IDC_ARROW), NULL, NULL));

  m_ImageView.Create(this, this);
  m_PaletteView.Create(this, this);
  m_ColorView.Create(this, this);
  m_AlphaView.Create(this, this);
  SetBitmap();

  // we're done creating the windows, so make sure everything is in the right place
  m_bCreated = true;

  RECT ClientRect;
  GetClientRect(&ClientRect);
  OnSize(0, ClientRect.right, ClientRect.bottom);

  // make sure the various views start with matching values
  m_ColorView.SetColor(rgbBlack);
  m_ImageView.SetColor(rgbaBlack);
  m_AlphaView.SetAlpha(255);
}

////////////////////////////////////////////////////////////////////////////////

CWindowStyleWindow::CWindowStyleWindow(const char* window_style)
: CSaveableDocumentWindow(window_style, IDR_WINDOWSTYLE, CSize(200, 150))
, m_bCreated(false)
, m_iSelectedBitmap(sWindowStyle::UPPER_LEFT)
, m_iZoomFactor(4)
, m_HighlightPen(new CPen(PS_SOLID, 1, 0xFF00FF))
{
  // load the window style
  if (m_WindowStyle.Load(window_style) == false)
  {
    MessageBox("Could not load window style.  Creating new.");
    m_WindowStyle.Create(16, 16);
  }
  
  // allocate DIB sections and empty them
  for (int i = 0; i < 9; i++)
    m_DIBs[i] = NULL;
  UpdateDIBSections();

  SetSaved(true);
  SetModified(false);

  // create the window and child widgets
  Create(AfxRegisterWndClass(0, ::LoadCursor(NULL, IDC_ARROW), NULL, NULL));

  m_ImageView.Create(this, this);
  m_PaletteView.Create(this, this);
  m_ColorView.Create(this, this);
  m_AlphaView.Create(this, this);
  SetBitmap();

  // we're done creating the windows, so make sure everything is in the right place
  m_bCreated = true;

  RECT ClientRect;
  GetClientRect(&ClientRect);
  OnSize(0, ClientRect.right, ClientRect.bottom);

  // make sure the various views start with matching values
  m_ColorView.SetColor(rgbBlack);
  m_ImageView.SetColor(rgbaBlack);
  m_AlphaView.SetAlpha(255);
}

////////////////////////////////////////////////////////////////////////////////

CWindowStyleWindow::~CWindowStyleWindow()
{
  for (int i = 0; i < 9; i++)
    delete m_DIBs[i];

  m_HighlightPen->DeleteObject();
  delete m_HighlightPen;
}

////////////////////////////////////////////////////////////////////////////////

void
CWindowStyleWindow::UpdateDIBSection(int bitmap)
{
  // delete the old DIB section and allocate a new one
  delete m_DIBs[bitmap];
  int dib_width  = m_WindowStyle.GetBitmap(bitmap).GetWidth()  * m_iZoomFactor;
  int dib_height = m_WindowStyle.GetBitmap(bitmap).GetHeight() * m_iZoomFactor;
  m_DIBs[bitmap] = new CDIBSection(dib_width, dib_height, 32);
  
  // fill the DIB with data
  BGRA* dest  = (BGRA*)m_DIBs[bitmap]->GetPixels();
  CImage32& b = m_WindowStyle.GetBitmap(bitmap);
  RGBA* src   = b.GetPixels();
  for (int by = 0; by < b.GetHeight(); by++)
    for (int bx = 0; bx < b.GetWidth(); bx++)
    {
      // get the pixel to draw
      RGBA rgba = src[by * b.GetWidth() + bx];
      BGRA bgra =
      {
        rgba.blue  = rgba.alpha * rgba.blue  / 256,
        rgba.green = rgba.alpha * rgba.green / 256,
        rgba.red   = rgba.alpha * rgba.red   / 256,
        0,
      };

      // fill the square with it
      for (int dy = 0; dy < m_iZoomFactor; dy++)
        for (int dx = 0; dx < m_iZoomFactor; dx++)
        {
          dest[(by * m_iZoomFactor + dy) * dib_width + (bx * m_iZoomFactor + dx)] = bgra;
        }
    }
}

////////////////////////////////////////////////////////////////////////////////

void
CWindowStyleWindow::UpdateDIBSections()
{
  for (int i = 0; i < 9; i++)
    UpdateDIBSection(i);
}

////////////////////////////////////////////////////////////////////////////////

void
CWindowStyleWindow::SetBitmap()
{
  CImage32& b = m_WindowStyle.GetBitmap(m_iSelectedBitmap);
  m_ImageView.SetImage(b);
}

////////////////////////////////////////////////////////////////////////////////

void
CWindowStyleWindow::SetZoomFactor(int factor)
{
  m_iZoomFactor = factor;

  // resize the DIBs
  UpdateDIBSections();
  Invalidate();

  // move everything to its correct place
  RECT Rect;
  GetClientRect(&Rect);
  OnSize(0, Rect.right, Rect.bottom);
}

////////////////////////////////////////////////////////////////////////////////

void
CWindowStyleWindow::SelectBitmap(int bitmap)
{
  m_iSelectedBitmap = bitmap;
  SetBitmap();
  Invalidate();
}

////////////////////////////////////////////////////////////////////////////////

static void BlackRect(CDC& dc, int x, int y, int w, int h)
{
  RECT Rect = { x, y, x + w, y + h };
  dc.FillRect(&Rect, CBrush::FromHandle((HBRUSH)GetStockObject(BLACK_BRUSH)));
}

////////////////////////////////////////////////////////////////////////////////

void
CWindowStyleWindow::DrawCorner(CDC& dc, int bitmap, int x, int y, int w, int h)
{
  CDIBSection* dib = m_DIBs[bitmap];

  // calculate correct corner offsets so the the images are flush with the image editor
  int offsetx = 0;
  int offsety = 0;
  switch (bitmap)
  {
    case sWindowStyle::UPPER_LEFT:
      offsetx = w - dib->GetWidth();
      offsety = h - dib->GetHeight();
      break;

    case sWindowStyle::UPPER_RIGHT:
      offsetx = 0;
      offsety = h - dib->GetHeight();
      break;

    case sWindowStyle::LOWER_LEFT:
      offsetx = w - dib->GetWidth();
      offsety = 0;
      break;

    case sWindowStyle::LOWER_RIGHT:
      offsetx = 0;
      offsety = 0;
      break;
  }

  // create a clipping region for the DIB and select it in
  CRgn region;
  region.CreateRectRgn(
    offsetx + x,
    offsety + y,
    offsetx + x + dib->GetWidth(),
    offsety + y + dib->GetHeight());
  dc.SelectClipRgn(&region);

  // draw the DIB
  dc.BitBlt(
    x + offsetx, y + offsety, dib->GetWidth(), dib->GetHeight(),
    CDC::FromHandle(dib->GetDC()),
    0, 0, SRCCOPY);

  // select the region opposite of the previous one
  dc.SelectClipRgn(NULL);
  dc.SelectClipRgn(&region, RGN_XOR);

  // fill the rest of the area with black
  BlackRect(dc, x, y, w, h);

  // remove the clipping region
  dc.SelectClipRgn(NULL);
  region.DeleteObject();

  // if it's the selected bitmap, put a pink rectangle around it
  if (bitmap == m_iSelectedBitmap)
  {
    dc.SaveDC();
    dc.SelectObject(m_HighlightPen);
    dc.SelectStockObject(NULL_BRUSH);
    dc.Rectangle(offsetx + x, offsety + y, x + offsetx + dib->GetWidth(), offsety + y + dib->GetHeight());
    dc.RestoreDC(-1);
  }
}

////////////////////////////////////////////////////////////////////////////////

void
CWindowStyleWindow::DrawEdgeH(CDC& dc, int bitmap, int x, int y, int x2, int h)
{
  CDIBSection* dib = m_DIBs[bitmap];

  // calculate correct edge offsets so the the images are flush with the image editor
  int offsety = 0;
  switch (bitmap)
  {
    case sWindowStyle::TOP:
      offsety = h - dib->GetHeight();
      break;

    case sWindowStyle::BOTTOM:
      offsety = 0;
      break;
  }

  // create a clipping region for the DIBs
  CRgn region;
  region.CreateRectRgn(
    x,
    offsety + y,
    x2,
    offsety + y + dib->GetHeight());
  dc.SelectClipRgn(&region);

  // draw the edge
  int x1 = x;
  while (x1 < x2)
  {
    dc.BitBlt(
      x1, offsety + y, dib->GetWidth(), dib->GetHeight(),
      CDC::FromHandle(dib->GetDC()),
      0, 0, SRCCOPY);
    x1 += dib->GetWidth();
  }

  // select the clipping region opposite of the previous one
  dc.SelectClipRgn(NULL);
  dc.SelectClipRgn(&region, RGN_XOR);

  // fill rest of area with black
  BlackRect(dc, x, y, x2 - x, h);

  // reset the clipping rectangle
  dc.SelectClipRgn(NULL);
  region.DeleteObject();

  // if bitmap is selected, draw pink selection square
  if (bitmap == m_iSelectedBitmap)
  {
    dc.SaveDC();
    dc.SelectObject(m_HighlightPen);
    dc.SelectStockObject(NULL_BRUSH);
    dc.Rectangle(x, offsety + y, x2, offsety + y + dib->GetHeight());
    dc.RestoreDC(-1);
  }
}

////////////////////////////////////////////////////////////////////////////////

void
CWindowStyleWindow::DrawEdgeV(CDC& dc, int bitmap, int x, int y, int y2, int w)
{
  CDIBSection* dib = m_DIBs[bitmap];

  // calculate correct edge offsets so the the images are flush with the image editor
  int offsetx = 0;
  switch (bitmap)
  {
    case sWindowStyle::LEFT:
      offsetx = w - dib->GetWidth();
      break;

    case sWindowStyle::RIGHT:
      offsetx = 0;
      break;
  }

  // create a clipping region for the DIBs
  CRgn region;
  region.CreateRectRgn(offsetx + x, y, offsetx + x + dib->GetWidth(), y2);
  dc.SelectClipRgn(&region);

  // draw the edge
  int y1 = y;
  while (y1 < y2)
  {
    dc.BitBlt(
      offsetx + x, y1, dib->GetWidth(), dib->GetHeight(),
      CDC::FromHandle(dib->GetDC()),
      0, 0, SRCCOPY);
    y1 += dib->GetHeight();
  }

  // select the clipping region opposite of the previous one
  dc.SelectClipRgn(NULL);
  dc.SelectClipRgn(&region, RGN_XOR);

  // fill the area with black
  BlackRect(dc, x, y, w, y2 - y);

  // reset the clipping rectangle
  dc.SelectClipRgn(NULL);
  region.DeleteObject();

  // if bitmap is selected, draw pink selection square
  if (bitmap == m_iSelectedBitmap)
  {
    dc.SaveDC();
    dc.SelectObject(m_HighlightPen);
    dc.SelectStockObject(NULL_BRUSH);
    dc.Rectangle(offsetx + x, y, offsetx + x + dib->GetWidth(), y2);
    dc.RestoreDC(-1);
  }
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CWindowStyleWindow::OnLButtonDown(UINT flags, CPoint point)
{
  RECT EditRect;
  GetEditRect(&EditRect);

  if (point.x < EditRect.left &&
      point.y < EditRect.top)
    SelectBitmap(sWindowStyle::UPPER_LEFT);
  else if (point.x >= EditRect.right &&
           point.y < EditRect.top)
    SelectBitmap(sWindowStyle::UPPER_RIGHT);
  else if (point.x < EditRect.left &&
           point.y >= EditRect.bottom)
    SelectBitmap(sWindowStyle::LOWER_LEFT);
  else if (point.x >= EditRect.right &&
           point.y >= EditRect.bottom)
    SelectBitmap(sWindowStyle::LOWER_RIGHT);
  else if (point.x < EditRect.left)
    SelectBitmap(sWindowStyle::LEFT);
  else if (point.y < EditRect.top)
    SelectBitmap(sWindowStyle::TOP);
  else if (point.x >= EditRect.right)
    SelectBitmap(sWindowStyle::RIGHT);
  else if (point.y >= EditRect.bottom)
    SelectBitmap(sWindowStyle::BOTTOM);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CWindowStyleWindow::OnSize(UINT type, int cx, int cy)
{
  if (m_bCreated)
  {
    RECT EditRect;
    GetEditRect(&EditRect);
    m_ImageView.MoveWindow(&EditRect);

    m_PaletteView.MoveWindow(cx - 60 - 32, 0, 60, cy - 60);
    m_ColorView.MoveWindow(cx - 60 - 32, cy - 60, 60, 60);
    m_AlphaView.MoveWindow(cx - 32, 0, 32, cy);

    Invalidate();
  }

  CSaveableDocumentWindow::OnSize(type, cx, cy);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CWindowStyleWindow::OnPaint()
{
  CPaintDC dc(this);

  RECT EditRect;
  GetEditRect(&EditRect);

  // corners
  DrawCorner(dc, sWindowStyle::UPPER_LEFT,  0, 0,
    GetBorderWidth_Left(),
    GetBorderWidth_Top());
  DrawCorner(dc, sWindowStyle::UPPER_RIGHT, EditRect.right, 0,
    GetBorderWidth_Right(),
    GetBorderWidth_Top());
  DrawCorner(dc, sWindowStyle::LOWER_LEFT,  0, EditRect.bottom,
    GetBorderWidth_Left(),
    GetBorderWidth_Bottom());
  DrawCorner(dc, sWindowStyle::LOWER_RIGHT, EditRect.right, EditRect.bottom, 
    GetBorderWidth_Right(),
    GetBorderWidth_Bottom());

  // edges
  DrawEdgeH(dc, sWindowStyle::TOP,    EditRect.left, 0, EditRect.right, GetBorderWidth_Top());
  DrawEdgeH(dc, sWindowStyle::BOTTOM, EditRect.left, EditRect.bottom, EditRect.right, GetBorderWidth_Bottom());
  DrawEdgeV(dc, sWindowStyle::LEFT,   0, EditRect.top, EditRect.bottom, GetBorderWidth_Left());
  DrawEdgeV(dc, sWindowStyle::RIGHT,  EditRect.right, EditRect.top, EditRect.bottom, GetBorderWidth_Right());
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CWindowStyleWindow::OnEditUpperLeft()
{
  SelectBitmap(sWindowStyle::UPPER_LEFT);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CWindowStyleWindow::OnEditTop()
{
  SelectBitmap(sWindowStyle::TOP);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void 
CWindowStyleWindow::OnEditUpperRight()
{
  SelectBitmap(sWindowStyle::UPPER_RIGHT);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void 
CWindowStyleWindow::OnEditRight()
{
  SelectBitmap(sWindowStyle::RIGHT);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void 
CWindowStyleWindow::OnEditLowerRight()
{
  SelectBitmap(sWindowStyle::LOWER_LEFT);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void 
CWindowStyleWindow::OnEditBottom()
{
  SelectBitmap(sWindowStyle::BOTTOM);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void 
CWindowStyleWindow::OnEditLowerLeft()
{
  SelectBitmap(sWindowStyle::LOWER_LEFT);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void 
CWindowStyleWindow::OnEditLeft()
{
  SelectBitmap(sWindowStyle::LEFT);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void 
CWindowStyleWindow::OnEditBackground()
{
  SelectBitmap(sWindowStyle::BACKGROUND);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CWindowStyleWindow::OnResizeSection()
{
  CImage32& b = m_WindowStyle.GetBitmap(m_iSelectedBitmap);
  CResizeDialog Dialog("Resize Window Style Section", b.GetWidth(), b.GetHeight());
  Dialog.SetRange(1, 32, 1, 32);
  if (Dialog.DoModal() == IDOK)
  {
    m_WindowStyle.GetBitmap(m_iSelectedBitmap).Resize(Dialog.GetWidth(), Dialog.GetHeight());

    SetModified(true);

    // update the window
    SetBitmap();
    UpdateDIBSections();
    Invalidate();

    // resize the window
    RECT Rect;
    GetClientRect(&Rect);
    OnSize(0, Rect.right, Rect.bottom);
  }
}

////////////////////////////////////////////////////////////////////////////////

void
CWindowStyleWindow::OnFillSectionRGB()
{
  if (MessageBox("Are you sure?", "Fill RGB", MB_YESNO) != IDYES)
    return;

  RGB rgb = m_ColorView.GetColor();
  CImage32& b = m_WindowStyle.GetBitmap(m_iSelectedBitmap);
  RGBA* dest = b.GetPixels();
  for (int iy = 0; iy < b.GetHeight(); iy++)
    for (int ix = 0; ix < b.GetWidth(); ix++)
    {
      dest[iy * b.GetWidth() + ix].red   = rgb.red;
      dest[iy * b.GetWidth() + ix].green = rgb.green;
      dest[iy * b.GetWidth() + ix].blue  = rgb.blue;
    }

  SetBitmap();
  SetModified(true);
  UpdateDIBSection(m_iSelectedBitmap);
  Invalidate();
}

////////////////////////////////////////////////////////////////////////////////

void
CWindowStyleWindow::OnFillSectionAlpha()
{
  if (MessageBox("Are you sure?", "Fill Alpha", MB_YESNO) != IDYES)
    return;

  byte alpha = m_AlphaView.GetAlpha();
  CImage32& b = m_WindowStyle.GetBitmap(m_iSelectedBitmap);
  RGBA* dest = b.GetPixels();
  for (int iy = 0; iy < b.GetHeight(); iy++)
    for (int ix = 0; ix < b.GetWidth(); ix++)
      dest[iy * b.GetWidth() + ix].alpha = alpha;

  SetBitmap();
  SetModified(true);
  UpdateDIBSection(m_iSelectedBitmap);
  Invalidate();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CWindowStyleWindow::OnZoom1x()
{
  SetZoomFactor(1);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CWindowStyleWindow::OnZoom2x()
{
  SetZoomFactor(2);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CWindowStyleWindow::OnZoom4x()
{
  SetZoomFactor(4);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CWindowStyleWindow::OnZoom8x()
{
  SetZoomFactor(8);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CWindowStyleWindow::OnCopy()
{
  m_ImageView.Copy();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CWindowStyleWindow::OnPaste()
{
  m_ImageView.Paste();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CWindowStyleWindow::OnUpdateEditUpperLeft(CCmdUI* cmdui)
{
  cmdui->SetCheck(m_iSelectedBitmap == sWindowStyle::UPPER_LEFT);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CWindowStyleWindow::OnUpdateEditTop(CCmdUI* cmdui)
{
  cmdui->SetCheck(m_iSelectedBitmap == sWindowStyle::TOP);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CWindowStyleWindow::OnUpdateEditUpperRight(CCmdUI* cmdui)
{
  cmdui->SetCheck(m_iSelectedBitmap == sWindowStyle::UPPER_RIGHT);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CWindowStyleWindow::OnUpdateEditRight(CCmdUI* cmdui)
{
  cmdui->SetCheck(m_iSelectedBitmap == sWindowStyle::RIGHT);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CWindowStyleWindow::OnUpdateEditLowerRight(CCmdUI* cmdui)
{
  cmdui->SetCheck(m_iSelectedBitmap == sWindowStyle::LOWER_RIGHT);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CWindowStyleWindow::OnUpdateEditBottom(CCmdUI* cmdui)
{
  cmdui->SetCheck(m_iSelectedBitmap == sWindowStyle::BOTTOM);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CWindowStyleWindow::OnUpdateEditLowerLeft(CCmdUI* cmdui)
{
  cmdui->SetCheck(m_iSelectedBitmap == sWindowStyle::LOWER_LEFT);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CWindowStyleWindow::OnUpdateEditLeft(CCmdUI* cmdui)
{
  cmdui->SetCheck(m_iSelectedBitmap == sWindowStyle::LEFT);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CWindowStyleWindow::OnUpdateEditBackground(CCmdUI* cmdui)
{
  cmdui->SetCheck(m_iSelectedBitmap == sWindowStyle::BACKGROUND);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CWindowStyleWindow::OnUpdateZoom1x(CCmdUI* cmdui)
{
  cmdui->SetCheck(m_iZoomFactor == 1);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CWindowStyleWindow::OnUpdateZoom2x(CCmdUI* cmdui)
{
  cmdui->SetCheck(m_iZoomFactor == 2);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CWindowStyleWindow::OnUpdateZoom4x(CCmdUI* cmdui)
{
  cmdui->SetCheck(m_iZoomFactor == 4);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CWindowStyleWindow::OnUpdateZoom8x(CCmdUI* cmdui)
{
  cmdui->SetCheck(m_iZoomFactor == 8);
}

////////////////////////////////////////////////////////////////////////////////

bool
CWindowStyleWindow::GetSavePath(char* path)
{
  CWindowStyleFileDialog Dialog(FDM_SAVE);
  if (Dialog.DoModal() != IDOK)
    return false;

  strcpy(path, Dialog.GetPathName());
  return true;
}

////////////////////////////////////////////////////////////////////////////////

bool
CWindowStyleWindow::SaveDocument(const char* path)
{
  return m_WindowStyle.Save(path);
}

////////////////////////////////////////////////////////////////////////////////

void
CWindowStyleWindow::IV_ImageChanged()
{
  CImage32& bitmap = m_WindowStyle.GetBitmap(m_iSelectedBitmap);
  memcpy(
    bitmap.GetPixels(),
    m_ImageView.GetPixels(),
    bitmap.GetWidth() * bitmap.GetHeight() * sizeof(RGBA));

  UpdateDIBSection(m_iSelectedBitmap);
  Invalidate();

  SetModified(true);
}

////////////////////////////////////////////////////////////////////////////////

void
CWindowStyleWindow::IV_ColorChanged(RGBA color)
{
  RGB rgb = { color.red, color.green, color.blue };
  m_ColorView.SetColor(rgb);
  m_AlphaView.SetAlpha(color.alpha);
}

////////////////////////////////////////////////////////////////////////////////

void
CWindowStyleWindow::PV_ColorChanged(RGB color)
{
  RGBA rgba = { color.red, color.green, color.blue, m_AlphaView.GetAlpha() };
  m_ImageView.SetColor(rgba);
  m_ColorView.SetColor(color);
}

////////////////////////////////////////////////////////////////////////////////

void
CWindowStyleWindow::CV_ColorChanged(RGB color)
{
  RGBA rgba = { color.red, color.green, color.blue, m_AlphaView.GetAlpha() };
  m_ImageView.SetColor(rgba);
}

////////////////////////////////////////////////////////////////////////////////

void
CWindowStyleWindow::AV_AlphaChanged(byte alpha)
{
  RGBA rgba = m_ImageView.GetColor();
  rgba.alpha = alpha;
  m_ImageView.SetColor(rgba);
}

////////////////////////////////////////////////////////////////////////////////

int
CWindowStyleWindow::GetBorderWidth_Left() const
{
  int w1 = m_WindowStyle.GetBitmap(sWindowStyle::LEFT).GetWidth();
  int w2 = m_WindowStyle.GetBitmap(sWindowStyle::UPPER_LEFT).GetWidth();
  int w3 = m_WindowStyle.GetBitmap(sWindowStyle::LOWER_LEFT).GetWidth();
  return max(w1, max(w2, w3)) * m_iZoomFactor;
}

////////////////////////////////////////////////////////////////////////////////

int
CWindowStyleWindow::GetBorderWidth_Top() const
{
  int h1 = m_WindowStyle.GetBitmap(sWindowStyle::TOP).GetHeight();
  int h2 = m_WindowStyle.GetBitmap(sWindowStyle::UPPER_LEFT).GetHeight();
  int h3 = m_WindowStyle.GetBitmap(sWindowStyle::UPPER_RIGHT).GetHeight();
  return max(h1, max(h2, h3)) * m_iZoomFactor;
}

////////////////////////////////////////////////////////////////////////////////

int
CWindowStyleWindow::GetBorderWidth_Right() const
{
  int w1 = m_WindowStyle.GetBitmap(sWindowStyle::RIGHT).GetWidth();
  int w2 = m_WindowStyle.GetBitmap(sWindowStyle::UPPER_RIGHT).GetWidth();
  int w3 = m_WindowStyle.GetBitmap(sWindowStyle::LOWER_RIGHT).GetWidth();
  return max(w1, max(w2, w3)) * m_iZoomFactor;
}

////////////////////////////////////////////////////////////////////////////////

int
CWindowStyleWindow::GetBorderWidth_Bottom() const
{
  int h1 = m_WindowStyle.GetBitmap(sWindowStyle::BOTTOM).GetHeight();
  int h2 = m_WindowStyle.GetBitmap(sWindowStyle::LOWER_LEFT).GetHeight();
  int h3 = m_WindowStyle.GetBitmap(sWindowStyle::LOWER_RIGHT).GetHeight();
  return max(h1, max(h2, h3)) * m_iZoomFactor;
}

////////////////////////////////////////////////////////////////////////////////

void
CWindowStyleWindow::GetEditRect(RECT* rect)
{
  RECT ClientRect;
  GetClientRect(&ClientRect);
  rect->left   = GetBorderWidth_Left();
  rect->top    = GetBorderWidth_Top();
  rect->right  = ClientRect.right  - GetBorderWidth_Right() - 60 - 32;
  rect->bottom = ClientRect.bottom - GetBorderWidth_Bottom();
}

////////////////////////////////////////////////////////////////////////////////
