#include "TilesetEditView.hpp"
#include "TilePropertiesDialog.hpp"
#include "NumberDialog.hpp"
#include "FileDialogs.hpp"
#include "resource.h"


#define ID_TILESET_UNDO         801
#define ID_TILESET_ROTATE_CW    802
#define ID_TILESET_ROTATE_CCW   803
#define ID_TILESET_FLIP_HORIZ   804
#define ID_TILESET_FLIP_VERT    805


BEGIN_MESSAGE_MAP(CTilesetEditView, CWnd)

  ON_WM_SIZE()
  ON_WM_HSCROLL()
  ON_WM_TIMER()

  ON_COMMAND(ID_TILESET_FILL,      OnTilesetFill)
  ON_COMMAND(ID_TILESET_FILLALPHA, OnTilesetFillAlpha)
  ON_COMMAND(ID_TILESET_COPY,      OnTilesetCopy)
  ON_COMMAND(ID_TILESET_PASTE,     OnTilesetPaste)
  ON_COMMAND(ID_TILESET_TILE_PROPERTIES, OnTilesetTileProperties)

  ON_COMMAND(ID_TILESET_UNDO,       OnTilesetUndo)
  ON_COMMAND(ID_TILESET_ROTATE_CW,  OnTilesetRotateCW)
  ON_COMMAND(ID_TILESET_ROTATE_CCW, OnTilesetRotateCCW)
  ON_COMMAND(ID_TILESET_FLIP_HORIZ, OnTilesetFlipHorizontal)
  ON_COMMAND(ID_TILESET_FLIP_VERT,  OnTilesetFlipVertical)

  ON_COMMAND(ID_TILESET_INSERTTILES, OnTilesetInsertTiles)
  ON_COMMAND(ID_TILESET_APPENDTILES, OnTilesetAppendTiles)

  ON_COMMAND(ID_TILESET_DELETETILE, OnTilesetDeleteTile)

  ON_COMMAND(ID_TILESET_REPLACEWITHIMAGE, OnTilesetReplaceWithImage)
  ON_COMMAND(ID_TILESET_INSERTIMAGE,      OnTilesetInsertImage)
  ON_COMMAND(ID_TILESET_APPENDIMAGE,      OnTilesetAppendImage)

END_MESSAGE_MAP()


////////////////////////////////////////////////////////////////////////////////

CTilesetEditView::CTilesetEditView()
: m_Handler(NULL)
, m_Tileset(NULL)
, m_CurrentTile(0)
, m_Created(false)
{
}

////////////////////////////////////////////////////////////////////////////////

CTilesetEditView::~CTilesetEditView()
{
}

////////////////////////////////////////////////////////////////////////////////

BOOL
CTilesetEditView::Create(CWnd* parent, CSwatchPalette* swatchpalette, CTilesetEditViewHandler* handler, sTileset* tileset)
{
  CWnd::Create(
    AfxRegisterWndClass(0, LoadCursor(NULL, IDC_ARROW), NULL, NULL),
    "",
    WS_CHILD | WS_VISIBLE | WS_HSCROLL,
    CRect(0, 0, 0, 0),
    parent,
    1000);

  m_Handler = handler;
  m_Tileset = tileset;
  m_SwatchPalette = swatchpalette;

  // create the views
  m_ImageView.Create(this, this);
  m_PaletteView.Create(this, this);
  m_ColorView.Create(this, this);
  m_AlphaView.Create(this, this);
  InitButtonBar();

  m_Created = true;

  // put everything in the right place
  RECT rect;
  GetClientRect(&rect);
  OnSize(0, rect.right, rect.bottom);

  UpdateImageView();

  SetTimer(6000, 500, NULL);
  OnTimer(6000);

  return TRUE;
}

////////////////////////////////////////////////////////////////////////////////

void
CTilesetEditView::TilesetChanged()
{
  UpdateImageView();
  UpdateScrollBar();
  Invalidate(); 
}

////////////////////////////////////////////////////////////////////////////////

void
CTilesetEditView::SelectTile(int tile)
{
  m_CurrentTile = tile;
  UpdateImageView();
  UpdateScrollBar();
}

////////////////////////////////////////////////////////////////////////////////

void
CTilesetEditView::SP_ColorSelected(RGBA color)
{
  byte alpha = color.alpha;
  RGB  rgb   = { color.red, color.green, color.blue };

  m_ImageView.SetColor(color);
  m_ColorView.SetColor(rgb);
  m_AlphaView.SetAlpha(alpha);
}

////////////////////////////////////////////////////////////////////////////////

void
CTilesetEditView::UpdateImageView()
{
  sTile& tile = m_Tileset->GetTile(m_CurrentTile);
  m_ImageView.SetImage(tile);
}

////////////////////////////////////////////////////////////////////////////////

void
CTilesetEditView::InitButtonBar()
{
  RECT ControlRect;
  CFont* pControlFont = CFont::FromHandle((HFONT)GetStockObject(DEFAULT_GUI_FONT));
  SetRect(&ControlRect, 0, 0, 0, 0);

  m_FillButton.Create("Fill", WS_CHILD | WS_VISIBLE, ControlRect, this, ID_TILESET_FILL);
  m_FillButton.SetFont(pControlFont);

  m_FillAlphaButton.Create("Fill Alpha", WS_CHILD | WS_VISIBLE, ControlRect, this, ID_TILESET_FILLALPHA);
  m_FillAlphaButton.SetFont(pControlFont);

  m_CopyButton.Create("Copy", WS_CHILD | WS_VISIBLE, ControlRect, this, ID_TILESET_COPY);
  m_CopyButton.SetFont(pControlFont);

  m_PasteButton.Create("Paste", WS_CHILD | WS_VISIBLE, ControlRect, this, ID_TILESET_PASTE);
  m_PasteButton.SetFont(pControlFont);

  m_PropertiesButton.Create("Prop.", WS_CHILD | WS_VISIBLE, ControlRect, this, ID_TILESET_TILE_PROPERTIES);
  m_PropertiesButton.SetFont(pControlFont);

  m_UndoButton.Create("Undo", WS_CHILD | WS_VISIBLE, ControlRect, this, ID_TILESET_UNDO);
  m_UndoButton.SetFont(pControlFont);
  m_UndoButton.EnableWindow(false);

  m_RotateCWButton.Create("Rotate CW", WS_CHILD | WS_VISIBLE, ControlRect, this, ID_TILESET_ROTATE_CW);
  m_RotateCWButton.SetFont(pControlFont);

  m_RotateCCWButton.Create("Rotate CCW", WS_CHILD | WS_VISIBLE, ControlRect, this, ID_TILESET_ROTATE_CCW);
  m_RotateCCWButton.SetFont(pControlFont);

  m_FlipHorizButton.Create("Flip Horz", WS_CHILD | WS_VISIBLE, ControlRect, this, ID_TILESET_FLIP_HORIZ);
  m_FlipHorizButton.SetFont(pControlFont);

  m_FlipVertButton.Create("Flip Vert", WS_CHILD | WS_VISIBLE, ControlRect, this, ID_TILESET_FLIP_VERT);
  m_FlipVertButton.SetFont(pControlFont);
}

////////////////////////////////////////////////////////////////////////////////

static void txMoveControl(CWnd& control, int x1, int y1, int x2, int y2)
{
  control.MoveWindow(x1, y1, x2 - x1, y2 - y1, FALSE);
  control.Invalidate();  // fix bug where window would paint on button
}

////////////////////////////////////////////////////////////////////////////////

static void twMoveControl(CWnd& control, int x, int y, int w, int h)
{
  control.MoveWindow(x, y, w, h, FALSE);
  control.Invalidate();
}

////////////////////////////////////////////////////////////////////////////////

void
CTilesetEditView::MoveButtonBar(int x, int y, int w, int h)
{
  // button bar structure:
  // fill         : 1
  // fill alpha   : 1
  // copy         : 1
  // paste        : 1
  // properties   : 1

  int total_spaces = 5;
  
  // first row
  txMoveControl(m_FillButton, x, y + 0 * h / total_spaces,
                          x + w, y + 1 * h / total_spaces);

  txMoveControl(m_FillAlphaButton, x, y + 1 * h / total_spaces,
                               x + w, y + 2 * h / total_spaces);

  txMoveControl(m_CopyButton, x, y + 2 * h / total_spaces,
                          x + w, y + 3 * h / total_spaces);

  txMoveControl(m_PasteButton, x, y + 3 * h / total_spaces,
                           x + w, y + 4 * h / total_spaces);

  txMoveControl(m_PropertiesButton, x, y + 4 * h / total_spaces,
                                x + w, y + 5 * h / total_spaces);


  // second row
  txMoveControl(m_UndoButton, x + w, y + 0 * h / total_spaces,
                          x + w + w, y + 1 * h / total_spaces);

  txMoveControl(m_RotateCWButton, x + w, y + 1 * h / total_spaces,
                              x + w + w, y + 2 * h / total_spaces);

  txMoveControl(m_RotateCCWButton, x + w, y + 2 * h / total_spaces,
                               x + w + w, y + 3 * h / total_spaces);

  txMoveControl(m_FlipHorizButton, x + w, y + 3 * h / total_spaces,
                               x + w + w, y + 4 * h / total_spaces);

  txMoveControl(m_FlipVertButton, x + w, y + 4 * h / total_spaces,
                              x + w + w, y + 5 * h / total_spaces);
}

////////////////////////////////////////////////////////////////////////////////

void
CTilesetEditView::CloseButtonBar()
{
  m_ImageView.DestroyWindow();
  m_PaletteView.DestroyWindow();
  m_ColorView.DestroyWindow();
  m_AlphaView.DestroyWindow();

  m_FillButton.DestroyWindow();
  m_CopyButton.DestroyWindow();
  m_PasteButton.DestroyWindow();
  m_PropertiesButton.DestroyWindow();
}

////////////////////////////////////////////////////////////////////////////////

void
CTilesetEditView::UpdateScrollBar()
{
  SCROLLINFO si;
  si.cbSize = sizeof(si);
  si.fMask  = SIF_ALL;

  // if the tileset only has one tile, this normally would only make the scrollbar
  // disappear, so have a special case
  if (m_Tileset->GetNumTiles() == 1)
  {
    si.nMin  = 0;
    si.nMax  = 0xFFFF;  // just a big number
    si.nPage = 0xFFFF - 1;
    si.nPos  = 0;
  }
  else
  {
    si.nMin   = 0;
    si.nMax   = m_Tileset->GetNumTiles() - 1;
    si.nPage  = 1;
    si.nPos   = m_CurrentTile;
  }

  SetScrollInfo(SB_HORZ, &si);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilesetEditView::OnSize(UINT type, int cx, int cy)
{
  const int PaletteWidth = 60;
  const int AlphaWidth = 32;
  const int ButtonBarWidth = 160;

  if (m_Created)
  {
    twMoveControl(m_ImageView, 0, 0, cx - PaletteWidth - AlphaWidth - ButtonBarWidth, cy);
    twMoveControl(m_PaletteView, cx - PaletteWidth - AlphaWidth - ButtonBarWidth, 0, PaletteWidth, cy - PaletteWidth);
    twMoveControl(m_ColorView, cx - PaletteWidth - AlphaWidth - ButtonBarWidth, cy - PaletteWidth, PaletteWidth, PaletteWidth);
    twMoveControl(m_AlphaView, cx - AlphaWidth - ButtonBarWidth, 0, AlphaWidth, cy);
    MoveButtonBar(cx - ButtonBarWidth, 0, ButtonBarWidth / 2, cy);
  }
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilesetEditView::OnHScroll(UINT uSBCode, UINT uPos, CScrollBar* pScrollBar)
{
  // stores the last tile number before change
  int iLastTile = m_CurrentTile;
  
  // act on the scroll code
  switch (uSBCode)
  {
    case SB_LEFT:
      m_CurrentTile = 0;
      break;

    case SB_RIGHT:
      m_CurrentTile = m_Tileset->GetNumTiles() - 1;
      break;

    case SB_LINELEFT:
    case SB_PAGELEFT:
      m_CurrentTile--;
      break;

    case SB_LINERIGHT:
    case SB_PAGERIGHT:
      m_CurrentTile++;
      break;

    case SB_THUMBPOSITION:
    case SB_THUMBTRACK:
      m_CurrentTile = uPos;
      break;

    default:
      return;
  }

  // now check if the last tile equals to the current tile
  if (iLastTile != m_CurrentTile)
    m_UndoButton.EnableWindow(false);

  // validate the current tile value
  if (m_CurrentTile < 0)
    m_CurrentTile = 0;
  else if (m_CurrentTile > m_Tileset->GetNumTiles() - 1)
    m_CurrentTile = m_Tileset->GetNumTiles() - 1;

  UpdateImageView();
  UpdateScrollBar();

  m_Handler->TEV_SelectedTileChanged(m_CurrentTile);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilesetEditView::OnTimer(UINT timer)
{
  if (m_ImageView.CanUndo())
    m_UndoButton.EnableWindow(TRUE);
  else
    m_UndoButton.EnableWindow(FALSE);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilesetEditView::OnTilesetFill()
{
  if (MessageBox("Are you sure?", "Fill", MB_YESNO) != IDYES)
    return;

  m_ImageView.FillRGB();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilesetEditView::OnTilesetFillAlpha()
{
  if (MessageBox("Are you sure?", "Fill Alpha", MB_YESNO) != IDYES)
    return;

  m_ImageView.FillAlpha();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilesetEditView::OnTilesetCopy()
{
  m_ImageView.Copy();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilesetEditView::OnTilesetPaste()
{
  m_ImageView.Paste();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilesetEditView::OnTilesetTileProperties()
{
  CTilePropertiesDialog Dialog(m_Tileset, m_CurrentTile);
  if (Dialog.DoModal() == IDOK)
    m_Handler->TEV_TilesetModified();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilesetEditView::OnTilesetUndo()
{
  m_ImageView.Undo();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilesetEditView::OnTilesetRotateCW()
{
  m_Tileset->GetTile(m_CurrentTile).RotateCW();
  UpdateImageView();

  m_Handler->TEV_TilesetModified();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilesetEditView::OnTilesetRotateCCW()
{
  m_Tileset->GetTile(m_CurrentTile).RotateCCW();
  UpdateImageView();

  m_Handler->TEV_TilesetModified();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilesetEditView::OnTilesetFlipHorizontal()
{
  m_Tileset->GetTile(m_CurrentTile).FlipHorizontal();
  UpdateImageView();

  m_Handler->TEV_TilesetModified();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilesetEditView::OnTilesetFlipVertical()
{
  m_Tileset->GetTile(m_CurrentTile).FlipVertical();
  UpdateImageView();

  m_Handler->TEV_TilesetModified();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilesetEditView::OnTilesetInsertTiles()
{
  CNumberDialog NumberDialog("Insert Tiles", "How many tiles do you want to insert?", 1, 1, 256);
  if (NumberDialog.DoModal() == IDOK)
  {
    m_Tileset->InsertTiles(m_CurrentTile, NumberDialog.GetValue());
    UpdateScrollBar();
    UpdateImageView();
    
    m_Handler->TEV_TilesetModified();
  }
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilesetEditView::OnTilesetAppendTiles()
{
  CNumberDialog NumberDialog("Append Tiles", "How many tiles do you want to append?", 1, 1, 256);
  if (NumberDialog.DoModal() == IDOK)
  {
    m_Tileset->AppendTiles(NumberDialog.GetValue());
    UpdateScrollBar();
    UpdateImageView();
    m_Handler->TEV_TilesetModified();
  }
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilesetEditView::OnUpdateTilesetDeleteTile(CCmdUI* cmdui)
{
  if (m_Tileset->GetNumTiles() > 1)
    cmdui->Enable();
  else
    cmdui->Enable(FALSE);
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilesetEditView::OnTilesetDeleteTile()
{
  m_Tileset->DeleteTiles(m_CurrentTile, 1);
  if (m_CurrentTile >= m_Tileset->GetNumTiles())
    m_CurrentTile--;

  m_Handler->TEV_TilesetModified();
  UpdateScrollBar();
  UpdateImageView();

  m_Handler->TEV_SelectedTileChanged(m_CurrentTile);
  m_Handler->TEV_TilesetModified();
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilesetEditView::OnTilesetReplaceWithImage()
{
  if (MessageBox("Are you sure?", NULL, MB_YESNO) == IDNO)
    return;

  CImageFileDialog FileDialog(FDM_OPEN);
  if (FileDialog.DoModal() == IDOK)
  {
    if (!m_Tileset->Import_Image(FileDialog.GetPathName()))
    {
      MessageBox("Error: Could not import image");
      return;
    }

    m_Handler->TEV_TilesetModified();
    UpdateScrollBar();

    UpdateImageView();
  }
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilesetEditView::OnTilesetInsertImage()
{
  CImageFileDialog FileDialog(FDM_OPEN);
  if (FileDialog.DoModal() == IDOK)
  {
    if (!m_Tileset->InsertImage(m_CurrentTile, FileDialog.GetPathName()))
    {
      MessageBox("Error: Could not insert image");
      return;
    }

    m_Handler->TEV_TilesetModified();
    UpdateScrollBar();
    UpdateImageView();

    m_Handler->TEV_TilesetModified();
  }
}

////////////////////////////////////////////////////////////////////////////////

afx_msg void
CTilesetEditView::OnTilesetAppendImage()
{
  CImageFileDialog FileDialog(FDM_OPEN);
  if (FileDialog.DoModal() == IDOK)
  {
    if (!m_Tileset->AppendImage(FileDialog.GetPathName()))
    {
      MessageBox("Error: Could not append image");
      return;
    }

    m_Handler->TEV_TilesetModified();
    UpdateScrollBar();
    UpdateImageView();

    m_Handler->TEV_TilesetModified();
  }
}

////////////////////////////////////////////////////////////////////////////////

void
CTilesetEditView::IV_ImageChanged()
{
  // store the old data
  memcpy(
    m_Tileset->GetTile(m_CurrentTile).GetPixels(),
    m_ImageView.GetPixels(),
    m_Tileset->GetTileWidth() * m_Tileset->GetTileHeight() * sizeof(RGBA));

  m_Handler->TEV_TileModified(m_CurrentTile);
}

////////////////////////////////////////////////////////////////////////////////

void
CTilesetEditView::IV_ColorChanged(RGBA color)
{
  RGBA rgba = m_ImageView.GetColor();
  RGB rgb = { rgba.red, rgba.green, rgba.blue };
  m_ColorView.SetColor(rgb);
  m_AlphaView.SetAlpha(rgba.alpha);
  m_SwatchPalette->UpdateColor(rgb);
  m_SwatchPalette->UpdateAlpha(rgba.alpha);
}

////////////////////////////////////////////////////////////////////////////////

void
CTilesetEditView::PV_ColorChanged(RGB color)
{
  m_ColorView.SetColor(color);
  byte alpha = m_AlphaView.GetAlpha();
  RGBA c = { color.red, color.green, color.blue, alpha };
  m_ImageView.SetColor(c);
  m_SwatchPalette->UpdateColor(color);
}

////////////////////////////////////////////////////////////////////////////////

void
CTilesetEditView::CV_ColorChanged(RGB color)
{
  byte alpha = m_AlphaView.GetAlpha();
  RGBA rgba = { color.red, color.green, color.blue, alpha };
  m_ImageView.SetColor(rgba);
  m_SwatchPalette->UpdateColor(color);
  m_SwatchPalette->UpdateAlpha(alpha);
}

////////////////////////////////////////////////////////////////////////////////

void
CTilesetEditView::AV_AlphaChanged(byte alpha)
{
  RGBA rgba = m_ImageView.GetColor();
  rgba.alpha = alpha;
  m_ImageView.SetColor(rgba);
  m_SwatchPalette->UpdateAlpha(alpha);
}

////////////////////////////////////////////////////////////////////////////////

